/*
 * Title:        GridSim Toolkit
 * Description:  GridSim (Grid Simulation) Toolkit for Modeling and Simulation
 *               of Parallel and Distributed Systems such as Clusters and Grids
 * Licence:      GPL - http://www.gnu.org/copyleft/gpl.html
 *
 * $Id: GridSimTags.java,v 1.35 2004/05/29 04:39:12 anthony Exp $
 */

package gridsim;

/**
 * Contains various static command tags that indicate a type of action that
 * needs to be undertaken by GridSim entities when they receive events.<br>
 * <b>NOTE:</b> To avoid conflicts with other tags, GridSim reserves negative
 * numbers, 0, and 1 - 299.
 *
 * @author       Manzur Murshed and Rajkumar Buyya
 * @version      3.0, May 2004
 * @since        GridSim Toolkit 1.0
 * @invariant $none
 */
public class GridSimTags
{
    /** Denotes the default baud rate for some GridSim entities */
    public static final int DEFAULT_BAUD_RATE = 9600;

    /** Schedules an entity without any delay */
    public static final double SCHEDULE_NOW = 0.0;

    /** Denotes the end of simulation */
    public static final int END_OF_SIMULATION = -1;

    //////////////////////////////////////////////////////////////////////

    /**
     * Denotes insignificant simulation entity or time. This tag will not be
     * used for identification purposes.
     */
    public static final int INSIGNIFICANT = 0;

    /** Sends an Experiment object between UserEntity and Broker entity */
    public static final int EXPERIMENT = 1;

    /**
     * Denotes a grid resource to be registered. This tag is normally used
     * between GridInformationService and GridResouce entity.
     */
    public static final int REGISTER_RESOURCE = 2;

    /**
     * Denotes a grid resource, that can support advance reservation, to be
     * registered. This tag is normally used between
     * GridInformationService and GridResouce entity.
     */
    public static final int REGISTER_RESOURCE_AR = 3;

    /**
     * Denotes a list of all resources, including the ones that can support
     * advance reservation. This tag is normally used between
     * GridInformationService and GridSim entity.
     */
    public static final int RESOURCE_LIST = 4;

    /**
     * Denotes a list of resources that only support
     * advance reservation. This tag is normally used between
     * GridInformationService and GridSim entity.
     */
    public static final int RESOURCE_AR_LIST = 5;

    /**
     * Denotes grid resource characteristics information. This tag is normally
     * used between GridSim and GridResource entity.
     */
    public static final int RESOURCE_CHARACTERISTICS = 6;

    /**
     * Denotes grid resource allocation policy. This tag is normally
     * used between GridSim and GridResource entity.
     */
    public static final int RESOURCE_DYNAMICS = 7;

    /**
     * Denotes a request to get the total number of Processing Elements (PEs)
     * of a resource. This tag is normally used between GridSim and GridResource
     * entity.
     */
    public static final int RESOURCE_NUM_PE = 8;

    /**
     * Denotes a request to record events for statistical purposes. This tag is
     * normally used between GridSim and GridStatistics entity.
     */
    public static final int RECORD_STATISTICS = 9;

    /** Denotes a request to get a statistical list */
    public static final int RETURN_STAT_LIST = 10;

    /**
     * Denotes a request to send an Accumulator object based on category into
     * an event scheduler. This tag is normally used between ReportWriter and
     * GridStatistics entity.
     */
    public static final int RETURN_ACC_STATISTICS_BY_CATEGORY = 11;

    /////////////////////////////////////////////////////////////

    // I intentionally put a gap to incorporate future tags
    // so I don't have to change the numbers!

    /**
     * Denotes the return of a Gridlet back to sender. This tag is
     * normally used by GridResource entity.
     */
    public static final int GRIDLET_RETURN = 20;

    /**
     * Denotes the submission of a Gridlet. This tag is normally
     * used between GridSim User and GridResource entity.
     */
    public static final int GRIDLET_SUBMIT = 21;

    /**
     * Denotes the submission of a Gridlet with an acknowledgement.
     * This tag is normally used between GridSim User and GridResource entity.
     */
    public static final int GRIDLET_SUBMIT_ACK = 22;

    /** Cancels a Gridlet submitted in the GridResource entity */
    public static final int GRIDLET_CANCEL = 23;

    /** Denotes the status of a Gridlet. */
    public static final int GRIDLET_STATUS = 24;

    /** Pauses a Gridlet submitted in the GridResource entity */
    public static final int GRIDLET_PAUSE = 25;

    /** Pauses a Gridlet submitted in the GridResource entity with an
     * acknowledgement
     */
    public static final int GRIDLET_PAUSE_ACK = 26;

    /** Resumes a Gridlet submitted in the GridResource entity */
    public static final int GRIDLET_RESUME = 27;

    /** Resumes a Gridlet submitted in the GridResource entity with an
     * acknowledgement
     */
    public static final int GRIDLET_RESUME_ACK = 28;

    /** Moves a Gridlet to another GridResource entity */
    public static final int GRIDLET_MOVE = 29;

    /** Moves a Gridlet to another GridResource entity with an acknowledgement
     */
    public static final int GRIDLET_MOVE_ACK = 30;


    ///////////////////////////////////////////////////////////////

    // The below tags are sent by AdvanceReservation to ARGridResource class

    /**
     * Commits a reservation <b>without</b> any Gridlets attached.
     * This tag is sent by AdvanceReservation to ARGridResource class.
     */
    public static final int SEND_AR_COMMIT_ONLY = 31;

    /**
     * Commits a reservation <b>with</b> one or more Gridlets attached.
     * This tag is sent by AdvanceReservation to ARGridResource class.
     */
    public static final int SEND_AR_COMMIT_WITH_GRIDLET = 32;

    /**
     * Requests for a new <b>advanced</b> reservation.
     * This tag is sent by AdvanceReservation to ARGridResource class.
     */
    public static final int SEND_AR_CREATE = 33;

    /**
     * Requests for a new <b>immediate</b> reservation.
     * This tag is sent by AdvanceReservation to ARGridResource class.
     */
    public static final int SEND_AR_CREATE_IMMEDIATE = 34;

    /**
     * Cancels an existing reservation.
     * This tag is sent by AdvanceReservation to ARGridResource class.
     */
    public static final int SEND_AR_CANCEL = 35;

    /**
     * Requests a list of busy time of a resource.
     * This tag is sent by AdvanceReservation to ARGridResource class.
     */
    public static final int SEND_AR_LIST_BUSY_TIME = 36;

    /**
     * Requests a list of free or empty time of a resource.
     * This tag is sent by AdvanceReservation to ARGridResource class.
     */
    public static final int SEND_AR_LIST_FREE_TIME = 37;

    /**
     * Queries the current status of a reservation.
     * This tag is sent by AdvanceReservation to ARGridResource class.
     */
    public static final int SEND_AR_QUERY = 38;

    /**
     * Modifies an existing reservation.
     * This tag is sent by AdvanceReservation to ARGridResource class.
     */
    public static final int SEND_AR_MODIFY = 39;

    ///////////////////////////////////////////////////////////////

    // Below denotes the status of a reservation during its lifetime.
    // This answer is given in reply to
    // AdvanceReservation.queryReservation() method.

    /** The reservation has not yet begun, i.e. the current simulation time is
     * before the start time.
     * This tag is sent by a resource or allocation policy to AdvanceReservation
     * class.
     */
    public static final int AR_STATUS_NOT_STARTED = 40;

    /**
     * The reservation has been accepted by a resource, but not yet been
     * committed by a user.
     * This tag is sent by a resource or allocation policy to AdvanceReservation
     * class.
     */
    public static final int AR_STATUS_NOT_COMMITTED = 41;

    /**
     * The reservation has been canceled by a user during execution or active
     * session.
     * This tag is sent by a resource or allocation policy to AdvanceReservation
     * class.
     */
    public static final int AR_STATUS_TERMINATED = 42;

    /** The reservation has begun and is currently being executed by a
     * destinated GridResource entity.
     * This tag is sent by a resource or allocation policy to AdvanceReservation
     * class.
     */
    public static final int AR_STATUS_ACTIVE = 43;

    /** The reservation has been completed.
     * This tag is sent by a resource or allocation policy to AdvanceReservation
     * class.
     */
    public static final int AR_STATUS_COMPLETED = 44;

    /** The reservation has been canceled before activation.
     * This tag is sent by a resource or allocation policy to AdvanceReservation
     * class.
     */
    public static final int AR_STATUS_CANCELED = 45;

    /**
     * The reservation has passed its expiry time before being committed.
     * Hence, a resource will not execute this reservation.
     * This tag is sent by a resource or allocation policy to AdvanceReservation
     * class.
     */
    public static final int AR_STATUS_EXPIRED = 46;

    /**
     * The reservation booking ID is invalid.
     * This tag is sent by a resource or allocation policy to AdvanceReservation
     * class.
     */
    public static final int AR_STATUS_ERROR_INVALID_BOOKING_ID = 47;

    /**
     * The reservation booking ID does not exist.
     * This tag is sent by a resource or allocation policy to AdvanceReservation
     * class.
     */
    public static final int AR_STATUS_RESERVATION_DOESNT_EXIST = 48;

    /**
     * Unknown error happens.
     * This tag is sent by a resource or allocation policy to AdvanceReservation
     * class.
     */
    public static final int AR_STATUS_ERROR = 49;

    ///////////////////////////////////////////////////////////////

    // The below tags are used in reply to
    // AdvanceReservation.cancelReservation() method.
    // These tags are sent by a resource or allocation policy.

    /**
     * Cancellation of a reservation fails.
     * This tag is sent by a resource or allocation policy to
     * AdvanceReservation entity.
     */
    public static final int AR_CANCEL_FAIL = 50;

    /**
     * Cancellation of a reservation fails due to invalid booking ID.
     * This tag is sent by a resource or allocation policy to
     * AdvanceReservation entity.
     */
    public static final int AR_CANCEL_FAIL_INVALID_BOOKING_ID = 51;

    /**
     * Cancellation of a reservation fails due to finished Gridlets.
     * This tag is sent by a resource or allocation policy to
     * AdvanceReservation entity.
     */
    public static final int AR_CANCEL_FAIL_GRIDLET_FINISHED = 52;

    /**
     * Cancellation of a reservation is successful. It could also means
     * a reservation has expired or completed.
     * This tag is sent by a resource or allocation policy to
     * AdvanceReservation entity.
     */
    public static final int AR_CANCEL_SUCCESS = 53;

    /**
     * Cancellation of a reservation fails since a resource can not
     * support Advance Reservation functionalities.
     * This tag is sent by a resource or allocation policy to
     * AdvanceReservation entity.
     */
    public static final int AR_CANCEL_ERROR_RESOURCE_CANT_SUPPORT = 54;

    /**
     * Cancellation of a reservation fails due to unknown error.
     * This tag is sent by a resource or allocation policy to
     * AdvanceReservation entity.
     */
    public static final int AR_CANCEL_ERROR = 55;

    ///////////////////////////////////////////////////////////////

    // The below tags are used in reply to
    // AdvanceReservation.commitReservation() method.
    // These tags are sent by a resource or allocation policy.

    /**
     * Committing a reservation is successful.
     * This tag is sent by a resource or allocation policy to
     * AdvanceReservation entity.
     */
    public static final int AR_COMMIT_SUCCESS = 60;

    /**
     * Committing a reservation is failed.
     * This tag is sent by a resource or allocation policy to
     * AdvanceReservation entity.
     */
    public static final int AR_COMMIT_FAIL = 61;

    /**
     * Committing a reservation is failed due to expiry.
     * This tag is sent by a resource or allocation policy to
     * AdvanceReservation entity.
     */
    public static final int AR_COMMIT_FAIL_EXPIRED = 62;

    /**
     * Committing a reservation is failed due to invalid booking ID.
     * This tag is sent by a resource or allocation policy to
     * AdvanceReservation entity.
     */
    public static final int AR_COMMIT_FAIL_INVALID_BOOKING_ID = 63;

    /**
     * Committing a reservation is failed due to a resource does not support
     * Advance Reservation functionalities.
     * This tag is sent by a resource or allocation policy to
     * AdvanceReservation entity.
     */
    public static final int AR_COMMIT_ERROR_RESOURCE_CANT_SUPPORT = 64;

    /**
     * Committing a reservation is failed due to unknown error.
     * This tag is sent by a resource or allocation policy to
     * AdvanceReservation entity.
     */
    public static final int AR_COMMIT_ERROR = 65;

    ///////////////////////////////////////////////////////////////

    // The below tags are used in reply to
    // AdvanceReservation.modifyReservation() method.
    // These tags are sent by a resource or allocation policy.

    /**
     * Modification of a reservation fails due to invalid booking ID.
     * This tag is sent by a resource or allocation policy to
     * AdvanceReservation entity.
     */
    public static final int AR_MODIFY_FAIL_INVALID_BOOKING_ID = 70;

    /**
     * Modification of a reservation fails since it is in active state.
     * This tag is sent by a resource or allocation policy to
     * AdvanceReservation entity.
     */
    public static final int AR_MODIFY_FAIL_RESERVATION_ACTIVE = 71;

    /**
     * Modification of a reservation fails due to invalid start time.
     * This tag is sent by a resource or allocation policy to
     * AdvanceReservation entity.
     */
    public static final int AR_MODIFY_FAIL_INVALID_START_TIME = 72;

    /**
     * Modification of a reservation fails due to invalid end time.
     * This tag is sent by a resource or allocation policy to
     * AdvanceReservation entity.
     */
    public static final int AR_MODIFY_FAIL_INVALID_END_TIME = 73;

    /**
     * Modification of a reservation fails due to invalid number of PEs
     * requested.
     * This tag is sent by a resource or allocation policy to
     * AdvanceReservation entity.
     */
    public static final int AR_MODIFY_FAIL_INVALID_NUM_PE = 74;

    /**
     * Modification of a reservation fails due to unknown error.
     * This tag is sent by a resource or allocation policy to
     * AdvanceReservation entity.
     */
    public static final int AR_MODIFY_ERROR = 75;

    /**
     * Modification of a reservation is successfull.
     * This tag is sent by a resource or allocation policy to
     * AdvanceReservation entity.
     */
    public static final int AR_MODIFY_SUCCESS = 76;

    /**
     * Modification of a reservation fails due to a resource that can not
     * support Advance Reservation functionalities.
     * This tag is sent by a resource or allocation policy to
     * AdvanceReservation entity.
     */
    public static final int AR_MODIFY_FAIL_RESOURCE_CANT_SUPPORT = 77;

    ///////////////////////////////////////////////////////////////

    // The below tags are used in reply to
    // AdvanceReservation.createReservation() method.
    // These tags are sent by a resource or allocation policy.
    // These tags have negative numbers as not to confuse with a generated
    // reservation ID done by a resource.

    /**
     * New request of a reservation fails due to invalid start time.
     * This tag is sent by a resource or allocation policy to
     * AdvanceReservation entity.
     */
    public static final int AR_CREATE_ERROR_INVALID_START_TIME = -1;

    /**
     * New request of a reservation fails due to invalid end time.
     * This tag is sent by a resource or allocation policy to
     * AdvanceReservation entity.
     */
    public static final int AR_CREATE_ERROR_INVALID_END_TIME = -2;

    /**
     * New request of a reservation fails due to invalid duration time.
     * This tag is sent by a resource or allocation policy to
     * AdvanceReservation entity.
     */
    public static final int AR_CREATE_ERROR_INVALID_DURATION_TIME = -3;

    /**
     * New request of a reservation fails due to invalid number of PEs
     * requested.
     * This tag is sent by a resource or allocation policy to
     * AdvanceReservation entity.
     */
    public static final int AR_CREATE_ERROR_INVALID_NUM_PE = -4;

    /**
     * New request of a reservation fails due to invalid resource ID.
     * This tag is sent by a resource or allocation policy to
     * AdvanceReservation entity.
     */
    public static final int AR_CREATE_ERROR_INVALID_RESOURCE_ID = -5;

    /**
     * New request of a reservation fails due to invalid resource name.
     * This tag is sent by a resource or allocation policy to
     * AdvanceReservation entity.
     */
    public static final int AR_CREATE_ERROR_INVALID_RESOURCE_NAME = -6;

    /**
     * New request of a reservation fails due to unknown error.
     * This tag is sent by a resource or allocation policy to
     * AdvanceReservation entity.
     */
    public static final int AR_CREATE_ERROR = -7;

    /**
     * New request of a reservation fails due to a resource that can not
     * support Advance Reservation functionalities.
     * This tag is sent by a resource or allocation policy to
     * AdvanceReservation entity.
     */
    public static final int AR_CREATE_FAIL_RESOURCE_CANT_SUPPORT = -8;

    /**
     * New request of a reservation fails due to not enough PEs.
     * This tag is sent by a resource or allocation policy to
     * AdvanceReservation entity.
     */
    public static final int AR_CREATE_FAIL_RESOURCE_NOT_ENOUGH_PE = -9;

    /**
     * New request of a reservation fails since trying to request more than
     * the max. number allowed by a resource.
     * This tag is sent by a resource or allocation policy to
     * AdvanceReservation entity.
     */
    // TODO: not sure at the moment.
    //public static final int AR_CREATE_FAIL_OVERLIMIT_MAX_PE = -10;

    /**
     * New request of a reservation fails due to a resource full in 1 second.
     * This tag is sent by a resource or allocation policy to
     * AdvanceReservation entity.
     */
    public static final int AR_CREATE_FAIL_RESOURCE_FULL_IN_1_SEC = -11;

    /**
     * New request of a reservation fails due to a resource full in 5 seconds.
     * This tag is sent by a resource or allocation policy to
     * AdvanceReservation entity.
     */
    public static final int AR_CREATE_FAIL_RESOURCE_FULL_IN_5_SECS = -12;

    /**
     * New request of a reservation fails due to full in 10 seconds.
     * This tag is sent by a resource or allocation policy to
     * AdvanceReservation entity.
     */
    public static final int AR_CREATE_FAIL_RESOURCE_FULL_IN_10_SECS = -13;

    /**
     * New request of a reservation fails due to a resource full in 15 seconds.
     * This tag is sent by a resource or allocation policy to
     * AdvanceReservation entity.
     */
    public static final int AR_CREATE_FAIL_RESOURCE_FULL_IN_15_SECS = -14;

    /**
     * New request of a reservation fails due to a resource full in 30 seconds.
     * This tag is sent by a resource or allocation policy to
     * AdvanceReservation entity.
     */
    public static final int AR_CREATE_FAIL_RESOURCE_FULL_IN_30_SECS = -15;

    /**
     * New request of a reservation fails due to a resource full in 45 seconds.
     * This tag is sent by a resource or allocation policy to
     * AdvanceReservation entity.
     */
    public static final int AR_CREATE_FAIL_RESOURCE_FULL_IN_45_SECS = -16;

    /**
     * New request of a reservation fails due to a resource full in 1 minute.
     * This tag is sent by a resource or allocation policy to
     * AdvanceReservation entity.
     */
    public static final int AR_CREATE_FAIL_RESOURCE_FULL_IN_1_MIN = -17;

    /**
     * New request of a reservation fails due to a resource full in 5 minutes.
     * This tag is sent by a resource or allocation policy to
     * AdvanceReservation entity.
     */
    public static final int AR_CREATE_FAIL_RESOURCE_FULL_IN_5_MINS = -18;

    /**
     * New request of a reservation fails due to full in 10 hours.
     * This tag is sent by a resource or allocation policy to
     * AdvanceReservation entity.
     */
    public static final int AR_CREATE_FAIL_RESOURCE_FULL_IN_10_MINS = -19;

    /**
     * New request of a reservation fails due to a resource full in 15 minutes.
     * This tag is sent by a resource or allocation policy to
     * AdvanceReservation entity.
     */
    public static final int AR_CREATE_FAIL_RESOURCE_FULL_IN_15_MINS = -20;

    /**
     * New request of a reservation fails due to a resource full in 30 minutes.
     * This tag is sent by a resource or allocation policy to
     * AdvanceReservation entity.
     */
    public static final int AR_CREATE_FAIL_RESOURCE_FULL_IN_30_MINS = -21;

    /**
     * New request of a reservation fails due to a resource full in 45 minutes.
     * This tag is sent by a resource or allocation policy to
     * AdvanceReservation entity.
     */
    public static final int AR_CREATE_FAIL_RESOURCE_FULL_IN_45_MINS = -22;

    /**
     * New request of a reservation fails due to full in 1 hour.
     * This tag is sent by a resource or allocation policy to
     * AdvanceReservation entity.
     */
    public static final int AR_CREATE_FAIL_RESOURCE_FULL_IN_1_HOUR = -23;

    /**
     * New request of a reservation fails due to full in 5 hours.
     * This tag is sent by a resource or allocation policy to
     * AdvanceReservation entity.
     */
    public static final int AR_CREATE_FAIL_RESOURCE_FULL_IN_5_HOURS = -24;

    /**
     * New request of a reservation fails due to full in 10 hours.
     * This tag is sent by a resource or allocation policy to
     * AdvanceReservation entity.
     */
    public static final int AR_CREATE_FAIL_RESOURCE_FULL_IN_10_HOURS = -25;

    /**
     * New request of a reservation fails due to full in 15 hours.
     * This tag is sent by a resource or allocation policy to
     * AdvanceReservation entity.
     */
    public static final int AR_CREATE_FAIL_RESOURCE_FULL_IN_15_HOURS = -26;

    /**
     * New request of a reservation fails due to full in 30 hours.
     * This tag is sent by a resource or allocation policy to
     * AdvanceReservation entity.
     */
    public static final int AR_CREATE_FAIL_RESOURCE_FULL_IN_30_HOURS = -27;

    /**
     * New request of a reservation fails due to full in 45 hours onwards.
     * This tag is sent by a resource or allocation policy to
     * AdvanceReservation entity.
     */
    public static final int AR_CREATE_FAIL_RESOURCE_FULL_IN_45_HOURS = -28;

    ///////////////////////////////////////////////////////////////

    /**
     * Private Constructor
     */
    private GridSimTags() {
        // empty
    }

} // end class

