/*
 * Title:        GridSim Toolkit
 * Description:  GridSim (Grid Simulation) Toolkit for Modeling and Simulation
 *               of Parallel and Distributed Systems such as Clusters and Grids
 * Licence:      GPL - http://www.gnu.org/copyleft/gpl.html
 *
 * $Id: GridSim.java,v 1.46 2004/05/29 04:41:29 anthony Exp $
 */

package gridsim;

import java.io.*;
import java.util.*;
import eduni.simjava.*;


/**
 * This class is mainly responsible in initialization, running and stopping of
 * the overall simulation.
 * GridSim must be initialized to set-up the
 * simulation environment before creating any other GridSim entities at the
 * user level. This method also prepares the system for simulation by creating
 * three GridSim internal entities - {@link gridsim.GridInformationService},
 * {@link gridsim.GridSimShutdown}, {@link gridsim.GridStatistics}. Invoking the
 * {@link #startGridSimulation()} method starts the Grid simulation.
 * All the resource and user entities must be instantiated in between invoking
 * the above two methods.
 * <p>
 * Since GridSim version 3.0, all of the I/O methods have been moved
 * into {@link gridsim.GridSimCore}.
 * As a result, this class only concentrates on recording statistics and
 * managing Gridlets. In addition, there are three different ways to initialize
 * GridSim simulation. These methods are:
 * <ul>
 *     <li> by using
 *          {@link #init(int, Calendar, boolean, String[], String[], String)}
 *          method. <br>
 *          This method will create {@link gridsim.GridStatistics},
 *          {@link gridsim.GridSimRandom}, {@link gridsim.GridSimShutdown} and
 *          {@link gridsim.GridInformationService} entity. <br>
 *
 *     <li> by using {@link #init(int, Calendar, boolean)} method. <br>
 *          This method will create {@link gridsim.GridSimRandom},
 *          {@link gridsim.GridSimShutdown} and
 *          {@link gridsim.GridInformationService} entity. <br>
 *
 *     <li> by using {@link #init(int, Calendar, boolean, boolean)} method.<br>
 *          This method will create {@link gridsim.GridSimRandom} and
 *          {@link gridsim.GridSimShutdown}. A different type of
 *          {@link gridsim.GridInformationService} entity needs to be entered
 *          using {@link #setGIS(GridInformationService)} method before running
 *          the simulation.
 * </ul>
 *
 * @author       Manzur Murshed and Rajkumar Buyya
 * @version      3.0, May 2004
 * @since        GridSim Toolkit 1.0
 * @see eduni.simjava.Sim_entity
 * @see gridsim.GridSimCore
 * @see gridsim.GridInformationService
 * @see gridsim.GridSimShutdown
 * @see gridsim.GridStatistics
 * @invariant $none
 */
public class GridSim extends GridSimCore
{
    // a temp array: [0] = gridlet id, [1] = user Id and [2] = dest resource id
    private int[] dataArray_ = new int[3];
    private Sim_event ev_;           // a temp variable
    private final int SIZE = 12;     // Integer object size incl. overhead

    /////////////////////////// STATIC variables ////////////////////

    /**
     * Simulation start date. This object is initialized
     * during the call to {@link #init(int, Calendar, boolean, String[],
     *             String[], String)} or
     * {@link #init(int, Calendar, boolean)}
     */
    public static Date SimulationStartDate = null;

    /**
     * A Random object. This object is initialized during the call to
     * {@link #init(int, Calendar, boolean, String[], String[], String)} or
     * {@link #init(int, Calendar, boolean)}.
     */
    public static GridSimRandom rand = null;

    private static int gisID_ = -1;         // id of GIS entity
    private static int shutdownID_ = -1;    // id of GridSimShutdown entity
    private static int statsID_ = -1;       // id of GridStatistics entity
    private static Calendar calendar_ = null;    // a Calendar object
    private static GridInformationService gis_ = null;   // a GIS object
    private final static int NOT_FOUND = -1;     // a constant

    ////////////////////////////////////////////////////////////////////////

    /**
     * Allocates a new GridSim object
     * <b>without</b> NETWORK communication channels: "input" and
     * "output" Sim_port. In summary, this object has <tt>NO</tt>
     * network communication or bandwidth speed.
     * @param name       the name to be associated with this entity (as
     *                   required by Sim_entity class from simjava package)
     * @throws Exception This happens when creating this entity before
     *                   initializing GridSim package or this entity name is
     *                   <tt>null</tt> or empty
     * @see gridsim.GridSim#init(int, Calendar, boolean, String[], String[],
     *          String)
     * @see gridsim.GridSim#init(int, Calendar, boolean)
     * @see eduni.simjava.Sim_entity
     * @pre name != null
     * @post $none
     */
    public GridSim(String name) throws Exception
    {
        super(name);
        ev_ = new Sim_event();
    }

    /**
     * Allocates a new GridSim object
     * <b>with</b> NETWORK communication channels: "input" and
     * "output" Sim_port. In addition, this method will create <tt>Input</tt>
     * and <tt>Output</tt> object.
     * @param name       the name to be associated with this entity (as
     *                   required by Sim_entity class from simjava package)
     * @param baudRate   network communication or bandwidth speed
     * @throws Exception This happens when creating this entity before
     *                   initializing GridSim package or this entity name is
     *                   <tt>null</tt> or empty
     * @see gridsim.GridSim#init(int, Calendar, boolean, String[], String[],
     *          String)
     * @see gridsim.GridSim#init(int, Calendar, boolean)
     * @see eduni.simjava.Sim_entity
     * @see gridsim.Input
     * @see gridsim.Output
     * @pre name != null
     * @pre baudRate > 0.0
     * @post $none
     */
    public GridSim(String name, double baudRate) throws Exception
    {
        super(name, baudRate);
        ev_ = new Sim_event();
    }

    /**
     * Gets simulation start date. If the return object is
     * <tt>null</tt>, then need to initialize it by calling
     * {@link #init(int, Calendar, boolean, String[], String[], String)} or
     * {@link #init(int, Calendar, boolean)}
     *
     * @return a Date object or <tt>null</tt> if it is empty
     * @deprecated As of GridSim 2.1, replaced by
     *             {@link #getSimulationStartDate()}
     * @see java.util.Date
     * @pre $none
     * @post $none
     */
    public static Date GetSimulationStartDate() {
        return getSimulationStartDate();
    }

    /**
     * Gets simulation start date. If the return object
     * is <tt>null</tt>, then need to initialize it by calling
     * {@link #init(int, Calendar, boolean, String[], String[], String)} or
     * {@link #init(int, Calendar, boolean)}
     *
     * @return a Date object
     * @see java.util.Date
     * @pre $none
     * @post $none
     */
    public static Date getSimulationStartDate() {
        return SimulationStartDate;
    }

    /**
     * Gets the initial simulation Calendar.
     * @return a Calendar object or <tt>null</tt> if GridSim hasn't been
     *         initialized
     * @see gridsim.GridSim#init(int, Calendar, boolean, String[], String[],
     *      String)
     * @see gridsim.GridSim#init(int, Calendar, boolean)
     * @pre $none
     * @post $none
     */
    public static Calendar getSimulationCalendar() {
        return calendar_;
    }

    /**
     * Initializes GridSim parameters.
     * This method should be called before creating any entities.
     * <p>
     * Inside this method, it will create the following GridSim entities:
     * <ul>
     *     <li>GridSimRandom
     *     <li>GridStatistics
     *     <li>GridInformationService
     *     <li>GridSimShutdown
     * </ul>
     * <p>
     * The Calendar object can be specified using
     * <tt>Calendar.getInstance()</tt> to denote the start of the simulation
     * time.
     * This simulation time is <b>very important</b> in handling
     * advanced reservations functionalities.
     *
     * @param numUser  the number of User Entities created.
     *                 This parameters indicates that
     *                 {@link gridsim.GridSimShutdown} first waits for
     *                 User Entities's END_OF_SIMULATION signal before
     *                 issuing terminate signal to other entities
     * @param cal          starting time for this simulation. If it is
     *        <tt>null</tt>, then the time will be taken from
     *        <tt>Calendar.getInstance()</tt>.
     * @param traceFlag    true if GridSim trace need to be written
     * @param excludeFromFile  an array of String containing list of files to
     *                         be excluded from statistics
     * @param excludeFromProcessing   an array of String containing list of
     *                                processings to be excluded from writing
     *                                into a file
     * @param reportWriterName  a <tt>ReportWriter</tt> entity name. This entity
     *                          can be found inside a gridbroker package.
     * @deprecated As of GridSim 2.1, replaced by
     *             {@link #init(int, Calendar, boolean, String[],
     *             String[], String)}
     * @see gridsim.GridSimShutdown
     * @see gridsim.GridStatistics
     * @see gridsim.GridInformationService
     * @see gridsim.GridSimRandom
     * @pre numUser >= 0
     * @post $none
     */
    public static void Init(int numUser, Calendar cal, boolean traceFlag,
            String[] excludeFromFile, String[] excludeFromProcessing,
            String reportWriterName)
    {
        init(numUser, cal, traceFlag, excludeFromFile,
                excludeFromProcessing, reportWriterName);
    }

    /**
     * Initializes GridSim parameters.
     * This method should be called before creating any entities.
     * <p>
     * Inside this method, it will create the following GridSim entities:
     * <ul>
     *     <li>GridSimRandom
     *     <li>GridStatistics
     *     <li>GridInformationService
     *     <li>GridSimShutdown
     * </ul>
     * <p>
     * The Calendar object can be specified using
     * <tt>Calendar.getInstance()</tt> to denote the start of the simulation
     * time.
     * This simulation time is <b>very important</b> in handling
     * advanced reservations functionalities.
     *
     * @param numUser  the number of User Entities created.
     *                 This parameters indicates that
     *                 {@link gridsim.GridSimShutdown} first waits for
     *                 User Entities's END_OF_SIMULATION signal before
     *                 issuing terminate signal to other entities
     * @param cal          starting time for this simulation. If it is
     *        <tt>null</tt>, then the time will be taken from
     *        <tt>Calendar.getInstance()</tt>.
     * @param traceFlag    true if GridSim trace need to be written
     * @param excludeFromFile  an array of String containing list of files to
     *                         be excluded from statistics
     * @param excludeFromProcessing   an array of String containing list of
     *                                processings to be excluded from writing
     *                                into a file
     * @param reportWriterName  a <tt>ReportWriter</tt> entity name. This entity
     *                          can be found inside a gridbroker package.
     * @see gridsim.GridSimShutdown
     * @see gridsim.GridStatistics
     * @see gridsim.GridInformationService
     * @see gridsim.GridSimRandom
     * @pre numUser >= 0
     * @post $none
     */
    public static void init(int numUser, Calendar cal, boolean traceFlag,
            String[] excludeFromFile, String[] excludeFromProcessing,
            String reportWriterName)
    {
        try
        {
            initCommonVariable(cal, traceFlag, numUser, reportWriterName);
            GridStatistics stat = null;

            // creates a GridStatistics object
            stat = new GridStatistics("GridStatistics", "GridSim_stat.txt",true,
                                      excludeFromFile, excludeFromProcessing);

            // create a GIS object
            gis_ = new GridInformationService("GridInformationService",
                                              GridSimTags.DEFAULT_BAUD_RATE);

            // set all the above entity IDs
            gisID_ = gis_.get_id();
            statsID_ = stat.get_id();
        }
        catch (Sim_exception s)
        {
            System.out.println("GridSim.init(): Unwanted errors happen");
            System.out.println( s.getMessage() );
        }
        catch (Exception e)
        {
            System.out.println("GridSim.init(): Unwanted errors happen");
            System.out.println( e.getMessage() );
        }
    }

    /**
     * Initializes GridSim parameters <b>without</b> any statistical
     * entities. Therefore, if a simulation requires to record any statistical
     * data, then need to use
     * {@link #init(int, Calendar, boolean, String[], String[], String)}
     * instead. This method should be called before creating any entities.
     * <p>
     * Inside this method, it will create the following GridSim entities:
     * <ul>
     *     <li>GridSimRandom
     *     <li>GridInformationService
     *     <li>GridSimShutdown
     * </ul>
     * <p>
     * The Calendar object can be specified using
     * <tt>Calendar.getInstance()</tt> to denote the start of the simulation
     * time.
     * This simulation time is <b>very important</b> in handling
     * advanced reservations functionalities.
     *
     * @param numUser  the number of User Entities created.
     *                 This parameters indicates that
     *                 {@link gridsim.GridSimShutdown} first waits for
     *                 User Entities's END_OF_SIMULATION signal before
     *                 issuing terminate signal to other entities
     * @param cal          starting time for this simulation. If it is
     *        <tt>null</tt>, then the time will be taken from
     *        <tt>Calendar.getInstance()</tt>
     * @param traceFlag    true if GridSim trace need to be written
     * @see gridsim.GridSimShutdown
     * @see gridsim.GridInformationService
     * @see gridsim.GridSimRandom
     * @see gridsim.GridSim#init(int,Calendar,boolean,String[],String[],String)
     * @pre numUser >= 0
     * @post $none
     */
    public static void init(int numUser, Calendar cal, boolean traceFlag)
    {
        try
        {
            initCommonVariable(cal, traceFlag, numUser, null);

            // create a GIS object
            gis_ = new GridInformationService("GridInformationService",
                                              GridSimTags.DEFAULT_BAUD_RATE);

            // set all the above entity IDs
            gisID_ = gis_.get_id();
        }
        catch (Sim_exception s)
        {
            System.out.println("GridSim.init(): Unwanted errors happen");
            System.out.println( s.getMessage() );
        }
        catch (Exception e)
        {
            System.out.println("GridSim.init(): Unwanted errors happen");
            System.out.println( e.getMessage() );
        }
    }

    /**
     * Initializes GridSim parameters <b>without</b> any statistical
     * entities. Therefore, if a simulation requires to record any statistical
     * data, then need to use
     * {@link #init(int, Calendar, boolean, String[], String[], String)}
     * instead. This method should be called before creating any entities.
     * <p>
     * Inside this method, it will create the following GridSim entities:
     * <ul>
     *     <li>GridSimRandom
     *     <li>GridInformationService -- if the parameter <tt>gis</tt> set to
     *         <tt>true</tt>. <br>
     *         NOTE: If you want to use your own GIS entity, you need
     *         to set <tt>gis</tt> parameter to <tt>false</tt>. Then, use
     *         {@link #setGIS(GridInformationService)} method before running
     *         or starting the simulation.
     *     <li>GridSimShutdown
     * </ul>
     * <p>
     * The Calendar object can be specified using
     * <tt>Calendar.getInstance()</tt> to denote the start of the simulation
     * time.
     * This simulation time is <b>very important</b> in handling
     * advanced reservations functionalities.
     *
     * @param numUser  the number of User Entities created.
     *                 This parameters indicates that
     *                 {@link gridsim.GridSimShutdown} first waits for
     *                 all user entities's END_OF_SIMULATION signal before
     *                 issuing terminate signal to other entities
     * @param cal          starting time for this simulation. If it is
     *        <tt>null</tt>, then the time will be taken from
     *        <tt>Calendar.getInstance()</tt>
     * @param traceFlag  <tt>true</tt> if GridSim trace need to be written
     * @param gis        <tt>true</tt> if you want to use a <b>DEFAULT</b>
     *                   {@link gridsim.GridInformationService} entity.
     * @see gridsim.GridSimShutdown
     * @see gridsim.GridInformationService
     * @see gridsim.GridSimRandom
     * @see gridsim.GridSim#setGIS(GridInformationService)
     * @see gridsim.GridSim#init(int,Calendar,boolean,String[],String[],String)
     * @pre numUser >= 0
     * @post $none
     */
    public static void init(int numUser, Calendar cal, boolean traceFlag,
                            boolean gis)
    {
        try
        {
            initCommonVariable(cal, traceFlag, numUser, null);
            if (gis == true)
            {
                // create a GIS object
                gis_ = new GridInformationService("GridInformationService",
                                    GridSimTags.DEFAULT_BAUD_RATE);

                // set all the above entity IDs
                gisID_ = gis_.get_id();
            }
        }
        catch (Sim_exception s)
        {
            System.out.println("GridSim.init(): Unwanted errors happen");
            System.out.println( s.getMessage() );
        }
        catch (Exception e)
        {
            System.out.println("GridSim.init(): Unwanted errors happen");
            System.out.println( e.getMessage() );
        }
    }

    /**
     * Sets a <tt>GridInformationService</tt> (GIS) entity.
     * This method is useful is you write a different type of GIS entity.
     * This method must be called before {@link #startGridSimulation()} method.
     * @param gis  a GIS object
     * @return <tt>true</tt> if successful, <tt>false</tt> otherwise
     * @pre gis != null
     * @post $none
     * @see gridsim.GridSim#startGridSimulation()
     */
    public static boolean setGIS(GridInformationService gis)
    {
        if (gis == null) {
            return false;
        }

        gis_ = gis;
        gisID_ = gis.get_id();
        return true;
    }

    /**
     * Initializes all the common attributes
     * @param cal   the starting time for this simulation. If it is
     *        <tt>null</tt>, then the time will be taken from
     *        <tt>Calendar.getInstance()</tt>.
     * @param traceFlag   true if GridSim trace need to be written
     * @throws Exception This happens when creating this entity before
     *                   initializing GridSim package or this entity name is
     *                   <tt>null</tt> or empty
     * @pre $none
     * @post $none
     */
    private static void initCommonVariable(Calendar cal, boolean traceFlag,
               int numUser, String reportWriterName) throws Exception
    {
        // NOTE: the order for the below 3 lines are important
        Sim_system.initialise();
        Sim_system.set_trc_level(1);
        Sim_system.set_auto_trace(traceFlag);

        // Set the current Wall clock time as the starting time of simulation
        calendar_ = cal;
        if (cal == null) {
            calendar_ = Calendar.getInstance();
        }

        SimulationStartDate = calendar_.getTime();
        rand = new GridSimRandom();

        // creates a GridSimShutdown object
        GridSimShutdown shutdown = new GridSimShutdown("GridSimShutdown",
                                              numUser, reportWriterName);
        shutdownID_ = shutdown.get_id();
    }

    /**
     * Starts the execution of GridSim simulation.
     * It waits for complete execution of all entitities, i.e. until
     * all entities threads reach non-RUNNABLE state by exiting from the
     * <tt>body()</tt> method. Then, it kills threads of all entities.
     * <p>
     * <b>Note</b>: This method should be called after all the entities
     *              have been setup and added, and their ports are linked.
     * @deprecated As of GridSim 2.1, replaced by {@link #startGridSimulation()}
     * @throws NullPointerException This happens when creating this entity
     *               before initializing GridSim package or this entity name is
     *               <tt>null</tt> or empty
     * @see gridsim.GridSim#init(int, Calendar, boolean, String[], String[],
     *          String)
     * @pre $none
     * @post $none
     */
    public static void Start() throws NullPointerException {
        startGridSimulation();
    }

    /**
     * Starts the execution of GridSim simulation.
     * It waits for complete execution of all entitities, i.e. until
     * all entities threads reach non-RUNNABLE state by exiting from the
     * <tt>body()</tt> method. Then, it kills threads of all entities.
     * <p>
     * <b>Note</b>: This method should be called after all the entities
     *              have been setup and added, and their ports are linked.
     * @throws NullPointerException This happens when creating this entity
     *              before initializing GridSim package or this entity name is
     *              <tt>null</tt> or empty
     * @see gridsim.GridSim#init(int, Calendar, boolean, String[], String[],
     *          String)
     * @pre $none
     * @post $none
     */
    public static void startGridSimulation() throws NullPointerException
    {
        System.out.println("Starting GridSim version 3.0");
        try {
            Sim_system.run();
        }
        catch (Sim_exception e)
        {
            throw new NullPointerException("GridSim.startGridSimulation() :" +
                    " Error - you haven't initialized GridSim.");
        }
    }

    /**
     * Gets the current simulation time (based on SimJava simulation clock)
     * @return The current simulation time from the simulation clock
     * @deprecated As of GridSim 2.1, replaced by {@link #clock()}
     * @see eduni.simjava.Sim_system#clock()
     * @pre $none
     * @post $result >= 0.0
     */
    public static double Clock() {
        return clock();
    }

    /**
     * Gets the current simulation time (based on SimJava simulation clock)
     * @return The current simulation time from the simulation clock
     * @see eduni.simjava.Sim_system#clock()
     * @pre $none
     * @post $result >= 0.0
     */
    public static double clock() {
        return Sim_system.clock();
    }

    /**
     * Causes the entity to hold for <tt>duration</tt> units of simulation time
     * @param duration the amount of time to hold
     * @deprecated As of GridSim 2.1, replaced by {@link #gridSimHold(double)}
     * @pre $none
     * @post $none
     */
    public void GridSimHold(double duration) {
        gridSimHold(duration);
    }

    /**
     * Causes the entity to hold for <tt>duration</tt> units of simulation time
     * @param duration the amount of time to hold
     * @pre $none
     * @post $none
     */
    public void gridSimHold(double duration)
    {
        // if duration is -ve, then no use to hold -ve time.
        if (duration < 0.0) {
            return;
        }

        super.sim_process(duration);
    }

    /**
     * Stops Grid Simulation (based on SimJava Sim_system.run_stop()).
     * This should be ony called if any of the user defined entities
     * <b>explicitly</b> want
     * to terminate simulation during execution.
     * @see eduni.simjava.Sim_system#run_stop()
     * @deprecated As of GridSim 2.1, replaced by {@link #stopGridSimulation()}
     * @throws NullPointerException This happens when creating this entity
     *               before initializing GridSim package or this entity name is
     *               <tt>null</tt> or empty
     * @see gridsim.GridSim#init(int, Calendar, boolean, String[], String[],
     *          String)
     * @pre $none
     * @post $none
     */
    public static void Stop() throws NullPointerException {
        stopGridSimulation();
    }

    /**
     * Stops Grid Simulation (based on SimJava Sim_system.run_stop()).
     * This should be ony called if any of the user defined entities
     * <b>explicitly</b> want
     * to terminate simulation during execution.
     * @throws NullPointerException This happens when creating this entity
     *                before initializing GridSim package or this entity name is
     *                <tt>null</tt> or empty
     * @see gridsim.GridSim#init(int, Calendar, boolean, String[], String[],
     *          String)
     * @see eduni.simjava.Sim_system#run_stop()
     * @pre $none
     * @post $none
     */
    public static void stopGridSimulation() throws NullPointerException
    {
        try {
            Sim_system.run_stop();
        }
        catch (Sim_exception e)
        {
            throw new NullPointerException("GridSim.stopGridSimulation() : " +
                    "Error - can't stop Grid Simulation.");
        }
    }

    /**
     * Gets the data passed in this event
     * @return A reference to the data
     * @deprecated As of GridSim 2.1, replaced by {@link #receiveEventObject()}
     * @pre $none
     * @post $result != null
     */
    protected Object ReceiveEventObject() {
        return receiveEventObject();
    }

    /**
     * Gets the data passed in this event
     * @return A reference to the data
     * @pre $none
     * @post $result != null
     */
    protected Object receiveEventObject()
    {
        super.sim_get_next(ev_);
        return ev_.get_data();
    }

    /**
     * Gets the data passed in this event
     * @param sourcePort a Sim_port object which is used to connect entities
     *                    for  event passing
     * @return A reference to the data or <tt>null</tt> if the source port is
     *         empty
     * @see eduni.simjava.Sim_port
     * @pre sourcePort != null
     * @post $none
     */
    protected Object ReceiveEventObject(Sim_port sourcePort) {
        return receiveEventObject(sourcePort);
    }

    /**
     * Gets the data passed in this event
     * @param sourcePort a Sim_port object which is used to connect entities
     *                    for  event passing
     * @return A reference to the data or <tt>null</tt> if the source port is
     *         empty
     * @see eduni.simjava.Sim_port
     * @pre sourcePort != null
     * @post $none
     */
    protected Object receiveEventObject(Sim_port sourcePort)
    {
        if (sourcePort == null) {
            return null;
        }

        super.sim_get_next( new Sim_from_port(sourcePort), ev_ );
        return ev_.get_data();
    }

    /**
     * Sends a Gridlet to the destination GridResource ID <tt>without</tt>
     * any delay. An acknowledgement to denote the successful of this method
     * is by default <tt>off or false</tt>.
     *
     * @param gl       a Gridlet object to be sent
     * @param resID    an unique resource ID
     * @return <tt>true</tt> if this Gridlet has been submitted to the
     *         destination GridResource, <tt>false</tt> otherwise.
     *         Submitting a Gridlet can be failed for the one or more
     *         following reasons:
     *         <ul>
     *              <li> if a GridResource ID doesn't exist
     *              <li> if a Gridlet ID doesn't exist
     *              <li> if a Gridlet's user ID doesn't exist
     *              <li> if a Gridlet object is <tt>null</tt> or empty;
     *              <li> if a Gridlet object has <tt>finished</tt> executing
     *                   beforehand
     *         </ul>
     *
     * @deprecated As of GridSim 2.1, replaced by {@link #gridletSubmit(Gridlet,
     *             int)}
     * @pre gl != null
     * @pre resID >= 0
     * @post $none
     */
    protected boolean GridletSubmit(Gridlet gl, int resID) {
        return gridletSubmit(gl, resID);
    }

    /**
     * Sends a Gridlet to the destination GridResource ID <tt>without</tt>
     * any delay. An acknowledgement to denote the successful of this method
     * is by default <tt>off or false</tt>.
     *
     * @param gl       a Gridlet object to be sent
     * @param resID    an unique resource ID
     * @return <tt>true</tt> if this Gridlet has been submitted to the
     *         destination GridResource, <tt>false</tt> otherwise.
     *         Submitting a Gridlet can be failed for the one or more
     *         following reasons:
     *         <ul>
     *              <li> if a GridResource ID doesn't exist
     *              <li> if a Gridlet ID doesn't exist
     *              <li> if a Gridlet's user ID doesn't exist
     *              <li> if a Gridlet object is <tt>null</tt> or empty;
     *              <li> if a Gridlet object has <tt>finished</tt> executing
     *                   beforehand
     *         </ul>
     * @pre gl != null
     * @pre resID >= 0
     * @post $none
     */
    protected boolean gridletSubmit(Gridlet gl, int resID) {
        return gridletSubmit(gl, resID, GridSimTags.SCHEDULE_NOW, false);
    }

    /**
     * Sends a Gridlet to the destination GridResource ID <tt>with</tt>
     * a specified delay.
     *
     * @param gl            a Gridlet object to be sent
     * @param resourceID    an unique resource ID
     * @param delay         delay time or <tt>0.0</tt> if want to execute NOW
     * @param ack           an acknowledgment status. <tt>true</tt> if want to
     *                      know the result of this method, <tt>false</tt>
     *                      otherwise or don't care.
     * @return <tt>true</tt> if this Gridlet has been submitted to the
     *         destination GridResource, <tt>false</tt> otherwise.
     *         Submitting a Gridlet can be failed for the one or more
     *         following reasons:
     *         <ul>
     *              <li> if the acknowledgment status in the parameter of this
     *                   method is set to <tt>false</tt>
     *              <li> if a GridResource ID doesn't exist
     *              <li> if a Gridlet ID doesn't exist
     *              <li> if a Gridlet's user ID doesn't exist
     *              <li> if the delay time is negative
     *              <li> if a Gridlet object is <tt>null</tt> or empty;
     *              <li> if a Gridlet object has <tt>finished</tt> executing
     *                   beforehand
     *         </ul>
     * @see gridsim.Gridlet#isFinished()
     * @see gridsim.GridSim#gridletReceive()
     * @pre gl != null
     * @pre resourceID >= 0
     * @pre delay >= 0.0
     * @post $none
     */
    protected boolean gridletSubmit(Gridlet gl, int resourceID,
                                    double delay, boolean ack)
    {
        // checks whether a Gridlet is empty or delay is -ve
        if (gl == null || delay < 0.0) {
            return false;
        }

        // checks whether a Gridlet has finished executing before
        if (gl.isFinished() == true)
        {
            System.out.println("GridSim.gridletSubmit(): Error - Gridlet #" +
                                   gl.getGridletID() + " for User #" +
                                   gl.getUserID() + " is already finished.");
            return false;
        }

        boolean valid = false;
        try
        {
            String errorMsg = "GridSim.gridletSubmit(): ";
            valid = validateValue(errorMsg, gl.getGridletID(),
                                  gl.getUserID(), resourceID);

            // if any of the above value invalid, then return
            if (valid == false) {
                return false;
            }

            // sends the gridlet to a destination GridResource id with ACK
            if (ack == true)
            {
                send(super.output, delay, GridSimTags.GRIDLET_SUBMIT_ACK,
                     new IO_data(gl, gl.getGridletFileSize(), resourceID)
                );

                // waiting for a response back from the GridResource
                Sim_type_p tag = new Sim_type_p(GridSimTags.GRIDLET_SUBMIT_ACK
                                                + gl.getGridletID() );

                // only look for ack for same Gridlet ID
                super.sim_get_next(tag, ev_);
                valid = ( (Boolean) ev_.get_data() ).booleanValue();
            }
            else   // sends without ACK
            {
                valid = false;
                send(super.output, delay, GridSimTags.GRIDLET_SUBMIT,
                     new IO_data(gl, gl.getGridletFileSize(), resourceID)
                );
            }
        }
        catch (Sim_exception sim)
        {
            valid = false;
            System.out.println(
                "GridSim.gridletSubmit(): Error from SimJava occurs.");
            System.out.println( sim.getMessage() );
        }
        catch (Exception e)
        {
            valid = false;
            System.out.println("GridSim.gridletSubmit(): Error occurs.");
            System.out.println( e.getMessage() );
        }

        return valid;
    }

    /**
     * Gets a Gridlet object passed in this event
     * @return A reference to Gridlet object or <tt>null</tt> if an error
     * occurs
     * @deprecated As of GridSim 2.1, replaced by {@link #gridletReceive()}
     * @pre $none
     * @post $none
     */
    protected Gridlet GridletReceive() {
        return gridletReceive();
    }

    /**
     * Gets a Gridlet object passed in this event
     * @return A reference to Gridlet object or <tt>null</tt> if an error
     * occurs
     * @pre $none
     * @post $none
     */
    protected Gridlet gridletReceive()
    {
        Gridlet gl = null;
        try {
            gl = (Gridlet) receiveEventObject(this.input);
        }
        catch (ClassCastException c) {
            return gl;
        }
        catch (Exception e) {
            return gl;
        }

        return gl;
    }

    /**
     * Receives a Gridlet object from a specific resource.
     * @param gridletId   a Gridlet ID
     * @param userId      a user ID
     * @param resId       a grid resource ID
     * @return a Gridlet object or <tt>null</tt> if there is none or an error.
     * @pre gridletId >= 0
     * @pre userId > 0
     * @pre resId > 0
     * @post $none
     */
    protected Gridlet gridletReceive(int gridletId, int userId, int resId)
    {
        String errorMsg = "GridSim.gridletReceive(): ";
        boolean valid = validateValue(errorMsg, gridletId, userId, resId);
        if (valid == false) {
            return null;
        }

        // waiting for a response from the GridResource entity
        Sim_type_p tag = new Sim_type_p(GridSimTags.GRIDLET_RETURN
                                                + gridletId);

        // only look for this type of ack for same Gridlet ID
        super.sim_get_next(tag, ev_);

        Gridlet gl = null;
        try {
            gl = (Gridlet) ev_.get_data();
        }
        catch (ClassCastException c) {
            gl = null;
        }
        catch (Exception e) {
            gl = null;
        }

        return gl;
    }

    /**
     * Sends a Gridlet based on their event tag to the destination resource ID.
     * @param errorMsg   a message containing which operation it belongs to
     * @param gridletId  a Gridlet ID
     * @param userId     a Gridlet's user or owner ID
     * @param resourceId    a GridResource ID
     * @param delay      sending delay time
     * @param tag        event tag (such as GridSimTags.GRIDLET_PAUSE, etc...)
     * @param ack        denotes whether want to have an acknowledgment or not
     * @return <tt>true</tt> if the Gridlet has been sent successfully,
     *         <tt>false</tt> otherwise
     * @pre errorMsg != null
     * @pre gridletId > 0
     * @pre userId > 0
     * @pre resourceId > 0
     * @pre delay >= 0.0
     * @pre tag > 0
     * @post $result = true || false
     */
    private boolean sendGridlet(String errorMsg, int gridletId, int userId,
                           int resourceId, double delay, int tag, boolean ack)
    {
        boolean valid = validateValue(errorMsg, gridletId, userId, resourceId);
        if (valid == false || delay < 0.0) {
            return false;
        }

        int size = 14;  // size of having 3 ints + 2 bytes overhead
        dataArray_[0] = gridletId;
        dataArray_[1] = userId;
        dataArray_[2] = NOT_FOUND;  // this index is only used by gridletMove()

        // if an ack is required, then change the tag
        int newTag = tag;
        if (ack == true)
        {
            switch (tag)
            {
                case GridSimTags.GRIDLET_PAUSE:
                    newTag = GridSimTags.GRIDLET_PAUSE_ACK;
                    break;

                case GridSimTags.GRIDLET_RESUME:
                    newTag = GridSimTags.GRIDLET_RESUME_ACK;
                    break;

                default:
                    break;
            }
        }

        // send this Gridlet
        send(super.output, delay, newTag,
             new IO_data(dataArray_, size, resourceId)
        );

        return true;
    }

    /**
     * Performs validation of the given parameters before sending a Gridlet
     * to a GridResource entity
     * @param msg        a message containing which operation it belongs to
     * @param gridletId  a Gridlet ID
     * @param userId     a Gridlet's user or owner ID
     * @param resourceId    a GridResource ID
     * @return <tt>true</tt> if the validation has passed successfully,
     *         <tt>false</tt> otherwise
     * @pre msg != null
     * @pre gridletId > 0
     * @pre userId > 0
     * @pre resourceId > 0
     * @post $result = true || false
     */
    private boolean validateValue(String msg, int gridletId, int userId,
                int resourceId)
    {
        boolean valid = true;

        // Gridlet ID must be 0 or positive
        if (gridletId < 0)
        {
            valid = false;
            System.out.println(msg + "Error - Gridlet ID must be >= 0, not " +
                    gridletId);
        }

        // User ID must be 0 or positive
        if (userId < 0)
        {
            valid = false;
            System.out.println(msg + "Error - User ID must be >= 0, not " +
                    userId);
        }

        // GridResource ID must be 0 or positive
        if (resourceId < 0)
        {
            valid = false;
            System.out.println(msg +
                    "Error - GridResource ID must be >= 0, not " + resourceId);
        }

        // if a grid resource ID doesn't exist in GIS list
        if (gis_.isResourceExist(resourceId) == false)
        {
            valid = false;
            System.out.println(msg + "Error - GridResource ID #" + resourceId +
                               " doesn't exist");
        }

        return valid;
    }

    /**
     * Cancels a Gridlet that is currently executing in a given GridResource
     * ID <tt>with</tt> a delay. <br>
     * <b>NOTE:</b> Canceling a Gridlet operation can take a long time over a
     *              slow network if the Gridlet size is big.
     * @param gl            a Gridlet object to be canceled
     * @param resourceId    an unique resource ID
     * @param delay         delay time or <tt>0.0</tt> if want to cancel NOW
     * @return the canceled Gridlet or <tt>null</tt if this operation fails.
     *         If a Gridlet has <tt>finished</tt> in time of cancellation, then
     *         this method will return the finished Gridlet.
     *         Canceling a Gridlet can be failed for the one or more
     *         following reasons:
     *         <ul>
     *              <li> if a GridResource ID doesn't exist
     *              <li> if a Gridlet ID doesn't exist
     *              <li> if a Gridlet's user ID doesn't exist
     *              <li> if the delay time is negative
     *              <li> if a Gridlet object is <tt>null</tt> or empty;
     *         </ul>
     * @pre gl != null
     * @pre resourceId >= 0
     * @pre delay >= 0.0
     * @post $none
     */
    protected Gridlet gridletCancel(Gridlet gl, int resourceId, double delay)
    {
        if (gl == null || delay < 0.0) {
            return null;
        }

        Gridlet obj = gridletCancel( gl.getGridletID(), gl.getUserID(),
                                    resourceId, delay );
        return obj;
    }

    /**
     * Cancels a Gridlet that is currently executing in a given GridResource
     * ID <tt>with</tt> a delay. <br>
     * <b>NOTE:</b> Canceling a Gridlet operation can be slow over a slow
     *              network if the Gridlet size is big.
     * @param gridletId     a Gridlet ID
     * @param userId        the user or owner ID of this Gridlet
     * @param resourceId    an unique resource ID to which this Gridlet was
     *                      previously sent to
     * @param delay         delay time or <tt>0.0</tt> if want to cancel NOW
     * @return the canceled Gridlet or <tt>null</tt if this operation fails.
     *         If a Gridlet has <tt>finished</tt> in time of cancellation, then
     *         this method will return the finished Gridlet.
     *         Canceling a Gridlet can be failed for the one or more
     *         following reasons:
     *         <ul>
     *              <li> if a GridResource ID doesn't exist
     *              <li> if a Gridlet ID doesn't exist
     *              <li> if a Gridlet's user ID doesn't exist
     *              <li> if the delay time is negative
     *         </ul>
     * @pre gridletId >= 0
     * @pre userId >= 0
     * @pre resourceId >= 0
     * @pre delay >= 0.0
     * @post $none
     */
    protected Gridlet gridletCancel(int gridletId, int userId, int resourceId,
                                    double delay)
    {
        Gridlet gl = null;
        try
        {
            String errorMsg = "GridSim.gridletCancel(): ";
            boolean valid = sendGridlet(errorMsg, gridletId, userId, resourceId,
                                    delay, GridSimTags.GRIDLET_CANCEL, false);

            if (valid == true)
            {
                // waiting for a response from the GridResource entity
                Sim_type_p tag = new Sim_type_p(GridSimTags.GRIDLET_CANCEL
                                                + gridletId);

                // only look for this type of ack for same Gridlet ID
                super.sim_get_next(tag, ev_);
                gl = (Gridlet) ev_.get_data();
            }
        }
        catch (Sim_exception e)
        {
            gl = null;
            System.out.println("GridSim.gridletCancel(): Error occurs.");
            System.out.println( e.getMessage() );
        }
        catch (Exception e)
        {
            gl = null;
            System.out.println("GridSim.gridletCancel(): Error occurs.");
            System.out.println( e.getMessage() );
        }

        return gl;
    }

    /**
     * Pauses a Gridlet that is currently executing in a given GridResource
     * ID <tt>with</tt> a delay.
     * @param gl            a Gridlet object to be sent
     * @param resId         an unique resource ID
     * @param delay         delay time or <tt>0.0</tt> if want to execute NOW
     * @return <tt>true</tt> if this Gridlet has been paused successfully in the
     *         destination GridResource, <tt>false</tt> otherwise.
     *         Pausing a Gridlet can be failed for the one or more
     *         following reasons:
     *         <ul>
     *              <li> if a GridResource ID doesn't exist
     *              <li> if a Gridlet ID doesn't exist
     *              <li> if a Gridlet's user ID doesn't exist
     *              <li> if the delay time is negative
     *              <li> if a Gridlet object is <tt>null</tt> or empty;
     *              <li> if a Gridlet object has <tt>finished</tt> executing
     *                   beforehand. The Gridlet needs to be retrieved by
     *                   using {@link #gridletReceive()}
     *         </ul>
     * @pre gl != null
     * @pre resId >= 0
     * @pre delay >= 0.0
     * @post $none
     */
    protected boolean gridletPause(Gridlet gl, int resId, double delay)
    {
        if (gl == null || delay < 0.0) {
            return false;
        }

        return gridletPause(gl.getGridletID(),gl.getUserID(),resId,delay,true);
    }

    /**
     * Pauses a Gridlet that is currently executing in a given GridResource
     * ID <tt>with</tt> a delay.
     * @param gridletId     a Gridlet ID
     * @param userId        the user or owner ID of this Gridlet
     * @param resourceId    an unique resource ID
     * @param delay         delay time or <tt>0.0</tt> if want to execute NOW
     * @param ack   an acknowledgement, i.e. <tt>true</tt> if wanted to know
     *        whether this operation is success or not, <tt>false</tt>
     *        otherwise (don't care)
     * @return <tt>true</tt> if this Gridlet has been paused successfully in the
     *         destination GridResource, <tt>false</tt> otherwise.
     *         Pausing a Gridlet can be failed for the one or more
     *         following reasons:
     *         <ul>
     *              <li> if a GridResource ID doesn't exist
     *              <li> if a Gridlet ID doesn't exist
     *              <li> if a Gridlet's user ID doesn't exist
     *              <li> if the delay time is negative
     *              <li> if a Gridlet object has <tt>finished</tt> executing
     *                   beforehand. The Gridlet needs to be retrieved by using
     *                   {@link #gridletReceive()}
     *         </ul>
     * @pre gridletId >= 0
     * @pre userId >= 0
     * @pre resourceId >= 0
     * @pre delay >= 0.0
     * @post $none
     */
    protected boolean gridletPause(int gridletId, int userId, int resourceId,
                                   double delay, boolean ack)
    {
        boolean valid = false;
        try
        {
            String errorMsg = "GridSim.gridletPause(): ";
            valid = sendGridlet(errorMsg, gridletId, userId, resourceId, delay,
                                GridSimTags.GRIDLET_PAUSE, ack);

            if (valid == true && ack == true)
            {
                // waiting for a response from the GridResource
                Sim_type_p tag = new Sim_type_p(GridSimTags.GRIDLET_PAUSE_ACK
                                                + gridletId);

                // only look for this type of ack for same Gridlet ID
                super.sim_get_next(tag, ev_);
                valid = ( (Boolean) ev_.get_data() ).booleanValue();
            }
        }
        catch (Sim_exception e)
        {
            valid = false;
            System.out.println("GridSim.gridletPause(): Error occurs.");
            System.out.println( e.getMessage() );
        }
        catch (Exception e)
        {
            valid = false;
            System.out.println("GridSim.gridletPause(): Error occurs.");
            System.out.println( e.getMessage() );
        }

        return valid;
    }

    /**
     * Gets the current status of this Gridlet in a given GridResource ID
     * @param gl    a Gridlet object
     * @param resourceId   a GridResource ID that executes this Gridlet object
     * @return the current Gridlet status or <tt>-1</tt> if not found.
     *         The various Gridlet status can be found in Gridlet class.
     * @see gridsim.Gridlet
     * @pre gl != null
     * @pre resourceId > 0
     * @post $none
     */
    protected int gridletStatus(Gridlet gl, int resourceId)
    {
        if (gl == null) {
            return NOT_FOUND;
        }

        return gridletStatus(gl.getGridletID(), gl.getUserID(), resourceId);
    }

    /**
     * Gets the current status of this Gridlet in a given GridResource ID
     * @param gridletId    a Gridlet ID
     * @param userId       the user or owner of this Gridlet object
     * @param resourceId   a GridResource ID that executes this Gridlet object
     * @return the current Gridlet status or <tt>-1</tt> if not found.
     *         The various Gridlet status can be found in Gridlet class.
     * @see gridsim.Gridlet
     * @pre gridletId > 0
     * @pre userId > 0
     * @pre resourceId > 0
     * @post $none
     */
    protected int gridletStatus(int gridletId, int userId, int resourceId)
    {
        int status = NOT_FOUND;
        try
        {
            String errorMsg = "GridSim.gridletStatus(): ";
            boolean valid = sendGridlet(errorMsg, gridletId, userId, resourceId,
                                        0.0, GridSimTags.GRIDLET_STATUS, false);

            if (valid == false) {
                return status;
            }

            // waiting for a response from the GridResource
            Sim_type_p tag = new Sim_type_p(GridSimTags.GRIDLET_STATUS
                                            + gridletId);

            // only look for this type of ack for same Gridlet ID
            super.sim_get_next(tag, ev_);
            status = ( (Integer) ev_.get_data() ).intValue();
        }
        catch (Sim_exception e)
        {
            status = NOT_FOUND;
            System.out.println("GridSim.gridletStatus(): Error occurs.");
            System.out.println( e.getMessage() );
        }
        catch (Exception e)
        {
            status = NOT_FOUND;
            System.out.println("GridSim.gridletStatus(): Error occurs.");
            System.out.println( e.getMessage() );
        }

        return status;
    }

    /**
     * Resumes a Gridlet that is currently pausing in a given GridResource
     * ID <tt>with</tt> a delay.<br>
     * <b>NOTE:</b> Resuming a Gridlet only works if it is currently on paused.
     * @param gl            a Gridlet object to be sent
     * @param resId         an unique resource ID
     * @param delay         delay time or <tt>0.0</tt> if want to execute NOW
     * @return <tt>true</tt> if this Gridlet has been resumed successfully in
     *         the destination GridResource, <tt>false</tt> otherwise.
     *         Resuming a Gridlet can be failed for the one or more
     *         following reasons:
     *         <ul>
     *              <li> if a GridResource ID doesn't exist
     *              <li> if a Gridlet ID doesn't exist
     *              <li> if a Gridlet's user ID doesn't exist
     *              <li> if the delay time is negative
     *              <li> if a Gridlet object is <tt>null</tt> or empty
     *              <li> if a Gridlet is not currently on paused
     *         </ul>
     * @pre gl != null
     * @pre resId >= 0
     * @pre delay >= 0.0
     * @post $none
     */
    protected boolean gridletResume(Gridlet gl, int resId, double delay)
    {
        if (gl == null || delay < 0.0) {
            return false;
        }

        return gridletResume(gl.getGridletID(),gl.getUserID(),resId,delay,true);
    }

    /**
     * Resumes a Gridlet that is currently pausing in a given GridResource
     * ID <tt>with</tt> a delay. <br>
     * <b>NOTE:</b> Resuming a Gridlet only works if it is currently on paused.
     * @param gridletId     a Gridlet ID
     * @param userId        the user or owner ID of this Gridlet
     * @param resourceId    an unique resource ID
     * @param delay         delay time or <tt>0.0</tt> if want to execute NOW
     * @param ack   an acknowledgement, i.e. <tt>true</tt> if wanted to know
     *        whether this operation is success or not, <tt>false</tt>
     *        otherwise (don't care)
     * @return <tt>true</tt> if this Gridlet has been resumed successfully in
     *         the destination GridResource, <tt>false</tt> otherwise.
     *         Resuming a Gridlet can be failed for the one or more
     *         following reasons:
     *         <ul>
     *              <li> if a GridResource ID doesn't exist
     *              <li> if a Gridlet ID doesn't exist
     *              <li> if a Gridlet's user ID doesn't exist
     *              <li> if the delay time is negative
     *              <li> if a Gridlet is not currently on paused
     *         </ul>
     * @pre gridletId >= 0
     * @pre userId >= 0
     * @pre resourceId >= 0
     * @pre delay >= 0.0
     * @post $none
     */
    protected boolean gridletResume(int gridletId, int userId, int resourceId,
                                    double delay, boolean ack)
    {
        boolean valid = false;
        try
        {
            String errorMsg = "GridSim.gridletResume(): ";
            valid = sendGridlet(errorMsg, gridletId, userId, resourceId, delay,
                                GridSimTags.GRIDLET_RESUME, ack);

            if (valid == true && ack == true)
            {
                // waiting for a response from the GridResource
                Sim_type_p tag = new Sim_type_p(GridSimTags.GRIDLET_RESUME_ACK
                                                + gridletId);

                // only look for this type of ack for same Gridlet ID
                super.sim_get_next(tag, ev_);
                valid = ( (Boolean) ev_.get_data() ).booleanValue();
            }
        }
        catch (Sim_exception e)
        {
            valid = false;
            System.out.println("GridSim.gridletResume(): Error occurs.");
            System.out.println( e.getMessage() );
        }
        catch (Exception e)
        {
            valid = false;
            System.out.println("GridSim.gridletResume(): Error occurs.");
            System.out.println( e.getMessage() );
        }

        return valid;
    }

    /**
     * Moves a Gridlet to the destination GridResource ID
     * @param gl   a Gridlet object
     * @param srcId   the GridResource ID that is currently executing this Gridlet
     * @param destId  the new GridResource ID
     * @param delay   simulation delay
     * @return <tt>true</tt> if this Gridlet has moved successfully,
     *         <tt>false</tt> otherwise. Moving a Gridlet can be failed, due
     *         to one or more following reasons:
     *      <ul>
     *         <li> if a Gridlet object is null
     *         <li> if one or both GridResource ID don't exist
     *         <li> if the delay is negative
     *         <li> if a Gridlet in the GridResource has completed beforehand.
     *              The Gridlet can be retrieved by using
     *              {@link #gridletReceive()}
     *      </ul>
     * @pre gl != null
     * @pre srcId > 0
     * @pre destId > 0
     * @pre delay >= 0.0
     * @post $result = true || false
     */
    protected boolean gridletMove(Gridlet gl,int srcId,int destId,double delay)
    {
        if (gl == null || delay < 0.0) {
            return false;
        }

        boolean success = gridletMove(gl.getGridletID(), gl.getUserID(),
                                      srcId, destId, delay, true);

        return success;
    }

    /**
     * Moves a Gridlet to the destination GridResource ID
     * @param gridletId   a Gridlet ID
     * @param userId      the owner or user ID of this Gridlet
     * @param srcId   the GridResource ID that is currently executing this
     *                Gridlet
     * @param destId  the new GridResource ID
     * @param delay   simulation delay
     * @param ack   an acknowledgement, i.e. <tt>true</tt> if wanted to know
     *        whether this operation is success or not, <tt>false</tt>
     *        otherwise (don't care)
     * @return <tt>true</tt> if this Gridlet has moved successfully,
     *         <tt>false</tt> otherwise. Moving a Gridlet can be failed, due
     *         to one or more following reasons:
     *      <ul>
     *         <li> if a Gridlet ID doesn't exist
     *         <li> if the owner of user ID of this Gridlet doesn't exist
     *         <li> if one or both GridResource ID don't exist
     *         <li> if the delay is negative
     *         <li> if a Gridlet in the GridResource has completed beforehand.
     *              The Gridlet can be retrieved by using
     *              {@link #gridletReceive()}
     *      </ul>
     * @pre gridletId > 0
     * @pre userId > 0
     * @pre srcId > 0
     * @pre destId > 0
     * @pre delay >= 0.0
     * @post $result = true || false
     */
    protected boolean gridletMove(int gridletId, int userId, int srcId,
                                  int destId, double delay, boolean ack)
    {
        String errorMsg = "GridSim.gridletMove(): ";

        // check whether the source Id is the same as destination Id
        if (srcId == destId)
        {
            System.out.println(errorMsg + "Error - Can't move a Gridlet to " +
                   "the same GridResource.");
            return false;
        }

        boolean valid = validateValue(errorMsg, gridletId, userId, srcId);
        if (valid == false || delay < 0.0) {
            return false;
        }

        // if a destination grid resource ID doesn't exist in GIS list
        if (gis_.isResourceExist(destId) == false)
        {
            System.out.println(errorMsg + "Error - GridResource ID #" + destId +
                   " doesn't exist. Hence, can't move Gridlet #" + gridletId);

            return false;
        }

        try
        {
            int size = 14;  // size of having 3 ints + 2 bytes overhead
            dataArray_[0] = gridletId;
            dataArray_[1] = userId;
            dataArray_[2] = destId;

            if (ack == true)
            {
                // sends the info to the destination GridResource
                send(super.output, delay, GridSimTags.GRIDLET_MOVE_ACK,
                     new IO_data(dataArray_, size, srcId)
                );

                // waiting for a response from the GridResource
                Sim_type_p tag = new Sim_type_p(GridSimTags.GRIDLET_SUBMIT_ACK
                                                + gridletId);

                // only look for this type of ack for same Gridlet ID
                super.sim_get_next(tag, ev_);
                valid = ( (Boolean) ev_.get_data() ).booleanValue();
            }
            else
            {
                valid = false;
                send(super.output, delay, GridSimTags.GRIDLET_MOVE,
                     new IO_data(dataArray_, size, srcId)
                );
            }
        }
        catch (Sim_exception e)
        {
            valid = false;
            System.out.println("GridSim.gridletMove(): Error occurs.");
            System.out.println( e.getMessage() );
        }
        catch (Exception ex)
        {
            valid = false;
            System.out.println("GridSim.gridletMove(): Error occurs.");
            System.out.println( ex.getMessage() );
        }

        return valid;
    }

    /**
     * Gets the name of this entity
     * @return the Entity name or <tt>null</tt> if this object does not have
     *         one
     * @deprecated As of GridSim 2.1, replaced by {@link #getEntityName()}
     * @pre $none
     * @post $none
     */
    public String GetEntityName() {
        return getEntityName();
    }

    /**
     * Gets the name of this entity
     * @return the Entity name or <tt>null</tt> if this object does not have
     *         one
     * @pre $none
     * @post $none
     */
    public String getEntityName()
    {
        try {
            return super.get_name();
        }
        catch (Sim_exception e) {
            return null;
        }
        catch (Exception e) {
            return null;
        }
    }

    /**
     * Gets name of the entity given its entity ID
     * @param entityID  the entity ID
     * @return the Entity name or <tt>null</tt> if this object does not have
     *         one
     * @deprecated As of GridSim 2.1, replaced by {@link #getEntityName(int)}
     * @pre entityID > 0
     * @post $none
     */
    public static String GetEntityName(int entityID) {
        return getEntityName(entityID);
    }

    /**
     * Gets name of the entity given its entity ID
     * @param entityID  the entity ID
     * @return the Entity name or <tt>null</tt> if this object does not have
     *         one
     * @pre entityID > 0
     * @post $none
     */
    public static String getEntityName(int entityID)
    {
        try {
            return Sim_system.get_entity(entityID).get_name();
        }
        catch (Sim_exception e) {
            return null;
        }
        catch (Exception e) {
            return null;
        }
    }

    /**
     * Gets the entity ID
     * @param entityName    an Entity name
     * @return the Entity ID or <tt>-1</tt> if it is not found
     * @deprecated As of GridSim 2.1, replaced by {@link #getEntityId(String)}
     * @pre entityName != null
     * @post $result >= -1
     */
    public static int GetEntityId(String entityName) {
        return getEntityId(entityName);
    }

    /**
     * Gets the entity ID given its name
     * @param entityName    an Entity name
     * @return the Entity ID or <tt>-1</tt> if it is not found
     * @pre entityName != null
     * @post $result >= -1
     */
    public static int getEntityId(String entityName)
    {
        if (entityName == null) {
            return NOT_FOUND;
        }

        try {
            return (int) Sim_system.get_entity_id(entityName);
        }
        catch (Sim_exception e) {
            return NOT_FOUND;
        }
        catch (Exception e) {
            return NOT_FOUND;
        }
    }

    /**
     * Gets the entity id of <tt>GridStatistics</tt>
     * @return the Entity ID or <tt>-1</tt> if it is not found
     * @deprecated As of GridSim 2.1, replaced by {@link
     *             #getGridStatisticsEntityId()}
     * @pre $none
     * @post $result >= -1
     */
    public static int GridStatisticsEntityId() {
        return getGridStatisticsEntityId();
    }

    /**
     * Gets the entity ID of <tt>GridStatistics</tt>
     * @return the Entity ID or <tt>-1</tt> if it is not found
     * @pre $none
     * @post $result >= -1
     */
    public static int getGridStatisticsEntityId() {
        return statsID_;
    }

    /**
     * Gets the entity ID of <tt>GridInformationService</tt>
     * @return the Entity ID or <tt>-1</tt> if it is not found
     * @deprecated As of GridSim 2.1, replaced by {@link
     *             #getGridInfoServiceEntityId()}
     * @pre $none
     * @post $result >= -1
     */
    public static int GridInformationServiceEntityId() {
        return getGridInfoServiceEntityId();
    }

    /**
     * Gets the entity ID of <tt>GridInformationService</tt>
     * @return the Entity ID or <tt>-1</tt> if it is not found
     * @pre $none
     * @post $result >= -1
     */
    public static int getGridInfoServiceEntityId() {
        return gisID_;
    }

    /**
     * Gets the entity id of {@link gridsim.GridSimShutdown}
     * @return the Entity ID or <tt>-1</tt> if it is not found
     * @deprecated As of GridSim 2.1, replaced by {@link
     *             #getGridSimShutdownEntityId()}
     * @pre $none
     * @post $result >= -1
     */
    public static int GridSimShutdownEntityId() {
        return getGridSimShutdownEntityId();
    }

    /**
     * Gets the entity id of {@link gridsim.GridSimShutdown}
     * @return the Entity ID or <tt>-1</tt> if it is not found
     * @pre $none
     * @post $result >= -1
     */
    public static int getGridSimShutdownEntityId() {
        return shutdownID_;
    }

    /**
     * Tells all user entities to shut down the simulation.
     * {@link gridsim.GridSimShutdown} entity waits for all users
     * termination before shuting down other entities.
     * @see gridsim.GridSimShutdown
     * @deprecated As of GridSim 2.1, replaced by {@link
     *             #shutdownUserEntity()}
     * @pre $none
     * @post $none
     */
    protected void ShutdownUserEntity() {
        shutdownUserEntity();
    }

    /**
     * Tells all user entities to shut down the simulation.
     * {@link gridsim.GridSimShutdown} entity waits for all users
     * termination before shuting down other entities.
     * @see gridsim.GridSimShutdown
     * @pre $none
     * @post $none
     */
    protected void shutdownUserEntity()
    {
        if (shutdownID_ != NOT_FOUND) {
            send(shutdownID_, 0.0, GridSimTags.END_OF_SIMULATION);
        }
    }

    /**
     * Tells the <tt>GridStatistics</tt> entity the end of the simulation
     * @deprecated As of GridSim 2.1, replaced by {@link
     *             #shutdownGridStatisticsEntity()}
     * @pre $none
     * @post $none
     */
    protected void ShutdownGridStatisticsEntity() {
        shutdownGridStatisticsEntity();
    }

    /**
     * Tells the <tt>GridStatistics</tt> entity the end of the simulation
     * @pre $none
     * @post $none
     */
    protected void shutdownGridStatisticsEntity()
    {
        if (statsID_ != NOT_FOUND) {
            send(statsID_, 0.0, GridSimTags.END_OF_SIMULATION);
        }
    }

    /**
     * Sends a request to Grid Information Service (GIS) entity to get the
     * list of Grid resources
     * @return A LinkedList containing GridResource ID (as an Integer object)
     *         or <tt>null</tt> if a GIS entity hasn't been created before
     * @deprecated As of GridSim 2.1, replaced by {@link
     *             #getGridResourceList()}
     * @pre $none
     * @post $none
     */
    public static LinkedList GetGridResourceList() {
        return getGridResourceList();
    }

    /**
     * Sends a request to Grid Information Service (GIS) entity to get the
     * list of all Grid resources
     * @return A LinkedList containing GridResource ID (as an Integer object)
     *         or <tt>null</tt> if a GIS entity hasn't been created before
     * @pre $none
     * @post $none
     */
    public static LinkedList getGridResourceList()
    {
        if (gis_ == null) {
            return null;
        }

        return gis_.getList();
    }

    /**
     * Checks whether a particular resource supports Advanced Reservation
     * functionalities or not.
     * @param resourceID    a resource ID
     * @return <tt>true</tt> if a resource supports Advanced Reservation
     *         functionalities, <tt>false</tt> otherwise
     * @pre resourceID > 0
     * @post $none
     */
    public static boolean resourceSupportAR(int resourceID)
    {
        if (gis_ == null) {
            return false;
        }
        return gis_.resourceSupportAR(resourceID);
    }

    /**
     * Checks whether a particular resource supports Advanced Reservation
     * functionalities or not.
     * @param resourceID    a resource ID
     * @return <tt>true</tt> if a resource supports Advanced Reservation
     *         functionalities, <tt>false</tt> otherwise
     * @pre resourceID != null
     * @post $none
     */
    public static boolean resourceSupportAR(Integer resourceID)
    {
        if (gis_ == null || resourceID == null) {
            return false;
        }
        return gis_.resourceSupportAR(resourceID);
    }

    /**
     * Sends a request to Grid Information Service (GIS) entity to get the
     * list of Grid resources only that support <b>Advanced Reservation</b>
     * @return A LinkedList containing GridResource ID (as an Integer object)
     *         or <tt>null</tt> if a GIS entity hasn't been created before
     * @pre $none
     * @post $none
     */
    public static LinkedList getAdvancedReservationList()
    {
        if (gis_ == null) {
            return null;
        }

        return gis_.getAdvReservList();
    }

    /**
     * Checks whether the given GridResource ID exists or not
     * @param id    a GridResource id
     * @return <tt>true</tt> if the given ID exists, <tt>false</tt> otherwise
     * @pre id >= 0
     * @post $none
     */
    public static boolean isResourceExist(int id)
    {
        if (gis_ == null) {
            return false;
        }

        return gis_.isResourceExist(id);
    }

    /**
     * Checks whether the given GridResource ID exists or not
     * @param id    a GridResource id
     * @return <tt>true</tt> if the given ID exists, <tt>false</tt> otherwise
     * @pre id != null
     * @post $none
     */
    public static boolean isResourceExist(Integer id)
    {
        if (gis_ == null || id == null) {
            return false;
        }

        return GridSim.isResourceExist(id);
    }

    /**
     * Gets the total number of PEs (Processing Elements) from a resource
     * @param resourceID  a resource ID
     * @return total number of PE or <tt>-1</tt> if invalid resource ID
     * @pre resourceID > 0
     * @post $none
     */
    public int getNumPE(int resourceID)
    {
        if (isResourceExist(resourceID) == false) {
            return NOT_FOUND;
        }

        super.send(super.output, 0.0, GridSimTags.RESOURCE_NUM_PE,
             new IO_data( new Integer(super.get_id()), SIZE, resourceID) );

        // waiting for a response from the GridResource
        Sim_type_p tag = new Sim_type_p(GridSimTags.RESOURCE_NUM_PE);

        // only look for this type of ack
        super.sim_get_next(tag, ev_);

        int result = -1;
        try
        {
            Integer obj = (Integer) ev_.get_data();
            result = obj.intValue();
        }
        catch (Exception e) {
            System.out.println(super.get_name() +
                    ".getNumPE(): Exception error.");
        }

        return result;
    }

    /**
     * Gets the number of PEs (Processing Elements) from a resource
     * @param resourceID  a resource ID
     * @return total number of PEs or <tt>-1</tt> if invalid resource ID
     * @pre resourceID != null
     * @post $none
     */
    public int getNumPE(Integer resourceID)
    {
        if (resourceID == null) {
            return NOT_FOUND;
        }

        return getNumPE( resourceID.intValue() );
    }

    /**
     * Gets a ResourceCharacteristics object for a given GridResource ID
     * @param resourceID   the resource ID
     * @return An object of ResourceCharacteristics or <tt>null</tt> if a
     *         GridResource doesn't exist or an error occurs
     * @see gridsim.ResourceCharacteristics
     * @deprecated As of GridSim 2.1, replaced by {@link
     *             #getResourceCharacteristics(int)}
     * @pre resourceID > 0
     * @post $none
     */
    public ResourceCharacteristics GetResourceCharacteristics(int resourceID) {
        return getResourceCharacteristics(resourceID);
    }

    /**
     * Gets a ResourceCharacteristics object for a given GridResource id
     * @param resourceID   the resource ID
     * @return An object of ResourceCharacteristics or <tt>null</tt> if a
     *         GridResource doesn't exist or an error occurs
     * @see gridsim.ResourceCharacteristics
     * @pre resourceID > 0
     * @post $none
     */
    public ResourceCharacteristics getResourceCharacteristics(int resourceID)
    {
        if (gis_.isResourceExist(resourceID) == false) {
            return null;
        }

        // Get Resource Characteristic Info: Send Request and Receive Event/Msg
        send(super.output, 0.0, GridSimTags.RESOURCE_CHARACTERISTICS,
             new IO_data( new Integer(super.get_id()), SIZE, resourceID)
        );

        try {
            return (ResourceCharacteristics) receiveEventObject(this.input);
        }
        catch (Exception e) {
            System.out.println(super.get_name() +
                    ".getResourceCharacteristics(): Exception error.");
        }

        return null;
    }

    /**
     * Gets the GridResource dynamic information
     * @param resourceID   the resource ID
     * @return An object of Accumulator containing the GridResource load or
     *         <tt>null</tt> if a GridResource doesn't exist or an error occurs
     * @deprecated As of GridSim 2.1, replaced by {@link
     *             #getResourceDynamicInfo(int)}
     * @pre resourceID > 0
     * @post $none
     */
    public Accumulator GetResourceDynamicInformation(int resourceID) {
        return getResourceDynamicInfo(resourceID);
    }

    /**
     * Gets the GridResource dynamic fnformation
     * @param resourceID   the resource ID
     * @return An object of Accumulator containing the GridResource load or
     *         <tt>null</tt> if a GridResource doesn't exist or an error occurs
     * @pre resourceID > 0
     * @post $none
     */
    public Accumulator getResourceDynamicInfo(int resourceID)
    {
        if (gis_.isResourceExist(resourceID) == false) {
            return null;
        }

        // Get Resource Dynamic information
        send(super.output, 0.0, GridSimTags.RESOURCE_DYNAMICS,
             new IO_data( new Integer(super.get_id()), SIZE, resourceID )
        );

        try {
            return (Accumulator) receiveEventObject(this.input);
        }
        catch (Exception e) {
            System.out.println(super.get_name() +
                    ".getResourceDynamicInfo(): Exception error.");
        }

        return null;
    }


    ////////////  METHOD FOR RECORDING Statistics Information BEGIN ////////

    /**
     * Records statistics during the event
     * @param category  a category name
     * @param data      a value to be recorded
     * @deprecated As of GridSim 2.1, replaced by {@link
     *             #recordStatistics(String, double)}
     * @pre category != null
     * @pre data >= 0.0
     * @post $none
     */
    public void RecordStatistics(String category, double data) {
        recordStatistics(category, data);
    }

    /**
     * Records statistics during the event
     * @param category  a category name
     * @param data      a value to be recorded
     * @pre category != null
     * @pre data >= 0.0
     * @post $none
     */
    public void recordStatistics(String category, double data)
    {
        if (statsID_ == NOT_FOUND || category == null) {
            return;
        }

        super.send(statsID_, 0.0, GridSimTags.RECORD_STATISTICS,
             new Stat( GridSim.clock(), category, super.get_name(), data )
        );
    }

    /**
     * Records statistics during the event
     * @param category  a category name
     * @param data      a value to be recorded
     * @deprecated As of GridSim 2.1, replaced by {@link
     *             #recordStatistics(String, int)}
     * @pre category != null
     * @pre data >= 0.0
     * @post $none
     */
    public void RecordStatistics(String category, int data) {
        recordStatistics(category, data);
    }

    /**
     * Records statistics during the event
     * @param category  a category name
     * @param data      a value to be recorded
     * @pre category != null
     * @pre data >= 0.0
     * @post $none
     */
    public void recordStatistics(String category, int data)
    {
        if (statsID_ == NOT_FOUND || category == null) {
            return;
        }

        super.send(statsID_, 0.0, GridSimTags.RECORD_STATISTICS,
             new Stat( GridSim.clock(), category, super.get_name(), data )
        );
    }

    /**
     * Records statistics during the event
     * @param category  a category name
     * @param data      a value to be recorded
     * @deprecated As of GridSim 2.1, replaced by {@link
     *             #recordStatistics(String, String)}
     * @pre category != null
     * @pre data != null
     * @post $none
     */
    public void RecordStatistics(String category, String data) {
        recordStatistics(category, data);
    }

    /**
     * Records statistics during the event
     * @param category  a category name
     * @param data      a value to be recorded
     * @pre category != null
     * @pre data != null
     * @post $none
     */
    public void recordStatistics(String category, String data)
    {
        if (statsID_ == NOT_FOUND || category == null || data == null) {
            return;
        }

        super.send(statsID_, 0.0, GridSimTags.RECORD_STATISTICS,
             new Stat( GridSim.clock(), category, super.get_name(), data )
        );
    }

    /**
     * Records statistics during the event
     * @param category  a category name
     * @param data      a value to be recorded
     * @deprecated As of GridSim 2.1, replaced by {@link
     *             #recordStatistics(String, boolean)}
     * @pre category != null
     * @pre data == true || false
     * @post $none
     */
    public void RecordStatistics(String category, boolean data) {
        recordStatistics(category, data);
    }

    /**
     * Records statistics during the event
     * @param category  a category name
     * @param data      a value to be recorded
     * @pre category != null
     * @pre data == true || false
     * @post $none
     */
    public void recordStatistics(String category, boolean data)
    {
        if (statsID_ == NOT_FOUND || category == null) {
            return;
        }

        super.send(statsID_, 0.0, GridSimTags.RECORD_STATISTICS,
             new Stat( GridSim.clock(), category, super.get_name(), data )
        );
    }

} // end class


