/*
 * Title:        Visual Modeler for GridSim Toolkit
 * Description:  This Visual Modeler enables the user to quickly create
 *               experiments on different Grid testbeds and generate the
 *               default Grid Broker source codes (in Java).
 *
 * $Id: WholeNumberField.java,v 1.4 2003/06/15 03:25:36 anthony Exp $
 */

package visualmodeler;

import javax.swing.*;
import javax.swing.text.*;

import java.awt.Toolkit;
import java.text.NumberFormat;
import java.text.ParseException;
import java.util.Locale;


/**
 * WholeNumerField setup a text field that accepts numbers only
 *
 * @author       Anthony Sulistio and Chee Shin Yeo
 * @version      1.1
 * @invariant $none
 */
public class WholeNumberField extends JTextField 
{
    private Toolkit toolkit;
    private NumberFormat integerFormatter;


    /**
     * Constructs a text field that accepts numbers only
     * @param value    value to be placed in the text field
     * @param columns  Maximum number of numbers supported by the text field
     * @pre value >= 0
     * @pre columns >= 0
     * @post $none
     */
    public WholeNumberField(int value, int columns) 
    {
        super(columns);
        toolkit = Toolkit.getDefaultToolkit();
        integerFormatter = NumberFormat.getNumberInstance(Locale.US);
        integerFormatter.setParseIntegerOnly(true);
        setValue(value);
    }

    /**
     * Gets the value in the text field
     * @return    Value in the text field
     * @pre $none
     * @post $result >= 0
     */
    public int getValue() 
    {
        int retVal = 0;
        try {
            retVal = integerFormatter.parse(getText()).intValue();
        } 
        catch (ParseException e) 
        {
            // This should never happen because insertString allows
            // only properly formatted data to get in the field.
            toolkit.beep();
        }
        return retVal;
    }

    /**
     * Set the value in the text field
     * @param value    value to be set
     * @pre value >= 0
     * @post $none
     */
    public void setValue(int value) {
        setText( integerFormatter.format(value) );
    }

    /**
     * Create default model for text field
     * @return      Document model for text field
     * @pre $none
     * @post $result != null
     */
    protected Document createDefaultModel() {
        return new WholeNumberDocument();
    }

    ///////////////// INTERNAL CLASS /////////////
    
    /**
     * WholeNumberDocument setups the document model for the text field
     * @invariant $none
     */
    protected class WholeNumberDocument extends PlainDocument 
    {
       /**
        * Restricts only numbers can be inserted into text field
        * @param offs the starting offset >= 0
        * @param str  the string to insert; does nothing with null/empty strings
        * @param attr the attributes for the inserted content
        * @throws BadLocationException if the starting offset is wrong
        * @pre offs >= 0
        * @pre str != null
        * @pre attr != null
        * @post $none
        */
        public void insertString(int offs, String str, AttributeSet attr)
                        throws BadLocationException
        {
            char[] source = str.toCharArray();
            
            int length = source.length;
            char[] result = new char[length];
            
            int j = 0;
            for (int i = 0; i < length; i++) 
            {
                if ( Character.isDigit(source[i]) ) {
                    result[j++] = source[i];
                }
                else {
                    toolkit.beep();
                }
            }

            super.insertString(offs, new String(result, 0, j), attr);
        }
    } // end class

} // end class

