/*
 * Title:        Visual Modeler for GridSim Toolkit
 * Description:  This Visual Modeler enables the user to quickly create
 *               experiments on different Grid testbeds and generate the
 *               default Grid Broker source codes (in Java).
 *
 * $Id: UserProperty.java,v 1.18 2003/06/19 07:31:00 anthony Exp $
 */

package visualmodeler;

// to handle loading XML. These packages exist since java 1.4
import org.w3c.dom.*;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.border.*;
import java.awt.*;
import java.awt.event.*;
import java.util.*;


/**
 * UserProperty stores the values needed for each grid user
 *
 * @author       Anthony Sulistio and Chee Shin Yeo
 * @version      1.1
 * @invariant $none
 */
public class UserProperty extends JDialog
                          implements ActionListener, WindowListener
{
    // GUI part
    // a hashtable to store the objects of JTextField, 20 so far.
    private Hashtable hashText_;
    private JButton ok_, cancel_;
    private JComboBox combo_;
    private JRadioButton radioFactor_, radioValue_;
    private Toolkit toolkit_;

    // Modal part that stores the user input values
    private int hour_, min_, sec_;
    private int gridSize_, lengthSize_, fileSize_, outputSize_;
    private double gridMin_, gridMax_, lengthMin_, lengthMax_;
    private double fileMin_, fileMax_, outputMin_, outputMax_;
    private double baudRate_, delay_;
    private double bNum_, dNum_;  // storing budget and deadline value
    private int id_;
    private String name_;

    // attribute related to scheduling policy of an user
    private int comboPolicy_;
    private final String[] comboValue_ =
                      { "No Optimisation",      "Optimise Cost",
                        "Optimise Cost Plus",   "Optimise Cost and Time",
                        "Optimise Time" };

    // a flag to determine whether the GUI components has created or not
    private boolean guiHasCreated_;

    // true if prev select is for factor, false for deadline/budget value
    private boolean curRadioFactor_;

    // true if XML data have been populated
    private boolean hasLoadXml_;


    /**
     * Allocates a new UserProperty object
     * @param id        a grid user id
     * @param name      a grid user name    
     * @param random    a flag to determine whether using random values to
     *                  create a grid user's properties or not
     * @param r         a Random object
     * @pre id >= 0
     * @pre name != null
     * @pre r != null
     * @post $none
     */
    public UserProperty(int id, String name, boolean random, Random r)
    {
        super.setModal(true);   // user can't open or use another window
        super.setSize(500, 500);
        super.setLocation(100, 100);
        super.setTitle("User Property Dialog");

        if (random == true) {
            randomValue(r);
        }
        else {
            defaultValue();
        }

        id_ = id;
        name_ = name;
        guiHasCreated_ = false;
        hasLoadXml_ = false;

        // init GUI components
        cancel_ = null;
        combo_ = null;
        hashText_ = null;
        ok_ = null;
        radioFactor_ = null;
        radioValue_ = null;
        toolkit_ = null;
    }

    /**
     * Gets the grid user name
     * @return user name
     * @pre $none
     * @post $result != null
     */
    public String getUserName() {
        return name_;
    }

    /**
     * Performs a certain task when an action occurs
     * @param   evt     an ActionEvent object
     * @pre evt != null
     * @post $none
     */
    public void actionPerformed(ActionEvent evt)
    {
        String cmd = evt.getActionCommand();

        // for ok button
        if (cmd.equalsIgnoreCase("ok") == true)
        {
            if (saveValue() == false) {
                return;
            }
        }

        // for cancel button
        else {
            resetValue();
        }

        removeListeners();
        dispose();
    }

    /**
     * Shows a grid user dialog
     * @pre $none
     * @post $none
     */
    public void showDialog()
    {
        if (guiHasCreated_ == false)
        {
            guiHasCreated_ = true;
            this.addWindowListener(this);
            initComponents();
        }

        ok_.addActionListener(this);
        cancel_.addActionListener(this);

        this.show();
    }

    /**
     * Generates a Java code regarding to grid users properties
     * @param indent    indentation
     * @return a Java code in String object
     * @pre indent != null
     * @post $result != null
     */
    public String generateCode(String indent)
    {
        String seed = "seed*997*(1+" + id_ + ")+1";
        final int VAL = 100;

        // do the init
        StringBuffer code = new StringBuffer(1000);
        code.append("\n");
        code.append(indent);
        code.append("///////// Create ");
        code.append(name_);
        code.append("\n");
        code.append(indent);
        code.append("random = new Random(");
        code.append(seed);
        code.append(");\n");
        code.append(indent);
        code.append("glList = new GridletList();\n");

        // Create the gridlets
        code.append(indent);
        code.append("count = (int) GridSimRandom.real(");
        code.append(gridSize_);
        code.append(", ");
        code.append(gridMin_ / VAL);
        code.append(", ");
        code.append(gridMax_ / VAL);
        code.append(", random.nextDouble());\n\n");

        // loop to add one gridlet at the time to the linked-list
        code.append(indent);
        code.append("// A loop that creates new Gridlets\n");
        code.append(indent);
        code.append("for (int i = 0; i < count; i++) {\n");
        code.append(indent);
        code.append("    ");
        code.append("double len = GridSimRandom.real(");
        code.append(lengthSize_);
        code.append(", ");
        code.append(lengthMin_ / VAL);
        code.append(", ");
        code.append(lengthMax_ / VAL);
        code.append(", random.nextDouble());\n\n");

        code.append(indent);
        code.append("    ");
        code.append("long file = (long) GridSimRandom.real(");
        code.append(fileSize_);
        code.append(", ");
        code.append(fileMin_ / VAL);
        code.append(", ");
        code.append(fileMax_ / VAL);
        code.append(", random.nextDouble());\n\n");

        code.append(indent);
        code.append("    ");
        code.append("long out = (long) GridSimRandom.real(");
        code.append(outputSize_);
        code.append(", ");
        code.append(outputMin_ / VAL);
        code.append(", ");
        code.append(outputMax_ / VAL);
        code.append(", random.nextDouble());\n\n");

        code.append(indent);
        code.append("    // Creates a new Gridlet entity \n");
        code.append(indent);
        code.append("    ");
        code.append("Gridlet gl = new Gridlet(i, len, file, out);\n");

        code.append(indent);
        code.append("    glList.add(gl);   // Adds a new Gridlet into a list\n");
        code.append(indent);
        code.append("}\n\n");

        // create the experiment
        code.append(indent);
        code.append("expt = new Experiment(");
        code.append(id_);
        code.append(", glList, ");
        code.append(comboPolicy_);
        code.append(", ");
        code.append(curRadioFactor_);
        code.append(", ");
        code.append(dNum_);
        code.append(", ");
        code.append(bNum_);
        code.append(",\n");
        code.append(indent);
        code.append("    \"report");
        code.append(id_);
        code.append(".txt\", resourceNameList);\n\n");

        // create the user entity
        code.append(indent);
        code.append("userEntity = new UserEntity(\"User");
        code.append(id_);
        code.append("\", expt, ");
        code.append(baudRate_);
        code.append(", ");
        code.append(seed);
        code.append(",\n    ");
        code.append(indent);
        code.append(delay_);
        code.append(", false);\n");

        return code.toString();
    }

    /**
     * Generates a XML code regarding to grid users properties
     * @param spaces    number of spaces
     * @return a XML code in String object
     * @pre spaces != null
     * @post $result != null
     */
    public String saveFile(String spaces)
    {
        final int VAL = 100;
        String indent = spaces + "    ";
        String tab = indent + "    ";
        String doubleTab = tab + "    ";

        // write the user header
        StringBuffer xml = new StringBuffer(1000);
        xml.append("\n\n");
        xml.append(spaces);
        xml.append("<user>\n");

        // write the user id
        xml.append(indent);
        xml.append("<id> ");
        xml.append(id_);
        xml.append(" </id>\n");

        // write the user name
        xml.append(indent);
        xml.append("<name> ");
        xml.append(name_);
        xml.append(" </name>\n");

        // write the baud rate
        xml.append(indent);
        xml.append("<baudRate> ");
        xml.append(baudRate_);
        xml.append(" </baudRate>\n");

        // write the time that consists of hour, minute and second
        xml.append(indent);
        xml.append("<time>\n");
        xml.append(tab);
        xml.append("<hour> ");
        xml.append(hour_);
        xml.append(" </hour>\n");
        xml.append(tab);
        xml.append("<minute> ");
        xml.append(min_);
        xml.append(" </minute>\n");
        xml.append(tab);
        xml.append("<second> ");
        xml.append(sec_);
        xml.append(" </second>\n");
        xml.append(indent);
        xml.append("</time>\n");

        // write the delay
        xml.append(indent);
        xml.append("<delay> ");
        xml.append(delay_);
        xml.append(" </delay>\n");

        // write scheduling strategy - choose one from the 5 options
        xml.append(indent);
        xml.append("<schedulingStrategy> ");
        xml.append(comboValue_[comboPolicy_]);
        xml.append(" </schedulingStrategy>\n");

        // write the gridlet property
        xml.append(indent);
        xml.append("<gridletProperty>\n");

        // write the gridlet size, min deviation and max deviation
        xml.append(tab);
        xml.append("<gridlet>\n");
        xml.append(doubleTab);
        xml.append("<size> ");
        xml.append(gridSize_);
        xml.append(" </size>\n");
        xml.append(doubleTab);
        xml.append("<minDeviation> ");
        xml.append( gridMin_ / VAL );
        xml.append(" </minDeviation>\n");
        xml.append(doubleTab);
        xml.append("<maxDeviation> ");
        xml.append( gridMax_ / VAL );
        xml.append(" </maxDeviation>\n");
        xml.append(tab);
        xml.append("</gridlet>\n");

        // write the length size, min deviation and max deviation
        xml.append(tab);
        xml.append("<length>\n");
        xml.append(doubleTab);
        xml.append("<size> ");
        xml.append(lengthSize_);
        xml.append(" </size>\n");
        xml.append(doubleTab);
        xml.append("<minDeviation> ");
        xml.append( lengthMin_ / VAL );
        xml.append(" </minDeviation>\n");
        xml.append(doubleTab);
        xml.append("<maxDeviation> ");
        xml.append( lengthMax_ / VAL );
        xml.append(" </maxDeviation>\n");
        xml.append(tab);
        xml.append("</length>\n");

        // write the file size, min deviation and max deviation
        xml.append(tab);
        xml.append("<file>\n");
        xml.append(doubleTab);
        xml.append("<size> ");
        xml.append(fileSize_);
        xml.append(" </size>\n");
        xml.append(doubleTab);
        xml.append("<minDeviation> ");
        xml.append(fileMin_ / VAL);
        xml.append(" </minDeviation>\n");
        xml.append(doubleTab);
        xml.append("<maxDeviation> ");
        xml.append(fileMax_ / VAL);
        xml.append(" </maxDeviation>\n");
        xml.append(tab);
        xml.append("</file>\n");

        // write the output size, min deviation and max deviation
        xml.append(tab);
        xml.append("<output>\n");
        xml.append(doubleTab);
        xml.append("<size> ");
        xml.append(outputSize_);
        xml.append(" </size>\n");
        xml.append(doubleTab);
        xml.append("<minDeviation> ");
        xml.append(outputMin_ / VAL);
        xml.append(" </minDeviation>\n");
        xml.append(doubleTab);
        xml.append("<maxDeviation> ");
        xml.append(outputMax_ / VAL);
        xml.append(" </maxDeviation>\n");
        xml.append(tab);
        xml.append("</output>\n");
        xml.append(indent);
        xml.append("</gridletProperty>\n");

        // write the budget and deadline property
        xml.append(indent);
        xml.append("<budgetDeadlineProperty>\n");

        // choose either it is a factor or a value
        if (curRadioFactor_ == true) 
        {
            xml.append(tab);
            xml.append("<type> factor </type>\n");
        }    
        else 
        {
            xml.append(tab);
            xml.append("<type> value </type>\n");
        }    

        // write the budget and deadline. don't forget the closing tag
        xml.append(tab);
        xml.append("<budget> ");
        xml.append(bNum_);
        xml.append(" </budget>\n");
        xml.append(tab);
        xml.append("<deadline> ");
        xml.append(dNum_);
        xml.append(" </deadline>\n");

        // don't forget the closing tags
        xml.append(indent);
        xml.append("</budgetDeadlineProperty>\n");
        xml.append(spaces);
        xml.append("</user>\n");
            
        return xml.toString();
    }

    /**
     * Checks whether XML file has been loaded or not
     * @return <tt>true</tt> if it has been loaded, <tt>false</tt> otherwise
     * @pre $none
     * @post $none
     */
    public boolean hasLoadXml() {
        return hasLoadXml_;
    }

    /**
     * Loads a XML code regarding to grid users
     * @param nodeList  a NodeList object
     * @pre nodeList != null
     * @post $none
     */
    public void loadXml(final NodeList nodeList)
    {
        hasLoadXml_ = true;
        Node node;
        String name, value;

        int length = nodeList.getLength();
        for (int i = 0; i < length; i++)
        {
            node = nodeList.item(i);

            // only element nodes that need to be take care,
            // the rests are ignored
            if (node.getNodeType() != Node.ELEMENT_NODE) {
                continue;
            }

            name = node.getNodeName();
            if (name.equals("id") == true)
            {
                value = node.getFirstChild().getNodeValue();

                // must trim the String otherwise get an exception
                int id = Integer.parseInt(value.trim());
                id_ = id;
            }

            else if (name.equals("name") == true)
            {
                value = node.getFirstChild().getNodeValue();
                name_ = value.trim();
            }

            else if (name.equals("baudRate") == true)
            {
                value = node.getFirstChild().getNodeValue();

                // must trim the String otherwise get an exception
                double num = Double.parseDouble(value.trim());
                baudRate_ = num;
            }

            else if (name.equals("time") == true)
            {
                // make another method to handle time
                loadXmlTime( node.getChildNodes() );
            }

            else if (name.equals("delay") == true)
            {
                value = node.getFirstChild().getNodeValue();

                // must trim the String otherwise get an exception
                double num = Double.parseDouble(value.trim());
                delay_ = num;
            }

            else if (name.equals("schedulingStrategy") == true)
            {
                // make another method to handle scheduling strategy
                value = node.getFirstChild().getNodeValue();
                loadXmlPolicy( value.trim() );
            }

            else if (name.equals("budgetDeadlineProperty") == true)
            {
                // make another method to handle budget and deadline
                loadXmlBudgetDeadline( node.getChildNodes() );
            }

            else if (name.equals("gridletProperty") == true)
            {
                // make another method to handle gridlet
                loadXmlGridlet( node.getChildNodes() );
            }

        }
    }

    /** 
     * An event that occurs when the window dialog is closing
     * @param evt   a WindowEvent object
     * @pre evt != null
     * @post $none
     */
    public void windowClosing(WindowEvent evt)
    {
        resetValue();
        removeListeners();
    }

    /** 
     * An event that occurs when the window dialog is closed
     * @param evt   a WindowEvent object
     * @pre evt != null
     * @post $none
     */ 
    public void windowClosed(WindowEvent evt) {
        //empty - not needed but have to due to WindowListener interface
    }

    /** 
     * An event that occurs when the window dialog is opened
     * @param evt   a WindowEvent object
     * @pre evt != null
     * @post $none
     */
    public void windowOpened(WindowEvent evt) {
        //empty - not needed but have to due to WindowListener interface
    }

    /** 
     * An event that occurs when the window dialog is iconified
     * @param evt   a WindowEvent object
     * @pre evt != null
     * @post $none
     */
    public void windowIconified(WindowEvent evt) {
        //empty - not needed but have to due to WindowListener interface
    }

    /** 
     * An event that occurs when the window dialog is deiconified
     * @param evt   a WindowEvent object
     * @pre evt != null
     * @post $none
     */
    public void windowDeiconified(WindowEvent evt) {
        //empty - not needed but have to due to WindowListener interface
    }

    /** 
     * An event that occurs when the window dialog is activated
     * @param evt   a WindowEvent object
     * @pre evt != null
     * @post $none
     */
    public void windowActivated(WindowEvent evt) {
        //empty - not needed but have to due to WindowListener interface
    }

    /** 
     * An event that occurs when the window dialog is deactivated
     * @param evt   a WindowEvent object
     * @pre evt != null
     * @post $none
     */
    public void windowDeactivated(WindowEvent evt) {
        //empty - not needed but have to due to WindowListener interface
    }


    //////////////////////////////////// PRIVATE METHODS ////////////////////

    /**
     * Allocates all grid user properties with default value
     * @pre $none
     * @post $none
     */
    private void defaultValue()
    {
        delay_ = comboPolicy_ = hour_ = min_ = sec_ = 0;
        gridSize_ = lengthSize_ = outputSize_ = fileSize_ = 0;
        gridMin_ = gridMax_ = lengthMin_ = lengthMax_ = 0.0;
        fileMin_ = fileMax_ = outputMin_ = outputMax_ = 0.0;

        curRadioFactor_ = false;
        baudRate_ = bNum_ = dNum_ = 100.0;
    }

    /**
     * Allocates all grid user properties with random values
     * @param r     a Random object
     * @pre r != null
     * @post $none
     */
    private void randomValue(Random r)
    {
        // Maximum value. Note: if MAX changed, pls modify bNum_ and dNum_
        final int MAX = 101;
        final double DIVIDER = 10.0;

        // with Random.nextInt(MAX) == Random.nextInt(100) if MAX = 101
        baudRate_ = r.nextInt(MAX*6);
        delay_ = r.nextInt(MAX);

        hour_ = r.nextInt(MAX);
        min_ = r.nextInt(MAX);
        sec_ = r.nextInt(MAX);

        gridSize_ = r.nextInt(MAX);
        gridMin_ = r.nextInt(MAX) / DIVIDER;
        gridMax_ = r.nextInt(MAX) / DIVIDER;

        lengthSize_ = r.nextInt(MAX*MAX);
        lengthMin_ = r.nextInt(MAX) / DIVIDER;
        lengthMax_ = r.nextInt(MAX) / DIVIDER;

        fileSize_ = r.nextInt(MAX);
        fileMin_ = r.nextInt(MAX) / DIVIDER;
        fileMax_ = r.nextInt(MAX) / DIVIDER;

        outputSize_ = r.nextInt(MAX);
        outputMin_ = r.nextInt(MAX) / DIVIDER;
        outputMax_ = r.nextInt(MAX) / DIVIDER;

        // for budget and deadline value
        // if it is false, then it's deadline and budget value.
        curRadioFactor_ = false;

        // if it is b-factor and d-factor, then the range is [0.0, 1.0]
        if (curRadioFactor_ == true)
        {
            // get integer value up to MAX = 100, then divide by 100
            bNum_ = r.nextInt(MAX) / (DIVIDER * DIVIDER);
            dNum_ = r.nextInt(MAX) / (DIVIDER * DIVIDER);
        }
        else
        {
            bNum_ = r.nextInt(MAX*MAX);
            dNum_ = r.nextInt(MAX*MAX);
        }

        // there are only 5 options, but the index starts at 0
        comboPolicy_ = r.nextInt(comboValue_.length);
    }

    /**
     * Initializes all GUI component for grid user dialog
     * @pre $none
     * @post $none
     */
    private void initComponents()
    {
        hashText_ = new Hashtable();
        toolkit_ = Toolkit.getDefaultToolkit();

        GridBagLayout gridbag = new GridBagLayout();
        GridBagConstraints c = new GridBagConstraints();
        Font font = new Font(null, Font.PLAIN, 14);
        JPanel panel = new JPanel(gridbag);

        int index = 0;
        c.insets = new Insets(10, 3, 10, 2);  // top, left, buttom, right
        c.weighty = 0.5;
        c.fill = GridBagConstraints.HORIZONTAL;
        c.gridy = index++;

        //////////////////
        // Display the user name
        JLabel label1 = new JLabel("User Name:");
        label1.setFont(font);
        panel.add(label1, c);

        c.gridwidth = GridBagConstraints.REMAINDER;
        JTextField text = new JTextField("" + name_, 20);
        panel.add(text, c);
        hashText_.put("user", text);

        ////////////////////
        // Display the baud rate
        c.gridwidth = 1;
        c.gridy = index++;
        JLabel label2 = new JLabel("Baud Rate:");
        label2.setFont(font);
        panel.add(label2, c);

        text = new JTextField("" + baudRate_, 4);
        panel.add(text, c);
        hashText_.put("baud", text);

        ///////////////////////
        // Display the max. simulation time
        c.gridy = index++;
        JLabel label3 = new JLabel("Max. simulation time:");
        label3.setFont(font);
        panel.add(label3, c);

        c.ipadx = 5;
        text = new JTextField("" + hour_, 3);
        panel.add(text, c);
        hashText_.put("hour", text);
        panel.add(new JLabel(" hour"), c);

        text = new JTextField("" + min_, 3);
        panel.add(text, c);
        hashText_.put("min", text);
        panel.add(new JLabel(" minute"), c);

        text = new JTextField("" + sec_, 3);
        panel.add(text, c);
        hashText_.put("sec", text);
        panel.add(new JLabel(" second"), c);

        /////////////////////////
        // Display the delay
        c.gridy = index++;
        c.ipadx = 0;
        JLabel label4 = new JLabel("Successive experiment delay: ");
        label4.setFont(font);
        panel.add(label4, c);

        text = new JTextField("" + delay_, 4);
        panel.add(text, c);
        hashText_.put("delay", text);
        panel.add(new JLabel("second"), c);

        ////////////
        c.gridy = index++;
        JLabel label = new JLabel("Scheduling strategy:");
        label.setFont(font);
        panel.add(label, c);

        // Create a combo box
        combo_ = new JComboBox(comboValue_);
        combo_.setSelectedIndex(comboPolicy_);  // default value is Random
        c.gridwidth = 4;
        panel.add(combo_, c);

        //////////////
        c.weighty = 1.0;   //request any extra vertical space
        c.gridy = index++;
        c.insets = new Insets(30, 0, 0, 0);  //top padding
        c.gridwidth = GridBagConstraints.REMAINDER;
        c.fill = GridBagConstraints.BOTH;

        JPanel gridletPanel = new JPanel(new GridLayout(5, 4, 15, 15));
        panel.add(gridletPanel, c);
        createGridlet(gridletPanel);

        //////////////
        c.gridy = index++;
        JPanel budgetDeadlinePanel = new JPanel(new BorderLayout(5, 5));
        panel.add(budgetDeadlinePanel, c);
        createBudgetDeadline(budgetDeadlinePanel);

        ////////////////
        // Create an OK button
        c.gridy = index++;
        c.gridx = 0;
        c.gridwidth = 1;
        c.fill = GridBagConstraints.NONE;
        ok_ = new JButton("OK");
        panel.add(ok_, c);

        // Create a Cancel button
        c.gridwidth = GridBagConstraints.REMAINDER;
        cancel_ = new JButton("Cancel");
        panel.add(cancel_, c);

        ///////////////////
        // Add panel to a scroll pane
        int vert = ScrollPaneConstants.VERTICAL_SCROLLBAR_ALWAYS;
        int horiz = ScrollPaneConstants.HORIZONTAL_SCROLLBAR_AS_NEEDED;
        JScrollPane scroll = new JScrollPane(panel, vert, horiz);
        getContentPane().add(scroll);
    }

    /**
     * Creates a panel about budget and deadline factors
     * @param panel     a JPanel object
     * @pre panel != null
     * @post $none
     */
    private void createBudgetDeadline(JPanel panel)
    {
        Font font = new Font(null, Font.PLAIN, 12);

        // Create a border and its title
        TitledBorder title = BorderFactory.createTitledBorder(
                                    "Budget and Deadline");
        title.setTitleFont(font.deriveFont(Font.BOLD, 16));
        panel.setBorder(title);

        ///////////////
        radioFactor_ = new JRadioButton("Factor-based");
        radioFactor_.setFont(font);
        radioFactor_.setActionCommand("factor");

        JPanel northPanel = new JPanel();
        northPanel.add(radioFactor_);

        //////////////
        radioValue_ = new JRadioButton("Value-based");
        radioValue_.setFont(font);
        radioValue_.setActionCommand("value");

        northPanel.add(radioValue_);
        panel.add(northPanel, BorderLayout.NORTH);

        // Group the radio buttons into one
        ButtonGroup group = new ButtonGroup();
        group.add(radioFactor_);
        group.add(radioValue_);

        if (curRadioFactor_ == true) {
            radioFactor_.setSelected(true);
        }
        else {
            radioValue_.setSelected(true);
        }

        ////////
        JPanel centerPanel = new JPanel(new GridBagLayout());
        createCenterComponent(centerPanel);
        panel.add(centerPanel, BorderLayout.CENTER);

        StringBuffer str = new StringBuffer(100);
        str.append("   Note: For factor-based, the range for both budget");
        str.append(" and deadline is [0.0, 1.0]");
        JLabel label = new JLabel( str.toString() );
        label.setFont(font);
        panel.add(label, BorderLayout.SOUTH);
    }

    /**
     * Creates a panel component
     * @param panel     a JPanel object
     * @pre panel != null
     * @post $none
     */
    private void createCenterComponent(JPanel panel)
    {
        Font font = new Font(null, Font.PLAIN, 12);
        GridBagConstraints c = new GridBagConstraints();

        int index = 0;
        c.gridy = index++;
        c.weighty = 1.0;
        c.fill = GridBagConstraints.HORIZONTAL;
        c.insets = new Insets(10, 5, 10, 10);  // top, left, buttom, right
        JLabel label = new JLabel("  Budget: ");
        label.setFont(font);
        panel.add(label, c);

        JTextField text = new JTextField("" + bNum_, 10);
        panel.add(text, c);
        hashText_.put("budget", text);

        //////////
        c.gridy = index++;
        label = new JLabel("  Deadline: ");
        label.setFont(font);
        panel.add(label, c);

        text = new JTextField("" + dNum_, 10);
        panel.add(text, c);
        hashText_.put("deadline", text);
    }

    /**
     * Creates a panel about Gridlets 
     * @param panel     a JPanel object
     * @pre panel != null
     * @post $none
     */
    private void createGridlet(JPanel panel)
    {
        Font font = new Font(null, Font.PLAIN, 12);
        GridBagConstraints c = new GridBagConstraints();

        // Create a border and its title
        TitledBorder title = BorderFactory.createTitledBorder("Gridlet");
        title.setTitleFont(font.deriveFont(Font.BOLD, 16));
        panel.setBorder(title);

        JLabel label = new JLabel("blah");
        label.setVisible(false);
        panel.add(label);

        int index = 0;
        c.gridy = index++;
        c.weighty = 1.0;
        c.gridwidth = 1;
        c.fill = GridBagConstraints.HORIZONTAL;
        c.insets = new Insets(10, 5, 10, 10);  // top, left, buttom, right
        label = new JLabel("           Size");
        label.setFont(font);
        panel.add(label);

        label = new JLabel("Min. Deviation (%)");
        label.setFont(font);
        panel.add(label);

        label = new JLabel("Max. Devation (%)");
        label.setFont(font);
        panel.add(label);

        c.gridy = index++;
        label = new JLabel("        Gridlet:");
        label.setFont(font);
        panel.add(label);

        JTextField text = new JTextField("" + gridSize_, 5);
        panel.add(text);
        hashText_.put("grid_size", text);

        text = new JTextField("" + gridMin_, 5);
        panel.add(text);
        hashText_.put("grid_min", text);

        text = new JTextField("" + gridMax_, 5);
        panel.add(text);
        hashText_.put("grid_max", text);

        /////////////////
        label = new JLabel("        Length:");
        label.setFont(font);
        panel.add(label);

        text = new JTextField("" + lengthSize_, 5);
        panel.add(text);
        hashText_.put("length_size", text);

        text = new JTextField("" + lengthMin_, 5);
        panel.add(text);
        hashText_.put("length_min", text);

        text = new JTextField("" + lengthMax_, 5);
        panel.add(text);
        hashText_.put("length_max", text);

        ////////////
        label = new JLabel("        File:");
        label.setFont(font);
        panel.add(label);

        text = new JTextField("" + fileSize_, 5);
        panel.add(text);
        hashText_.put("file_size", text);

        text = new JTextField("" + fileMin_, 5);
        panel.add(text);
        hashText_.put("file_min", text);

        text = new JTextField("" + fileMax_, 5);
        panel.add(text);
        hashText_.put("file_max", text);

        /////////////////
        label = new JLabel("        Output:");
        label.setFont(font);
        panel.add(label);

        text = new JTextField("" + outputSize_, 5);
        panel.add(text);
        hashText_.put("output_size", text);

        text = new JTextField("" + outputMin_, 5);
        panel.add(text);
        hashText_.put("output_min", text);

        text = new JTextField("" + outputMax_, 5);
        panel.add(text);
        hashText_.put("output_max", text);
    }

    /**
     * Removes all GUI listeners
     * @pre $none
     * @post $none
     */
    private void removeListeners()
    {
        ok_.removeActionListener(this);
        cancel_.removeActionListener(this);
    }

    /**
     * Saves the current values
     * @return <tt>true</tt> if the values have been saved successfully,
     *         <tt>false</tt> otherwise
     * @pre $none
     * @post $none
     */
    private boolean saveValue()
    {
        boolean flag = true;
        comboPolicy_ = combo_.getSelectedIndex();
        name_ = ( (JTextField) hashText_.get("user") ).getText();

        String errorMsg = "";  // a flag to denote where the error comes from

        try
        {
            errorMsg = "baud rate";
            JTextField text = (JTextField) hashText_.get("baud");
            baudRate_ = checkValue(baudRate_, text);

            errorMsg = "delay";
            text = (JTextField) hashText_.get("delay");
            delay_ = checkValue(delay_, text);

            errorMsg = "hour";
            text = (JTextField) hashText_.get("hour");
            hour_ = checkValue(hour_, text);

            errorMsg = "minute";
            text = (JTextField) hashText_.get("min");
            min_ = checkValue(min_, text);

            errorMsg = "second";
            text = (JTextField) hashText_.get("sec");
            sec_ = checkValue(sec_, text);

            /////////////
            errorMsg = "gridlet size";
            text = (JTextField) hashText_.get("grid_size");
            gridSize_ = checkValue(gridSize_, text);

            errorMsg = "gridlet min deviation";
            text = (JTextField) hashText_.get("grid_min");
            gridMin_ = checkValue(gridMin_, text, "deviation");

            errorMsg = "gridlet max deviation";
            text = (JTextField) hashText_.get("grid_max");
            gridMax_ = checkValue(gridMax_, text, "deviation");

            ///////////////
            errorMsg = "length size";
            text = (JTextField) hashText_.get("length_size");
            lengthSize_ = checkValue(lengthSize_, text);

            errorMsg = "length min deviation";
            text = (JTextField) hashText_.get("length_min");
            lengthMin_ = checkValue(lengthMin_, text, "deviation");

            errorMsg = "length max deviation";
            text = (JTextField) hashText_.get("length_max");
            lengthMax_ = checkValue(lengthMax_, text, "deviation");

            ////////////////
            errorMsg = "file size";
            text = (JTextField) hashText_.get("file_size");
            fileSize_ = checkValue(fileSize_, text);

            errorMsg = "file min deviation";
            text = (JTextField) hashText_.get("file_min");
            fileMin_ = checkValue(fileMin_, text, "deviation");

            errorMsg = "file max deviation";
            text = (JTextField) hashText_.get("file_max");
            fileMax_ = checkValue(fileMax_, text, "deviation");

            /////////////////
            errorMsg = "output size";
            text = (JTextField) hashText_.get("output_size");
            outputSize_ = checkValue(outputSize_, text);

            errorMsg = "output min deviation";
            text = (JTextField) hashText_.get("output_min");
            outputMin_ = checkValue(outputMin_, text, "deviation");

            errorMsg = "output max deviation";
            text = (JTextField) hashText_.get("output_max");
            outputMax_ = checkValue(outputMax_, text, "deviation");

            //////////////////
            errorMsg = "budget";
            text = (JTextField) hashText_.get("budget");
            bNum_ = checkValue(bNum_, text, "factor");

            errorMsg = "deadline";
            text = (JTextField) hashText_.get("deadline");
            dNum_ = checkValue(dNum_, text, "factor");

            if (radioFactor_.isSelected() == true) {
                curRadioFactor_ = true;
            }
            else {
                curRadioFactor_ = false;
            }
        }

        catch (NumberFormatException e)
        {
            toolkit_.beep();
            JOptionPane.showMessageDialog(this,
                    "Invalid value for number of " + errorMsg + ".",
                    errorMsg + " value error", JOptionPane.ERROR_MESSAGE);

            flag = false;
        }

        catch (Exception e)
        {
            toolkit_.beep();
            JOptionPane.showMessageDialog( this, e.getMessage(),
                    errorMsg + " value error", JOptionPane.ERROR_MESSAGE );

            flag = false;
        }

        return flag;
    }

    /**
     * Checks whether the given value is correct or not
     * @param value     given number
     * @param text      a JTextField object
     * @return the correct number
     * @throws Exception if the given number is incorrect
     * @pre text != null
     * @post $result >= 0
     */
    private double checkValue(double value, JTextField text) throws Exception
    {
        value = new Double( text.getText() ).doubleValue();

        if (value < 0.0) {
            throw new Exception("Invalid for having negative value.");
        }

        return value;
    }

    /**
     * Another overloaded method. 
     * Checks whether the given value is correct or not
     * @param value     given number
     * @param text      a JTextField object
     * @return the correct number
     * @throws Exception if the given number is incorrect
     * @pre text != null
     * @post $result >= 0
     */
    private int checkValue(int value, JTextField text) throws Exception
    {
        value = new Integer( text.getText() ).intValue();

        if (value < 0) {
            throw new Exception("Invalid for having negative value.");
        }

        return value;
    }

    /**
     * Another overloaded method. 
     * Checks whether the given value is correct or not
     * @param value     given number
     * @param text      a JTextField object
     * @param type      type of a JTextField object
     * @return the correct number
     * @throws Exception if the given number is incorrect
     * @pre text != null
     * @pre type != null
     * @post $result >= 0
     */
    private double checkValue(double value, JTextField text, String type)
                        throws Exception
    {
        value = new Double( text.getText() ).doubleValue();

        if (value < 0.0) {
            throw new Exception("Invalid for having negative value.");
        }
        
        if (type.equals("deviation") == true)
        {
            if (value > 100.0) {
                throw new Exception("Invalid value for exceeding 100.");
            }
        }

        else if (type.equals("factor") == true)
        {
            if (radioFactor_.isSelected() == true)
            {
                if (value > 1.0) {
                    throw new Exception("Invalid value for exceeding 1.");
                }
             }
        }

        return value;
    }

    /**
     * Resets all the values
     * @pre $none 
     * @post $none
     */
    private void resetValue()
    {
        resetTextValue(name_, "user");
        resetTextValue(baudRate_, "baud");
        resetTextValue(delay_, "delay");
        resetTextValue(hour_, "hour");
        resetTextValue(min_, "min");
        resetTextValue(sec_, "sec");
        resetTextValue(bNum_, "budget");
        resetTextValue(dNum_, "deadline");

        resetTextValue(gridSize_, "grid_size");
        resetTextValue(gridMin_, "grid_min");
        resetTextValue(gridMax_, "grid_max");

        resetTextValue(lengthSize_, "length_size");
        resetTextValue(lengthMin_, "length_min");
        resetTextValue(lengthMax_, "length_max");

        resetTextValue(outputSize_, "output_size");
        resetTextValue(outputMin_, "output_min");
        resetTextValue(outputMax_, "output_max");

        resetTextValue(fileSize_, "file_size");
        resetTextValue(fileMin_, "file_min");
        resetTextValue(fileMax_, "file_max");

        combo_.setSelectedIndex(comboPolicy_);

        if (curRadioFactor_ == true) {
            radioFactor_.setSelected(true);
        }
        else {
            radioValue_.setSelected(true);
        }
    }

    /**
     * Resets a given key object with the specified value
     * @param value  object value 
     * @param key    object key for a hash table
     * @pre value != null
     * @pre key != null
     * @post $none
     */
    private void resetTextValue(String value, String key)
    {
        JTextField text = (JTextField) hashText_.get(key);
        text.setText(value);
        hashText_.put(key, text);
    }

    /**
     * Overloaded method. Resets a given key object with the specified value.
     * @param num    object value
     * @param key    object key for a hash table
     * @pre num >= 0.0
     * @pre key != null
     * @post $none
     */
    private void resetTextValue(double num, String key)
    {
        JTextField text = (JTextField) hashText_.get(key);
        text.setText("" + num);
        hashText_.put(key, text);
    }

    /**
     * Overloaded method. Resets a given key object with the specified value.
     * @param num    object value
     * @param key    object key for a hash table
     * @pre num >= 0
     * @pre key != null
     * @post $none
     */
    private void resetTextValue(int num, String key)
    {
        JTextField text = (JTextField) hashText_.get(key);
        text.setText("" + num);
        hashText_.put(key, text);
    }

    /**
     * Loads a XML code regarding to time allocation
     * @param nodeList  a NodeList object
     * @pre nodeList != null
     * @post $none
     */
    private void loadXmlTime(final NodeList nodeList)
    {
        Node node;
        String name, value;

        int length = nodeList.getLength();
        for (int i = 0; i < length; i++)
        {
            node = nodeList.item(i);

            // only element nodes that need to be take care,
            // the rests are ignored
            if (node.getNodeType() != Node.ELEMENT_NODE) {
                continue;
            }

            name = node.getNodeName();
            if (name.equals("hour") == true)
            {
                value = node.getFirstChild().getNodeValue();

                // must trim the String otherwise get an exception
                int id = Integer.parseInt(value.trim());
                hour_ = id;
            }

            else if (name.equals("minute") == true)
            {
                value = node.getFirstChild().getNodeValue();

                // must trim the String otherwise get an exception
                int id = Integer.parseInt(value.trim());
                min_ = id;
            }

            else if (name.equals("second") == true)
            {
                value = node.getFirstChild().getNodeValue();

                // must trim the String otherwise get an exception
                int id = Integer.parseInt(value.trim());
                sec_ = id;
            }
        }

    }

    /**
     * Loads a XML code regarding to a scheduling policy
     * @param value     a scheduling policy
     * @pre value != null
     * @post $none
     */
    private void loadXmlPolicy(String value)
    {
        int i = 0;
        int length = comboValue_.length;
        
        for (i = 0; i < length; i++)
        {
            if (comboValue_[i].equals(value) == true) {
                break;
            }
        }

        if (i == comboValue_.length) {
            i = 0;
        }

        comboPolicy_ = i;
    }

    /**
     * Loads a XML code regarding to budget and deadline properties
     * @param nodeList  a NodeList object
     * @pre nodeList != null
     * @post $none
     */
    private void loadXmlBudgetDeadline(final NodeList nodeList)
    {
        Node node;
        String name, value;

        int length = nodeList.getLength();
        for (int i = 0; i < length; i++)
        {
            node = nodeList.item(i);

            // only element nodes that need to be take care,
            // the rests are ignored
            if (node.getNodeType() != Node.ELEMENT_NODE) {
                continue;
            }

            name = node.getNodeName();
            if (name.equals("type") == true)
            {
                value = node.getFirstChild().getNodeValue().trim();
                if (value.equals("factor") == true) {
                    curRadioFactor_ = true;
                }
                else {
                    curRadioFactor_ = false;
                }
            }

            else if (name.equals("budget") == true)
            {
                value = node.getFirstChild().getNodeValue();

                // must trim the String otherwise get an exception
                double num = Double.parseDouble(value.trim());
                bNum_ = num;
            }

            else if (name.equals("deadline") == true)
            {
                value = node.getFirstChild().getNodeValue();

                // must trim the String otherwise get an exception
                double num = Double.parseDouble(value.trim());
                dNum_ = num;
            }
        }
    }

    /**
     * Loads a XML code regarding to Gridlets
     * @param nodeList  a NodeList object
     * @pre nodeList != null
     * @post $none
     */
    private void loadXmlGridlet(final NodeList nodeList)
    {
        Node node;
        String name;

        int length = nodeList.getLength();
        for (int i = 0; i < length; i++)
        {
            node = nodeList.item(i);

            // only element nodes that need to be take care,
            // the rests are ignored
            if (node.getNodeType() != Node.ELEMENT_NODE) {
                continue;
            }

            name = node.getNodeName();
            loadXmlGridletProperty(node.getChildNodes(), name);
        }

    }

    /**
     * Loads a XML code regarding to 
     * @param nodeList  a NodeList object
     * @param type      object type
     * @pre nodeList != null
     * @pre type != null
     * @post $none
     */
    private void loadXmlGridletProperty(final NodeList nodeList,
                        final String type)
    {
        Node node;
        String name, value;
        final int PERCENT = 100;

        int length = nodeList.getLength();
        for (int i = 0; i < length; i++)
        {
            node = nodeList.item(i);

            // only element nodes that need to be take care,
            // the rests are ignored
            if (node.getNodeType() != Node.ELEMENT_NODE) {
                continue;
            }

            name = node.getNodeName();
            if (name.equals("size") == true)
            {
                value = node.getFirstChild().getNodeValue();

                // must trim the String otherwise get an exception
                int num = Integer.parseInt(value.trim());

                if (type.equals("gridlet") == true) {
                    gridSize_ = num;
                }
                else if (type.equals("length") == true) {
                    lengthSize_ = num;
                }
                else if (type.equals("file") == true) {
                    fileSize_ = num;
                }
                else if (type.equals("output") == true) {
                    outputSize_ = num;
                }
            }

            else if (name.equals("minDeviation") == true)
            {
                value = node.getFirstChild().getNodeValue();

                // must trim the String otherwise get an exception
                double num = Double.parseDouble(value.trim());

                if (type.equals("gridlet") == true) {
                    gridMin_ = num * PERCENT;
                }
                else if (type.equals("length") == true) {
                    lengthMin_ = num * PERCENT;
                }
                else if (type.equals("file") == true) {
                    fileMin_ = num * PERCENT;
                }
                else if (type.equals("output") == true) {
                    outputMin_ = num * PERCENT;
                }
            }

            else if (name.equals("maxDeviation") == true)
            {
                value = node.getFirstChild().getNodeValue();

                // must trim the String otherwise get an exception
                double num = Double.parseDouble(value.trim());

                if (type.equals("gridlet") == true) {
                    gridMax_ = num * PERCENT;
                }
                else if (type.equals("length") == true) {
                    lengthMax_ = num * PERCENT;
                }
                else if (type.equals("file") == true) {
                    fileMax_ = num * PERCENT;
                }
                else if (type.equals("output") == true) {
                    outputMax_ = num * PERCENT;
                }
            }
        }
    }

} // end class

