/*
 * Title:        Visual Modeler for GridSim Toolkit
 * Description:  This Visual Modeler enables the user to quickly create
 *               experiments on different Grid testbeds and generate the
 *               default Grid Broker source codes (in Java).
 *
 * $Id: UserModel.java,v 1.14 2003/06/19 07:31:00 anthony Exp $
 */

package visualmodeler;

import java.util.Vector;
import java.util.Random;
import java.util.Observable;

// to handle loading XML. These packages exist since java 1.4
import org.w3c.dom.*;


/**
 * UserModel stores and creates the user objects
 *
 * @author       Anthony Sulistio and Chee Shin Yeo
 * @version      1.1
 * @invariant $none
 */
public class UserModel extends Observable
{
    private Vector user_;
    private int userCounter_;

    // a flag to know whether the values has been changed since the last save
    private boolean hasChanged_;

    
    /**
     * Allocates a new UserModel object
     * @pre $none
     * @post $none
     */
    public UserModel()
    {
        userCounter_ = 1;   // so the user counter starts at 1 not 0
        user_ = new Vector();
        hasChanged_ = false;
    }

    /**
     * Gets all grid user names
     * @return a list of names in Vector object
     * @pre $none
     * @post $result != null
     */
    public Vector getAllUserName()
    {
        Vector name = new Vector();
        int size = user_.size();
        
        for (int i = 0; i < size; i++)
        {
            String temp = ( (UserProperty) user_.elementAt(i) ).getUserName();
            name.add(temp);
        }

        return name;
    }

    /**
     * Gets the grid user counter
     * @return a counter
     * @pre $none
     * @post $result >= 0
     */
    public int getUserCounter() {
        return userCounter_;
    }

    /**
     * Gets the total number of grid users
     * @return total number 
     * @pre $none
     * @post $result >= 0
     */
    public int getTotalUser() {
        return user_.size();
    }

    /**
     * Creates a given number of grid users
     * @param num   how many grid users to be created
     * @param random    a flag to indicate whether random values are needed or
     *                  not
     * @pre num >= 0
     * @post $none
     */
    public void createUser(int num, boolean random)
    {
        // if creating zero user, then ignore
        if (num == 0) {
            return;
        }

        String name = null;
        Random r = new Random();

        for (int i = 0; i < num; i++)
        {
            name = "User " + userCounter_;
            UserProperty property = new UserProperty(userCounter_, name,
                                            random, r);
            user_.add(property);
            userCounter_++;
        }

        hasChanged_ = true;
        setChanged();
        notifyObservers();
    }

    /**
     * Removes all grid users
     * @pre $none
     * @post $none
     */
    public void removeAllUser()
    {
        if (user_.size() == 0) {
            return;
        }

        hasChanged_ = true;
        user_.removeAllElements();
        setChanged();
        notifyObservers();
    }

    /**
     * Remoes a grid user at a given location in a data structure
     * @param index   location in a data structure
     * @pre index >= 0
     * @post $none
     */
    public void removeUser(int index)
    {
        try
        {
            user_.removeElementAt(index);

            // if the vector is empty, then reset the total user
            if (user_.size() == 0) {
                userCounter_ = 1;
            }
        }
        catch (Exception e) {
            System.out.println("Error - the grid user is not found.");
        }

        hasChanged_ = true;
        setChanged();
        notifyObservers();
    }

    /**
     * Shows the dialog properties of a given grid user
     * @param index     location in a data structure
     * @pre index >= 0
     * @post $none
     */
    public void showProperty(int index)
    {
        UserProperty obj = (UserProperty) user_.elementAt(index);
        String oldName = obj.getUserName();

        obj.showDialog();

        // if the user name has changed then notify other observers
        if (obj.getUserName().equals(oldName) == false)
        {
            setChanged();
            notifyObservers();
        }

    }

    /**
     * Shows the dialog properties of a given grid user
     * @param value     a grid user name
     * @pre value != null
     * @post $none
     */
    public void showProperty(String value)
    {
        UserProperty obj = (UserProperty) user_.elementAt(0);

        int size = user_.size();
        for (int i = 0; i < size; i++)
        {
            obj = (UserProperty) user_.elementAt(i);
            if (obj.getUserName().equals(value) == true) {
                break;
            }
        }

        String oldName = obj.getUserName();
        obj.showDialog();

        // if the user name has changed then notify other observers
        if (obj.getUserName().equals(oldName) == false)
        {
            setChanged();
            notifyObservers();
        }
    }

    /**
     * Generates a Java code regarding to grid users
     * @param indent    indentation
     * @return a Java code in String object
     * @pre indent != null
     * @post $result != null
     */
    public String generateCode(String indent)
    {
        StringBuffer code = new StringBuffer(1000);
        code.append("\n\n");
        code.append(indent);
        code.append("// The starting point of creating Grid user entities\n");
        code.append(indent);
        code.append("GridletList glList;      // A list of Gridlets or jobs\n");
        code.append(indent);
        code.append("Experiment expt;\n");
        code.append(indent);
        code.append("UserEntity userEntity;\n");
        code.append(indent);
        code.append("int count = 0;\n");

        int size = user_.size();
        for (int i = 0; i < size; i++) 
        {
            code.append( ( (UserProperty) user_.elementAt(i) 
                       ).generateCode(indent) );
        }

        return code.toString();
    }

    /**
     * Generates a XML code regarding to grid users
     * @param indent    indentation
     * @return a XML code in String object
     * @pre indent != null
     * @post $result != null
     */
    public String saveFile(String indent)
    {
        hasChanged_ = false;
        String tab = indent + "    ";  // any previous empty spaces + 4 more
        StringBuffer xml = new StringBuffer(1000);
        xml.append("\n\n");
        xml.append(indent);
        xml.append("<gridUser>\n");

        final int size = user_.size();
        xml.append(tab);
        xml.append("<totalUser> ");
        xml.append(size);
        xml.append(" </totalUser>");

        for (int i = 0; i < size; i++) {
            xml.append( ( (UserProperty) user_.elementAt(i) ).saveFile(tab) );
        }

        xml.append("\n");
        xml.append(indent);
        xml.append("</gridUser>\n");
        
        return xml.toString();
    }

    /**
     * Checks whether the properties of grid user has changed or not
     * @return <tt>true</tt> if it has changed, <tt>false</tt> otherwise
     * @pre $none
     * @post $none
     */
    public boolean hasChanged() {
        return hasChanged_;
    }

    /**
     * Deletes all existing values and starts fresh 
     * @pre $none
     * @post $none
     */
    public void newValue()
    {
        // Note: the order of removeAllUser() and hasChanged_ must not be
        // changed. This is because inside removeAllUser(), it modifies
        // hasChanged_ value
        this.removeAllUser();
        hasChanged_ = false;
        userCounter_ = 1;
    }

    /**
     * Loads a XML code regarding to grid users
     * @param nodeList  a NodeList object
     * @throws Exception if error occurs on reading or parsing XML code 
     * @pre nodeList != null
     * @post $none
     */
    public void loadXml(final NodeList nodeList) throws Exception
    {
        final int size = nodeList.getLength();
        Node node;

        for (int j = 0; j < size; j++)
        {
            node = nodeList.item(j);
            if (node.getNodeType() == Node.ELEMENT_NODE) {
                loadXmlUser(node, size);
            }
        }

        userCounter_++;
        setChanged();
        notifyObservers();
    }


    ////////////////////////// PRIVATE METHODS /////////////////////////////


    /**
     * Loads a XML code regarding to grid users sizes
     * @param node  a Node object
     * @param size  grid user sizes
     * @throws Exception if error occurs on reading or parsing XML code 
     * @pre node != null
     * @pre size >= 0
     * @post $none
     */
    private void loadXmlUser(final Node node, final int size) throws Exception
    {
        try
        {
            String nodeName = node.getNodeName();

            if (nodeName.equals("totalUser") == true)
            {
                String value = node.getFirstChild().getNodeValue();

                // must trim the String otherwise get an exception
                int num = Integer.parseInt(value.trim());
                int totalUser = (size/2) - 1;

                // this causing other unknown exception, so becareful
                if (num < 1 || totalUser != num) {
                    throw new NumberFormatException("<totalUser> error");
                }
            }
            else if (nodeName.equals("user") == true) {
                loadXmlUserProperty( node.getChildNodes() );
            }
        }
        catch (NumberFormatException e)
        {
            System.out.println("Exception = " + e.getMessage());
        }
        catch (Exception obj)
        {
            System.out.println("Exception = " + obj.getMessage());
        }

    }

    /**
     * Loads a XML code regarding to grid users properties
     * @param nodeList  a NodeList object
     * @throws Exception if error occurs on reading or parsing XML code 
     * @pre nodeList != null
     * @post $none
     */
    private void loadXmlUserProperty(final NodeList nodeList) throws Exception
    {
        Node node;
        String name, value;

        for (int i = 0; i < nodeList.getLength(); i++)
        {
            node = nodeList.item(i);
            name = node.getNodeName();

            if (node.getNodeType() == Node.ELEMENT_NODE &&
                name.equals("id") == true)
            {
                value = node.getFirstChild().getNodeValue();

                // must trim the String otherwise get an exception
                int id = Integer.parseInt(value.trim());
                userCounter_ = id;

                UserProperty obj = new UserProperty(id, null, false, null);
                obj.loadXml(nodeList);
                user_.add(obj);
                break;
            }
        }
    }

} // end class

