/*
 * Title:        Visual Modeler for GridSim Toolkit
 * Description:  This Visual Modeler enables the user to quickly create
 *               experiments on different Grid testbeds and generate the
 *               default Grid Broker source codes (in Java).
 *
 * $Id: ResourceProperty.java,v 1.10 2003/06/29 07:31:52 anthony Exp $
 */

package visualmodeler;

// to handle loading XML. These packages exist since java 1.4
import org.w3c.dom.*;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.border.*;
import java.awt.*;
import java.awt.event.*;
import java.util.*;


/**
 * ResourceProperty stores the values needed for each grid resource
 *
 * @author       Anthony Sulistio and Chee Shin Yeo
 * @version      1.1
 * @invariant $none
 */
public class ResourceProperty extends JDialog
                implements ActionListener, ChangeListener, WindowListener,
                           ListSelectionListener
{
    private JSlider peakSlider_, offSlider_, holidaySlider_;
    private int peakLoad_, offLoad_, holidayLoad_;

    private double baudRate_;
    private int id_;        // resource id - this cannot be changed & unique
    private String name_;   // resource name

    ////////// GUI part
    // labels to display the values
    private JLabel peakLabel_, offLabel_, holidayLabel_;
    private JTextField resText_;    // display the resource name
    private JTextField baudText_;   // display baud rate value
    private JButton ok_, cancel_;

    // a flag to determine whether the GUI components has created or not
    private boolean guiHasCreated_;

    ///////// resource characteristics
    private int comboPolicy_;
    private final String[] comboValue_ = { "Time-shared", "Space-shared" };

    private String arch_;        // architecture name
    private String os_;          // operating system
    private double timeZone_;    // time zone based on GMT
    private double price_;       // cost per sec

    // to display machines - this should be in a separate class?
    private Vector machine_;
    private Vector machineName_;

    ////////// GUI for resource characteristics
    private JComboBox combo_;    // for allocation policy
    private JPanel charPanel_;   // Another panel for resource characteristics
    private JTextField archText_, osText_, timeZoneText_, priceText_;
    private JList machineList_;
    private JPanel machinePanel_;   // to display the CPUs of a machine


    /** 
     * Allocates a new ResourceProperty object
     * @param id    resource id
     * @param name  resource name
     * @param random    <tt>true</tt> if assigning grid resource properties
     *                  randomly, <tt>false</tt> otherwise
     * @param r     a Random object
     * @pre id >= 0
     * @pre name != null
     * @pre r != null
     * @post $none
     */
    public ResourceProperty(int id, String name, boolean random, Random r)
    {
        super.setModal(true);  // user can't open or use another window
        super.setSize(500, 500);
        super.setLocation(150, 100);
        super.setTitle("Resource Property");
        this.addWindowListener(this);

        machine_ = new Vector();
        machineName_ = new Vector();
        machinePanel_ = new JPanel();

        if (random == true) {
            randomValue(r);
        }
        else {
            defaultValue();
        }

        id_ = id;
        name_ = name;
        guiHasCreated_ = false;


        // initializes all GUI components
        archText_ = null;
        baudText_ = null;
        cancel_ = null;
        charPanel_ = null;
        combo_ = null;
        holidayLabel_ = null;
        holidaySlider_ = null;
        machineList_ = null;
        offLabel_ = null;
        offSlider_ = null;
        ok_ = null;
        osText_ = null;
        peakLabel_ = null;
        peakSlider_ = null;
        priceText_ = null;
        resText_ = null;
        timeZoneText_ = null;
    }

    /** 
     * Gets the resource name
     * @return resource name
     * @pre $none
     * @post $result != null
     */
    public String getResourceName() {
        return name_;
    }

    /** 
     * Gets resource id
     * @return resource id
     * @pre $none
     * @post $result >= 0
     */
    public int getId() {
        return id_;
    }

    /** 
     * A method that listens to user's call
     * @param evt   an ActionEvent object
     * @pre evt != null
     * @post $none
     */
    public void actionPerformed(ActionEvent evt) 
    {
        if (evt.getActionCommand().equalsIgnoreCase("ok") == true)
        {
            if (saveValue() == false) {
                return;
            }
        }
            // for cancel button
        else {
            resetValue();
        }

        removeListeners();
        dispose();
    }

    /** 
     * A method that refreshes the slider bars when they received a change
     * @param evt     a ChangeEvent object
     * @pre evt != null
     * @post $none
     */
    public void stateChanged(ChangeEvent evt)
    {
        // have to divide by 10 since the slider value [1, 10] in integer
        JSlider slider = (JSlider) evt.getSource();
        int num = slider.getValue();
        double value = num / 10.0;

        if (slider == peakSlider_) {
            peakLabel_.setText("Peak Load:   " + value);
        }
        else if (slider == offSlider_) {
            offLabel_.setText("Off-peak Load:   " + value);
        }
        else {
            holidayLabel_.setText("Holiday Load:   " + value);
        }
    }

    /** 
     * A method that shows a dialog for a particular list selection
     * @param evt   a ListSelectionEvent object
     * @pre evt != null
     * @post $none
     */
    public void valueChanged(ListSelectionEvent evt)
    {
        if ( evt.getValueIsAdjusting() ) {
            return;
        }

        int i = machineList_.getSelectedIndex();
        ((ResourceMachine) machine_.elementAt(i)).showDialog(machinePanel_);
    }

    /** 
     * Shows the grid resource dialog
     * @pre $none
     * @post $none
     */
    public void showDialog()
    {
        if (guiHasCreated_ == false)
        {
            guiHasCreated_ = true;
            initComponents();
        }

        ok_.addActionListener(this);
        cancel_.addActionListener(this);
        machineList_.addListSelectionListener(this);
        peakSlider_.addChangeListener(this);
        offSlider_.addChangeListener(this);
        holidaySlider_.addChangeListener(this);

        this.show();
    }

    /** 
     * Generates a Java code for grid resource
     * @param indent    indentation
     * @return a Java code in String object
     * @pre indent != null
     * @post $result != null
     */
    public String generateCode(String indent)
    {
        String archName = "\"" + arch_ + "\"";
        String osName = "\"" + os_ + "\"";
        String resName = "\"" + name_ + "\"";

        StringBuffer code = new StringBuffer(1000);
        code.append("\n");
        code.append(indent);
        code.append("///////// Create ");
        code.append(name_);
        
        code.append("\n");
        code.append(indent);
        code.append("mList = new MachineList();\t// A list of Machines\n");

        int size = machine_.size();
        for (int i = 0; i < size; i++) 
        {
            code.append( ( (ResourceMachine) machine_.elementAt(i)
                        ).generateCode(indent) );
        }

        code.append("\n");
        code.append(indent);
        code.append("// Creates a new ResourceCharacteristics entity\n");
        code.append(indent);
        code.append("resConfig = new ResourceCharacteristics(");
        code.append(archName);
        code.append(", ");
        code.append(osName);
        code.append(",\n");
        code.append(indent);
        code.append("    mList, ");
        code.append(comboPolicy_);
        code.append(", ");
        code.append(timeZone_);
        code.append(", ");
        code.append(price_);
        code.append(");\n\n");

        code.append(indent);
        code.append("// Creates a new GridResource entity\n");
        code.append(indent);
        code.append("gridRes = new GridResource(");
        code.append(resName);
        code.append(", ");
        code.append(baudRate_);
        code.append(", seed*(" + id_ + "+1)+1,\n");
        code.append(indent);
        code.append("    resConfig, ");
        code.append(peakLoad_);
        code.append(", ");
        code.append(offLoad_);
        code.append(", ");
        code.append(holidayLoad_);
        code.append(", Weekends, Holidays);\n");

        return code.toString();
    }

    /** 
     * Saves grid resource properties into XML format
     * @param spaces    indentation
     * @return XML code 
     * @pre spaces != null
     * @post $result != null
     */
    public String saveFile(String spaces)
    {
        final double DIV = 10.0;
        String indent = spaces + "    ";
        String tab = indent + "    ";
        String doubleTab = tab + "    ";

        // write the resource header
        StringBuffer xml = new StringBuffer(1000);
        xml.append("\n\n");
        xml.append(spaces);
        xml.append("<resource>\n");

        // write the resource id
        xml.append(indent);
        xml.append("<id> ");
        xml.append(id_);
        xml.append(" </id>\n");

        // write the resource name
        xml.append(indent);
        xml.append("<name> ");
        xml.append(name_);
        xml.append(" </name>\n");

        // write the baud rate
        xml.append(indent);
        xml.append("<baudRate> ");
        xml.append(baudRate_);
        xml.append(" </baudRate>\n");

        // write the load values
        xml.append(indent);
        xml.append("<peakLoad> ");
        xml.append(peakLoad_ / DIV);
        xml.append(" </peakLoad>\n");
        
        xml.append(indent);
        xml.append("<offPeakLoad> ");
        xml.append(offLoad_ / DIV);
        xml.append(" </offPeakLoad>\n");
        
        xml.append(indent);
        xml.append("<holidayLoad> ");
        xml.append(holidayLoad_ / DIV);
        xml.append(" </holidayLoad>\n");

        // write the resource characteristics
        xml.append(indent);
        xml.append("<resourceCharacteristics>\n");

        // write the architecture name
        xml.append(tab);
        xml.append("<architecture> ");
        xml.append(arch_);
        xml.append(" </architecture>\n");

        // write the os name
        xml.append(tab);
        xml.append("<operatingSystem> ");
        xml.append(os_);
        xml.append(" </operatingSystem>\n");

        // write the time zone
        xml.append(tab);
        xml.append("<timeZone> ");
        xml.append(timeZone_);
        xml.append(" </timeZone>\n");

        // write the price
        xml.append(tab);
        xml.append("<gridDollarPerApplication> ");
        xml.append(price_);
        xml.append(" </gridDollarPerApplication>\n");

        // write the allocation policy
        xml.append(tab);
        xml.append("<allocationPolicy> ");
        xml.append(comboValue_[comboPolicy_]);
        xml.append(" </allocationPolicy>\n\n");

        // write the machines
        xml.append(tab);
        xml.append("<machineList>\n");

        // write the total machine
        final int size = machine_.size();
        xml.append(doubleTab);
        xml.append("<totalMachine> ");
        xml.append(size);
        xml.append(" </totalMachine>\n");

        // loop to write the machine names
        for (int i = 0; i < size; i++) 
        {
            xml.append( ( (ResourceMachine)machine_.elementAt(i)
                      ).saveFile(doubleTab) );
        }
        
        // don't forget the closing tag
        xml.append("\n");
        xml.append(tab);
        xml.append("</machineList>\n");
        xml.append(indent);
        xml.append("</resourceCharacteristics>\n");
        xml.append(spaces);
        xml.append("</resource>\n");

        return xml.toString();
    }

    /** 
     * Loads a section of grid resource from XML file
     * @param nodeList  a NodeList object
     * @throws Exception    if error occurs in parsing and retrieving XML file 
     * @pre nodeList != null
     * @post $none
     */
    public void loadXml(final NodeList nodeList) throws Exception
    {
        //hasLoadXml_ = true;
        Node node;
        String name, value;
        final int LOAD = 10;

        int length = nodeList.getLength();
        for (int i = 0; i < length; i++)
        {
            node = nodeList.item(i);

            // only element nodes that need to be take care,
            // the rests are ignored
            if (node.getNodeType() != Node.ELEMENT_NODE) {
                continue;
            }

            name = node.getNodeName();
            if (name.equals("id") == true)
            {
                value = node.getFirstChild().getNodeValue();

                // must trim the String otherwise get an exception
                int id = Integer.parseInt(value.trim());
                id_ = id;
            }

            else if (name.equals("name") == true)
            {
                value = node.getFirstChild().getNodeValue();
                name_ = value.trim();
            }

            else if (name.equals("baudRate") == true)
            {
                value = node.getFirstChild().getNodeValue();

                // must trim the String otherwise get an exception
                double num = Double.parseDouble(value.trim());
                baudRate_ = num;
            }

            else if (name.equals("peakLoad") == true)
            {
                value = node.getFirstChild().getNodeValue();

                // must trim the String otherwise get an exception
                double num = Double.parseDouble(value.trim());
                peakLoad_ = (int) (num * LOAD);
            }

            else if (name.equals("offPeakLoad") == true)
            {
                value = node.getFirstChild().getNodeValue();

                // must trim the String otherwise get an exception
                double num = Double.parseDouble(value.trim());
                offLoad_ = (int) (num * LOAD);
            }

            else if (name.equals("holidayLoad") == true)
            {
                value = node.getFirstChild().getNodeValue();

                // must trim the String otherwise get an exception
                double num = Double.parseDouble(value.trim());
                holidayLoad_ = (int) (num * LOAD);
            }

            else if (name.equals("resourceCharacteristics") == true)
            {
                // make another method to handle resource characteristics
                loadXmlResourceCharacteristics( node.getChildNodes() );
            }
        }
    }

    /** 
     * An event that occurs when the window dialog is closing
     * @param evt   a WindowEvent object
     * @pre evt != null
     * @post $none
     */
    public void windowClosing(WindowEvent evt)
    {
        resetValue();
        removeListeners();
    }
    
    /** 
     * An event that occurs when the window dialog is closed
     * @param evt   a WindowEvent object
     * @pre evt != null
     * @post $none
     */ 
    public void windowClosed(WindowEvent evt) {
        //empty - not needed but have to due to WindowListener interface
    }

    /** 
     * An event that occurs when the window dialog is opened
     * @param evt   a WindowEvent object
     * @pre evt != null
     * @post $none
     */
    public void windowOpened(WindowEvent evt) {
        //empty - not needed but have to due to WindowListener interface
    }

    /** 
     * An event that occurs when the window dialog is iconified
     * @param evt   a WindowEvent object
     * @pre evt != null
     * @post $none
     */
    public void windowIconified(WindowEvent evt) {
        //empty - not needed but have to due to WindowListener interface
    }

    /** 
     * An event that occurs when the window dialog is deiconified
     * @param evt   a WindowEvent object
     * @pre evt != null
     * @post $none
     */
    public void windowDeiconified(WindowEvent evt) {
        //empty - not needed but have to due to WindowListener interface
    }

    /** 
     * An event that occurs when the window dialog is activated
     * @param evt   a WindowEvent object
     * @pre evt != null
     * @post $none
     */
    public void windowActivated(WindowEvent evt) {
        //empty - not needed but have to due to WindowListener interface
    }

    /** 
     * An event that occurs when the window dialog is deactivated
     * @param evt   a WindowEvent object
     * @pre evt != null
     * @post $none
     */
    public void windowDeactivated(WindowEvent evt) {
        //empty - not needed but have to due to WindowListener interface
    }



    //////////////////////////// PRIVATE METHODS ///////////////////////////

    /** 
     * Creates a new grid resource with default values for its properties
     * @pre $none
     * @post $none
     */
    private void defaultValue()
    {
        peakLoad_ = 0;
        offLoad_ = 0;
        holidayLoad_ = 0;
        baudRate_ = 100.0;
        price_ = 100.0;
        timeZone_ = 0;
        comboPolicy_ = 0;

        arch_ = "Intel";
        os_ = "Red Hat Linux";

        // create default machine
        machine_.add(new ResourceMachine(0, false, null));
        machineName_.add("Machine 0");

        machine_.add(new ResourceMachine(1, false, null));
        machineName_.add("Machine 1");
    }

    /** 
     * Creates grid resource properties with random values
     * @param r   a Random object
     * @pre r != null
     * @post $none
     */
    private void randomValue(Random r)
    {
        int RANGE = 11;

        // must multiply by 10 since the value of nextDouble() is [0, 0.99999]
        baudRate_ = r.nextInt(1001);

        // set into the range [0, 11), i.e. 0-10 not 11
        peakLoad_ = r.nextInt(RANGE);
        offLoad_ = r.nextInt(RANGE);
        holidayLoad_ = r.nextInt(RANGE);

        // there are only 2 options, but the index starts at 0
        comboPolicy_ = r.nextInt(2);

        // TODO: GMT +13 and -12, but below is always postive value
        timeZone_ = r.nextInt(RANGE);

        price_ = r.nextDouble() * 100;
        int index = 0;

        int MAX = 4;
        String[][] list = { 
                {"Compaq AlphaServer",  "OSF1"},    {"Sun Ultra", "Solaris"},
                {"Intel Pentium", "Linux"},         {"SGI Origin", "Irix"}
        };

        index = r.nextInt(MAX);
        arch_ = list[index][0];
        os_   = list[index][1];
        
        // create machines
        index = r.nextInt(RANGE);

        // machine must have at least two CPU or gridlet
        if (index < 1) {
            index = r.nextInt(RANGE);
        }
        
        for (int i = 0; i < index; i++)
        {
            machine_.add(new ResourceMachine(i, true, r));
            machineName_.add("Machine " + i);
        }

    }

    /** 
     * Initializes all the dialog components
     * @pre $none
     * @post $none
     */
    private void initComponents()
    {
        GridBagLayout gridbag = new GridBagLayout();
        GridBagConstraints c = new GridBagConstraints();
        JPanel panel = new JPanel(gridbag);

        c.insets = new Insets(5, 3, 5, 3);
        Font font = new Font(null, Font.PLAIN, 14);

        int index = 0;
        c.weighty = 0.5;
        c.fill = GridBagConstraints.HORIZONTAL;
        c.gridy = index++;

        //////////////////
        // Display the user name
        JLabel label = new JLabel("Resource Name:");
        label.setFont(font);
        panel.add(label, c);

        c.gridwidth = GridBagConstraints.REMAINDER;
        resText_ = new JTextField(name_, 20);
        panel.add(resText_, c);

        ////////////////////
        // Display the baud rate
        c.gridwidth = 1;
        c.gridy = index++;
        label = new JLabel("Baud Rate:");
        label.setFont(font);
        panel.add(label, c);

        baudText_ = new JTextField("" + baudRate_, 5);
        panel.add(baudText_, c);

        ///////////////////////
        // Create a label to display the slider default value
        c.gridy = index++;
        c.weighty = 0.5;
        c.fill = GridBagConstraints.HORIZONTAL;
        double num = peakLoad_ / 10.0;
        peakLabel_ = new JLabel("Peak Load:   " + num);
        peakLabel_.setFont(font);
        panel.add(peakLabel_, c);

        // Create the slider with min value, max value and init value
        peakSlider_ = new JSlider(0, 10, peakLoad_);
        createSlider(panel, c, peakSlider_);

        // Create a label to display the slider default value
        c.gridy = index++;
        num = offLoad_ / 10.0;
        offLabel_ = new JLabel("Off-peak Load:   " + num);
        offLabel_.setFont(font);
        panel.add(offLabel_, c);

        // Create the slider with min value, max value and init value
        offSlider_ = new JSlider(0, 10, offLoad_);
        createSlider(panel, c, offSlider_);

        // Create a label to display the slider default value
        c.gridy = index++;
        num = holidayLoad_ / 10.0;
        holidayLabel_ = new JLabel("Holiday Load:   " + num);
        holidayLabel_.setFont(font);
        panel.add(holidayLabel_, c);

        // Create the slider with min value, max value and init value
        holidaySlider_ = new JSlider(0, 10, holidayLoad_);
        createSlider(panel, c, holidaySlider_);

        //////////////////////////////////
        // Add another panel for Resource Characteristics
        charPanel_ = new JPanel(gridbag);
        c.weighty = 1.0;   //request any extra vertical space
        c.gridy = index++;
        c.insets = new Insets(25, 0, 10, 0);  //top & bottom padding
        c.gridwidth = GridBagConstraints.REMAINDER;
        c.fill = GridBagConstraints.BOTH;
        panel.add(charPanel_, c);

        // Create another panel that contains an experiment value
        initCharacteristic();

        // Create an OK button
        index += 2;
        c.gridy = index++;
        c.gridx = 1;
        c.gridwidth = 1;
        c.fill = GridBagConstraints.NONE;
        ok_ = new JButton("OK");
        panel.add(ok_, c);

        // Create a Cancel button
        c.gridwidth = GridBagConstraints.REMAINDER;
        cancel_ = new JButton("Cancel");
        panel.add(cancel_, c);

        // Add panel to a scroll pane
        int vert = ScrollPaneConstants.VERTICAL_SCROLLBAR_AS_NEEDED;
        int horiz = ScrollPaneConstants.HORIZONTAL_SCROLLBAR_AS_NEEDED;
        JScrollPane scroll = new JScrollPane(panel, vert, horiz);

        getContentPane().add(scroll);
    }

    /** 
     * Initializes all components regarding to resouce characteristic 
     * properties
     * @pre $none
     * @post $none
     */
    private void initCharacteristic()
    {
        GridBagConstraints c = new GridBagConstraints();
        c.insets = new Insets(10, 5, 5, 15);  // top - left - bottom - right
        Font font = new Font(null, Font.PLAIN, 12);

        TitledBorder title = BorderFactory.createTitledBorder("Characteristic");
        title.setTitleFont(font.deriveFont(Font.BOLD, 18));
        charPanel_.setBorder(title);

        // init all the values
        int index = 0;
        c.weighty = 0.5;
        c.fill = GridBagConstraints.HORIZONTAL;

        // Display the architecture name
        c.gridy = index++;
        JLabel label = new JLabel("Architecture:");
        label.setFont(font);
        charPanel_.add(label, c);

        c.gridwidth = GridBagConstraints.REMAINDER;
        archText_ = new JTextField(arch_, 20);
        charPanel_.add(archText_, c);

        ////////////////////
        // Display the OS
        c.gridwidth = 1;
        c.gridy = index++;
        label = new JLabel("Operating System:");
        label.setFont(font);
        charPanel_.add(label, c);

        //c.gridwidth = GridBagConstraints.REMAINDER;
        osText_ = new JTextField(os_, 20);
        charPanel_.add(osText_, c);

        ////////////////////
        // Display the time zone
        c.gridwidth = 1;
        c.gridy = index++;
        label = new JLabel("Time Zone (GMT) :");
        label.setFont(font);
        charPanel_.add(label, c);

        timeZoneText_ = new JTextField("" + timeZone_, 5);
        charPanel_.add(timeZoneText_, c);

        ////////////////////
        // Display the cost per sec
        c.gridwidth = 1;
        c.gridy = index++;
        label = new JLabel("Grid $ / application operation:  ");
        label.setFont(font);
        charPanel_.add(label, c);

        priceText_ = new JTextField("" + price_, 5);
        charPanel_.add(priceText_, c);

        /////////////////
        c.gridy = index++;
        label = new JLabel("Allocation policy:");
        label.setFont(font);
        charPanel_.add(label, c);

        // Create a combo box
        combo_ = new JComboBox(comboValue_);
        combo_.setSelectedIndex(comboPolicy_);  // default value is Random
        c.gridwidth = GridBagConstraints.REMAINDER;
        charPanel_.add(combo_, c);

        /**************/
        label = new JLabel("List of Machines");
        label.setFont(font.deriveFont(Font.BOLD, 16));
        c.weighty = 1.0;   //request any extra vertical space
        c.insets = new Insets(25,0,0,0);  //top padding
        c.gridy = index++;
        c.gridx = 1;
        c.fill = GridBagConstraints.BOTH;
        c.gridwidth = GridBagConstraints.RELATIVE;
        charPanel_.add(label, c);

        // reset the values
        c.fill = GridBagConstraints.HORIZONTAL;
        c.gridwidth = GridBagConstraints.REMAINDER;
        c.weighty = 0.0;
        c.gridx = 0;

        ///////////////////////////////
        // Create a list of machines
        machineList_ = new JList(machineName_);
        machineList_.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
        machineList_.setSelectedIndex(0);

        int vert = ScrollPaneConstants.VERTICAL_SCROLLBAR_AS_NEEDED;
        int horiz = ScrollPaneConstants.HORIZONTAL_SCROLLBAR_AS_NEEDED;
        JScrollPane listScrollPane = new JScrollPane(machineList_, vert,horiz);

        // initially display the first machine
        ((ResourceMachine) machine_.elementAt(0)).showDialog(machinePanel_);

        JSplitPane splitPane = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT,
                                   listScrollPane, machinePanel_);

        splitPane.setOneTouchExpandable(true);
        splitPane.setDividerLocation(150);

        c.gridy = index++;
        charPanel_.add(splitPane, c);
    }

    /** 
     * Creates a slider bar 
     * @param panel   a JPanel object
     * @param c       a GridBagConstraints object
     * @param slider  a JSlider object
     * @pre $none
     * @post $none
     */
    private void createSlider(JPanel panel, GridBagConstraints c,
                        JSlider slider)
    {
        slider.setMajorTickSpacing(1);   // increment every 1 point
        slider.setPaintLabels(true);     // display the label for MajorTick
        slider.setPaintTicks(true);

        //Create the label table
        Hashtable labelTable = new Hashtable();
        float i = 0;
        for (int counter = 0; counter < 7; counter++)
        {
            labelTable.put( new Integer(counter), new JLabel("" + i) );
            i += 0.1;
        }

        // have to be hardcoded since after 6, the value will be wrong 
        // in windows
        labelTable.put( new Integer(7), new JLabel("0.7") );
        labelTable.put( new Integer(8), new JLabel("0.8") );
        labelTable.put( new Integer(9), new JLabel("0.9") );
        labelTable.put( new Integer(10), new JLabel("1.0") );

        // add the hashtable into the slider
        slider.setLabelTable( labelTable );

        c.gridwidth = GridBagConstraints.REMAINDER;
        c.weightx = 3;
        c.gridx = 2;
        panel.add(slider, c);

        // reset the values
        c.gridwidth = 1;
        c.weightx = 0;
        c.gridx = 0;
    }

    /** 
     * Removes all GUI listeners
     * @pre $none
     * @post $none
     */
    private void removeListeners()
    {
        // remove all the listeners to prevent memory leakages
        machineList_.removeListSelectionListener(this);
        ok_.removeActionListener(this);
        cancel_.removeActionListener(this);
        peakSlider_.removeChangeListener(this);
        offSlider_.removeChangeListener(this);
        holidaySlider_.removeChangeListener(this);
    }

    /** 
     * Saves the current grid resource properties
     * @return <tt>true</tt> if the values have been saved successfully, 
     *         <tt> false</tt> otherwise
     * @pre $none
     * @post $none
     */
    private boolean saveValue()
    {
        name_ = resText_.getText();
        peakLoad_ = peakSlider_.getValue();
        offLoad_ = offSlider_.getValue();
        holidayLoad_ = holidaySlider_.getValue();
        comboPolicy_ = combo_.getSelectedIndex();

        String errorMsg = "";  // a flag to denote where the error comes from

        try
        {
            errorMsg = "baud rate";
            baudRate_ = checkValue(baudRate_, baudText_, false);

            errorMsg = "time zone";
            timeZone_ = checkValue(timeZone_, timeZoneText_, true);

            errorMsg = "cost per sec";
            price_ = checkValue(price_, priceText_, false);
        }
        catch (NumberFormatException e)
        {
            JOptionPane.showMessageDialog(this,
                    "Invalid value for " + errorMsg + ".",
                    errorMsg + " value error", JOptionPane.ERROR_MESSAGE);

            return false;
        }

        catch (Exception e)
        {
            JOptionPane.showMessageDialog(this, e.getMessage(),
                    errorMsg + " value error", JOptionPane.ERROR_MESSAGE);

            return false;
        }

        int size = machine_.size();
        for (int i = 0; i < size; i++) {
            ( (ResourceMachine) machine_.elementAt(i) ).saveValue();
        }

        return true;
    }

    /** 
     * Checks whether the given value is correct or not
     * @param value     the number to be checked
     * @param text      a JTextField object
     * @param timeZone  <tt>true</tt> if it is a time zone, <tt>false</tt> 
     *                  otherwise
     * @return the correct number
     * @throws Exception    if the value is not in a valid format
     * @pre text != null
     * @post $none
     */
    private double checkValue(double value, JTextField text, boolean timeZone)
                        throws Exception
    {
        value = new Double( text.getText() ).doubleValue();

        if (value < 0.0 && timeZone == false) {
            throw new Exception("Invalid for having negative value.");
        }

        if (timeZone == true)
        {
            // the limit for time zone is GMT + 13:00 and GMT - 12:00
            if (value > 13 || value < -12)
            {
                throw new Exception("Invalid value for GMT Time Zone.\n" +
                        "The valid limits are GMT + 13:00 and GMT - 12:00");
            }
        }

        return value;
    }

    /** 
     * Resets the current values back to the previous ones
     * @pre $none
     * @post $none
     */
    private void resetValue()
    {
        double num = peakLoad_ / 10.0;
        peakLabel_.setText("Peak Load:   " + num);
        peakSlider_.setValue(peakLoad_);

        num = offLoad_ / 10.0;
        offLabel_.setText("Off-peak Load:   " + num);
        offSlider_.setValue(offLoad_);

        num = holidayLoad_ / 10.0;
        holidayLabel_.setText("Holiday Load:   " + num);
        holidaySlider_.setValue(holidayLoad_);

        // reseting the text fields based on the order of apperances
        resText_.setText(name_);
        baudText_.setText("" + baudRate_);
        archText_.setText(arch_);
        osText_.setText(os_);
        timeZoneText_.setText("" + timeZone_);
        priceText_.setText("" + price_);

        combo_.setSelectedIndex(comboPolicy_);

        // reset the values for each gridlet
        int size = machine_.size();
        for (int i = 0; i < size; i++) {
            ( (ResourceMachine) machine_.elementAt(i) ).resetValue();
        }
    }

    /** 
     * Loads a section of grid resource characteristics from XML file
     * @param nodeList  a NodeList object
     * @pre nodeList != null
     * @post $none
     */
    private void loadXmlResourceCharacteristics(final NodeList nodeList)
    {
        Node node;
        String name, value;

        int length = nodeList.getLength();
        for (int i = 0; i < length; i++)
        {
            node = nodeList.item(i);

            // only element nodes that need to be take care,
            // the rests are ignored
            if (node.getNodeType() != Node.ELEMENT_NODE) {
                continue;
            }

            name = node.getNodeName();
            if (name.equals("architecture") == true)
            {
                value = node.getFirstChild().getNodeValue();
                arch_ = value.trim();
            }

            else if (name.equals("operatingSystem") == true)
            {
                value = node.getFirstChild().getNodeValue();
                os_ = value.trim();
            }

            else if (name.equals("timeZone") == true)
            {
                value = node.getFirstChild().getNodeValue();

                // must trim the String otherwise get an exception
                double num = Double.parseDouble(value.trim());
                timeZone_ = num;
            }

            else if (name.equals("gridDollarPerApplication") == true)
            {
                value = node.getFirstChild().getNodeValue();

                // must trim the String otherwise get an exception
                double num = Double.parseDouble(value.trim());
                price_ = num;
            }

            else if (name.equals("allocationPolicy") == true)
            {
                value = node.getFirstChild().getNodeValue();
                loadXmlPolicy( value.trim() );
            }

            else if (name.equals("machineList") == true)
            {
                machine_.clear();
                machineName_.clear();
                loadXmlMachine( node.getChildNodes() );
            }
        }

    }

    /** 
     * Loads a section of grid resource scheduling policy from XML file
     * @param value   scheduling policy
     * @pre value != null
     * @post $none
     */
    private void loadXmlPolicy(String value)
    {
        int i = 0;
        int length = comboValue_.length;
        
        for (i = 0; i < length; i++)
        {
            if (comboValue_[i].equals(value) == true) {
                break;
            }
        }

        if (i == length) {
            i = 0;
        }

        comboPolicy_ = i;
    }

    /** 
     * Loads a section of grid resource machine from XML file
     * @param nodeList a NodeList object
     * @pre nodeList != null
     * @post $none
     */
    private void loadXmlMachine(final NodeList nodeList) 
    {
        final int size = nodeList.getLength();
        Node node;

        for (int j = 0; j < size; j++) 
        {
            node = nodeList.item(j);
            if (node.getNodeType() == Node.ELEMENT_NODE) {
                loadXmlResourceMachine(node, size);
            }
        }
    }

    /** 
     * Loads a section of grid resource machine size from XML file
     * @param node  a Node object
     * @param size  a Machine size
     * @pre node != null
     * @pre size >= 0
     * @post $none
     */
    private void loadXmlResourceMachine(final Node node, final int size)
    {
        try
        {
            String nodeName = node.getNodeName();

            if (nodeName.equals("totalMachine") == true)
            {
                String value = node.getFirstChild().getNodeValue();

                // must trim the String otherwise get an exception
                int num = Integer.parseInt(value.trim());
                int total = (size/2) - 1;

                // this causing other unknown exception, so becareful
                if (num < 1 || total != num) {
                    throw new NumberFormatException("<totalMachine> error");
                }
            }

            else if (nodeName.equals("machine") == true) {
                loadXmlResourceMachineProperty( node.getChildNodes() );
            }
        }
        catch (NumberFormatException e)
        {
            System.out.println(e.getMessage());
            e.printStackTrace();
            //throw new Exception("Invalid value in <totalUser>");
        }
        catch (Exception obj)
        {
            System.out.println("obj = " + obj.getMessage());
            obj.printStackTrace();
        }

    }

    /** 
     * Loads a section of grid resource machine properties from XML file
     * @param nodeList  a NodeList object
     * @pre  nodeList != null
     * @post $none
     */
    private void loadXmlResourceMachineProperty(final NodeList nodeList)
    {
        Node node;
        String name, value;

        int length = nodeList.getLength();
        for (int i = 0; i < length; i++)
        {
            node = nodeList.item(i);
            name = node.getNodeName();

            if (node.getNodeType() == Node.ELEMENT_NODE &&
                name.equals("id") == true)
            {
                value = node.getFirstChild().getNodeValue();

                // must trim the String otherwise get an exception
                int id = Integer.parseInt(value.trim());

                ResourceMachine obj = new ResourceMachine(id, false, null);
                obj.loadXml(nodeList);
                machine_.add(obj);
                machineName_.add("Machine " + id);

                break;
            }

            // NOTE: ignoring <totalPE> for this time being
        }
    }


} // end class

