/*
 * Title:        Visual Modeler for GridSim Toolkit
 * Description:  This Visual Modeler enables the user to quickly create
 *               experiments on different Grid testbeds and generate the
 *               default Grid Broker source codes (in Java).
 *
 * $Id: ResourceModel.java,v 1.9 2003/06/19 07:31:00 anthony Exp $
 */

package visualmodeler;

// to handle loading XML. These packages exist since java 1.4
import org.w3c.dom.*;

import java.util.Observable;
import java.util.Vector;
import java.util.Random;


/**
 * ResourceModels stores and creates the resource objects
 *
 * @author       Anthony Sulistio and Chee Shin Yeo
 * @version      1.1
 * @invariant $none
 */
public class ResourceModel extends Observable
{
    private Vector res_;
    private int resCounter_;
    private final int SIZE = 1000;   // size of a StringBuffer

    // a flag to know whether the values have changed since the last save
    private boolean hasChanged_;


    /**
     * Allocates a new ResourceModel object
     * @pre $none
     * @post $none
     */
    public ResourceModel()
    {
        resCounter_ = 1;   // so the resource counter starts at 1 not 0
        res_ = new Vector();
        hasChanged_ = false;
    }

    /**
     * Gets the name of all resources
     * @return a list of resources in Vector object
     * @pre $none
     * @post $result != null
     */
    public Vector getAllResourceName()
    {
        Vector name = new Vector();
        String str;

        int size = res_.size();
        for (int i = 0; i < size; i++)
        {
            str = ( (ResourceProperty) res_.elementAt(i) ).getResourceName();
            name.add(str);
        }

        return name;
    }

    /**
     * Gets the resource counter
     * @return resource counter
     * @pre $none
     * @post $result > 0
     */
    public int getResourceCounter() {
        return resCounter_;
    }

    /**
     * Gets the total number of resources
     * @return total number of resources
     * @pre $none
     * @post $result >= 0
     */
    public int getTotalResource() {
        return res_.size();
    }

    /**
     * Creates resources based on a specific number
     * @param num   number of resources to be created
     * @param random  <tt>true</tt> if the resource properties need to be
     *                random, <tt>false</tt> otherwise
     * @pre num >= 0
     * @post $none
     */
    public void createResource(int num, boolean random)
    {
        // if creating zero resource, then ignore
        if (num == 0) {
            return;
        }

        String name = null;
        Random r = new Random();

        for (int i = 0; i < num; i++)
        {
            name = "Resource " + resCounter_;
            ResourceProperty obj = new ResourceProperty(resCounter_, name,
                                            random, r);
            res_.add(obj);
            resCounter_++;
        }

        hasChanged_ = true;
        setChanged();
        notifyObservers();
    }

    /**
     * Removes all the resources available
     * @pre $none
     * @post $none
     */
    public void removeAllResource()
    {
        if (res_.size() == 0) {
            return;
        }
        
        hasChanged_ = true;
        res_.removeAllElements();
        setChanged();
        notifyObservers();
    }

    /**
     * Removes a resource for a particular given location number
     * @param index  the location of a resource in the data structure
     * @pre index >= 0
     * @post $none
     */
    public void removeResource(int index)
    {
        try
        {
            res_.removeElementAt(index);

            // if the vector is empty, then reset the total user
            if (res_.size() == 0) {
                resCounter_ = 1;
            }            
        }
        catch (Exception e) {
            e.printStackTrace();
        }

        hasChanged_ = true;
        setChanged();
        notifyObservers();
    }

    /**
     * Shows the property of a particular grid resource
     * @param index  the location of a resource in the data structure
     * @pre index >= 0
     * @post $none
     */
    public void showProperty(int index)
    {
        // if Vector res_ is empty, then no point of showing its property
        if (res_.size() == 0) {
            return;
        }
            
        ResourceProperty obj = (ResourceProperty) res_.elementAt(index);
        String oldName = obj.getResourceName();
        obj.showDialog();

        // if the resource name has changed then notify other observers
        if (obj.getResourceName().equals(oldName) == false)
        {
            setChanged();
            notifyObservers();
        }
    }

    /**
     * Shows the property of a particular grid resource
     * @param value  a grid resource name
     * @pre value != null
     * @post $none
     */
    public void showProperty(String value)
    {
        // if Vector res_ is empty, then no point of showing its property
        if (res_.size() == 0) {
            return;
        }
        
        ResourceProperty obj = (ResourceProperty) res_.elementAt(0);

        // a loop that searching for a specific grid resource name
        for (int i = 0; i < res_.size(); i++)
        {
            obj = (ResourceProperty) res_.elementAt(i);
            if (obj.getResourceName().equals(value) == true) {
                break;
            }
        }

        String oldName = obj.getResourceName();
        obj.showDialog();

        // if the resource name has changed then notify other observers
        if (obj.getResourceName().equals(oldName) == false)
        {
            setChanged();
            notifyObservers();
        }
    }

    /**
     * Generates a specific Java code for the creation of grid resources
     * @param indent    indentation
     * @return a Java code in String object
     * @pre indent != null
     * @post $result != null
     */
    public String generateCode(String indent)
    {
        StringBuffer code = new StringBuffer(SIZE);
        code.append("\n");
        code.append(indent);
        code.append("// The starting point of creating Grid Resource entities"); 
        code.append("\n");
        code.append(indent);
        code.append("LinkedList Weekends = new LinkedList();\n");
        code.append(indent);
        code.append("Weekends.add(new Integer(Calendar.SATURDAY));\n");
        code.append(indent);
        code.append("Weekends.add(new Integer(Calendar.SUNDAY));\n");
        code.append(indent);
        code.append("LinkedList Holidays = new LinkedList();\n\n");
        code.append(indent);
        code.append("PEList peList;");
        code.append(indent);
        code.append("// A list of PEs (Processing Elements)\n");
        code.append(indent);
        code.append("MachineList mList;        // A list of Machine entities\n");
        code.append(indent);
        code.append("ResourceCharacteristics resConfig;\n");
        code.append(indent);
        code.append("GridResource gridRes;     // A GridResource entity\n\n");

        Vector vector = getAllResourceName();
        code.append(indent);
        code.append("String[] resourceNameList = { ");

        int length = vector.size();
        for (int n = 0; n < length; n++)
        {
            if (n % 4 == 0) 
            {
                code.append("\n    ");
                code.append(indent);
            }

            code.append("\"");
            code.append( vector.get(n) );
            code.append("\",  ");
        }


        code.append("\n");
        code.append(indent);
        code.append("};\n");  // closing bracket for the array

        // loop to generate each individual resource property code
        length = res_.size();
        for (int i = 0; i < length; i++)
        {
            code.append( ( (ResourceProperty)res_.elementAt(i) 
                    ).generateCode(indent) );
        }

        return code.toString();
    }

    /**
     * Saves grid resource properties into XML format
     * @param indent    indentation
     * @return a String object
     * @pre indent != null
     * @post $result != null
     */
    public String saveFile(String indent)
    {
        hasChanged_ = false;
        String tab = indent + "    ";  // any previous empty spaces + 4 more
        StringBuffer xml = new StringBuffer(SIZE);
        xml.append("\n\n");
        xml.append(indent);
        xml.append("<gridResource>\n");

        final int size = res_.size();
        xml.append(tab);
        xml.append("<totalResource> ");
        xml.append(size);
        xml.append(" </totalResource>");

        for (int i = 0; i < size; i++) 
        {
            xml.append( ( (ResourceProperty) res_.elementAt(i) 
                    ).saveFile(tab) );
        }

        xml.append("\n");
        xml.append(indent);
        xml.append("</gridResource>\n");
        
        return xml.toString();
    }

    /**
     * Checks whether one of grid resource properties has changed or not
     * @return <tt>true</tt> if it has changed, <tt>false</tt> otherwise
     * @pre $none
     * @post $none
     */
    public boolean hasChanged() {
        return hasChanged_;
    }

    /**
     * Removes all previous values and starts from fresh
     * @pre $none
     * @post $none
     */
    public void newValue()
    {
        // Note: the order of removeAllResource() and hasChanged_ must not be
        // changed. This is because inside removeAllResource(), it modifies
        // hasChanged_ value
        this.removeAllResource();
        hasChanged_ = false;
        resCounter_ = 1;
    }

    /**
     * Loads a specific XML code regarding to grid resource
     * @param nodeList  a NodeList object
     * @throws Exception if error occurs during loading XML
     * @pre nodeList != null
     * @post $none
     */
    public void loadXml(final NodeList nodeList) throws Exception
    {
        final int size = nodeList.getLength();
        Node node;

        for (int j = 0; j < size; j++)
        {
            node = nodeList.item(j);
            if (node.getNodeType() == Node.ELEMENT_NODE) {
                loadXmlResource(node, size);
            }
        }

        resCounter_++;
        setChanged();
        notifyObservers();
    }

    //////////////////////// PRIVATE METHODS /////////////////////////

    /**
     * Loads a specific section of XML regarding to grid resource 
     * @param node  a Node object
     * @param size  number of Node objects
     * @throws Exception    if error occurs during loading XML
     * @pre node != null
     * @pre size >= 0
     * @post $none
     */
    private void loadXmlResource(final Node node, final int size)
                        throws Exception
    {
        try
        {
            String nodeName = node.getNodeName();

            if (nodeName.equals("totalResource") == true)
            {
                String value = node.getFirstChild().getNodeValue();

                // must trim the String otherwise get an exception
                int num = Integer.parseInt(value.trim());
                int totalRes = (size/2) - 1;

                // this causing other unknown exception, so becareful
                if (num < 1 || totalRes != num) {
                    throw new NumberFormatException("<totalResource> error");
                }
                
            }

            else if (nodeName.equals("resource") == true) {
                loadXmlResourceProperty( node.getChildNodes() );
            }
            
        }
        catch (NumberFormatException e)
        {
            System.out.println(e.getMessage());
            e.printStackTrace();
            //throw new Exception("Invalid value in <totalUser>");
        }
        catch (Exception obj)
        {
            System.out.println("obj = " + obj.getMessage());
            obj.printStackTrace();
        }

    }

    /**
     * Loads a specific section of XML regarding to grid resource properties
     * @param nodeList  a NodeList object
     * @throws Exception    if error occurs during loading XML
     * @pre nodeList != null
     * @post $none
     */
    private void loadXmlResourceProperty(final NodeList nodeList)
                        throws Exception
    {
        Node node;
        String name, value;

        for (int i = 0; i < nodeList.getLength(); i++)
        {
            node = nodeList.item(i);
            name = node.getNodeName();

            if (node.getNodeType() == Node.ELEMENT_NODE &&
                name.equals("id") == true)
            {
                value = node.getFirstChild().getNodeValue();

                // must trim the String otherwise get an exception
                int id = Integer.parseInt(value.trim());
                resCounter_ = id;

                ResourceProperty obj =new ResourceProperty(id,null,false,null);
                obj.loadXml(nodeList);
                res_.add(obj);
                break;
            }
        }
    }

} // end class

