/*
 * Title:        Visual Modeler for GridSim Toolkit
 * Description:  This Visual Modeler enables the user to quickly create
 *               experiments on different Grid testbeds and generate the
 *               default Grid Broker source codes (in Java).
 *
 * $Id: ResourceMachine.java,v 1.8 2003/06/19 07:31:00 anthony Exp $
 */

package visualmodeler;

// to handle loading XML. These packages exist since java 1.4
import org.w3c.dom.*;

import javax.swing.table.AbstractTableModel;
import javax.swing.table.TableColumn;
import javax.swing.*;
import java.awt.*;
import java.util.Hashtable;
import java.util.Random;
import java.lang.Integer;


/**
 * ResourceMachine stores the values assossiated for each machine
 *
 * @author       Anthony Sulistio and Chee Shin Yeo
 * @version      1.1
 * @invariant $none
 */
public class ResourceMachine
{
    private int id_;        // machine id - unique and can't be changed
    private Hashtable pe_;
    private JPanel panel_;
    private boolean guiCreated_;
    private PETable peTable_;


    /**
     * Allocates a new ResourceMachine object
     * @param id    resource id
     * @param random    a flag to determine whether to randomize values or not
     * @param r     a Random object
     * @pre r != null
     * @post $none
     */
    public ResourceMachine(int id, boolean random, Random r)
    {
        id_ = id;
        pe_ = new Hashtable();
        panel_ = new JPanel();
        guiCreated_ = false;
        peTable_ = null;

        if (random == true) {
            randomValue(r);
        }
        else {
            pe_.put(new Integer(0), new Integer(51) );
        }
    }

    /**
     * Gets the machine id
     * @return the machine id
     * @pre $none
     * @post $result >= 0
     */
    public int getMachineId() {
        return id_;
    }

    /**
     * Gets the number of PEs (Processing Elements)
     * @return number of PEs
     * @pre $none
     * @post $result >= 0
     */
    public int getNumPE() {
        return pe_.size();
    }

    /**
     * Generates the Java code for the creation of PEs
     * @param indent    indentation
     * @return a piece of Java code
     * @pre indent != null
     * @post $result != null
     */
    public String generateCode(String indent)
    {
        StringBuffer code = new StringBuffer(500);
        code.append("\n");
        code.append(indent);
        code.append("peList = new PEList();      // A list of PEs\n");

        Integer obj = null;
        int size = pe_.size();
        
        for (int i = 0; i < size; i++)
        {
            obj = (Integer) pe_.get( new Integer(i) );

            code.append(indent);
            code.append("peList.add( new PE(");
            code.append(i);
            code.append(", ");
            code.append( obj.intValue() ); 
            code.append(") );\n");
        }

        code.append("\n");
        code.append(indent);
        code.append("// Adding a new Machine into the list\n");
        code.append(indent);
        code.append("mList.add( new Machine(");
        code.append(id_);
        code.append(", peList) );\n\n");

        return code.toString();
    }

    /**
     * Saves an information about Machine and PE into XML format
     * @param indent    indentation
     * @return an XML information
     * @pre indent != null
     * @post $result != null
     */
    public String saveFile(String indent)
    {
        String tab = indent + "    ";
        StringBuffer xml = new StringBuffer(500);
        xml.append("\n");
        xml.append(indent);
        xml.append("<machine>\n");

        // write the machine id
        xml.append(tab);
        xml.append("<id> ");
        xml.append(id_);
        xml.append(" </id>\n");

        // write the total of PEs
        int size = pe_.size();
        xml.append(tab);
        xml.append("<totalPE> ");
        xml.append(size);
        xml.append(" </totalPE>\n");

        Integer obj = null;
        for (int i = 0; i < size; i++)
        {
            obj = (Integer) pe_.get( new Integer(i) );

            // writing PE id
            xml.append(tab);
            xml.append("<peId> ");
            xml.append(i);
            xml.append(" </peId>\n");

            // writing MIPSRating
            xml.append(tab);
            xml.append("<MIPSRating> ");
            xml.append( obj.intValue() );
            xml.append(" </MIPSRating>\n");
        }

        // don't forget the closing tag
        xml.append("\n");
        xml.append(indent);
        xml.append("</machine>\n");
        
        return xml.toString();
    }

    /**
     * Shows a Resource Machine dialog
     * @param panel a JPanel object
     * @pre panel != null
     * @post $none
     */
    public void showDialog(JPanel panel)
    {
        // remove any previous components stored in this panel
        panel.removeAll();
        if (guiCreated_ == false)
        {
            guiCreated_ = true;
            initComponents();
        }

        panel.add(this.panel_);
        panel.revalidate();
        panel.repaint();
    }

    /**
     * Saves the value of PEs
     * @pre $none
     * @post $none
     */
    public void saveValue()
    {
        if (guiCreated_ == false) {
            return;
        }
        
        peTable_.saveValue();
    }

    /**
     * Resets the value of PEs into the previous ones
     * @pre $none
     * @post $none
     */
    public void resetValue()
    {
        if (guiCreated_ == false) {
            return;
        }

        peTable_.resetValue();
    }

    /**
     * Loads XML file
     * @param nodeList a NodeList object
     * @pre nodeList != null
     * @post $none
     */
    public void loadXml(final NodeList nodeList) 
    {
        //hasLoadXml_ = true;
        Node node;
        String name, value;
        Integer id = null;
        Integer rating = null;

        // reset the hashtable
        pe_.clear();

        for (int i = 0; i < nodeList.getLength(); i++)
        {
            node = nodeList.item(i);

            // only element nodes that need to be take care,
            // the rests are ignored
            if (node.getNodeType() != Node.ELEMENT_NODE) {
                continue;
            }

            name = node.getNodeName();
            if (name.equals("peId") == true)
            {
                value = node.getFirstChild().getNodeValue();
                id = new Integer( value.trim() );
            }

            else if (name.equals("MIPSRating") == true)
            {
                value = node.getFirstChild().getNodeValue();
                rating = new Integer( value.trim() );
            }

            if (id != null && rating != null)
            {
                pe_.put(id, rating);
                id = null;
                rating = null;
            }
        }

    }


    //////////////////////// Private Methods
    
    /**
     * Populates all the fields with some random values
     * @param r   a Random object
     * @pre r != null
     * @post $none
     */
    private void randomValue(Random r)
    {
        int total = r.nextInt(21);

        // machine must have at least 2 CPU
        while(true) 
        {
            total = r.nextInt(21);
            if (total > 2) {
                break;
            }
        }

        // init the MIPS rating
        Integer rating = new Integer( r.nextInt(500) );
        for (int i = 0; i < total; i++) {
            pe_.put(new Integer(i), rating);
        }
    }

    /**
     * Initializes all the components 
     * @pre $none
     * @post $none
     */
    private void initComponents()
    {
        peTable_ = new PETable();
        JTable table = new JTable(peTable_);

        // for Dimension(width, height)
        table.setPreferredScrollableViewportSize( new Dimension(180, 250) );

        TableColumn column = null;
        // column 0 is PE Id set to small
        column = table.getColumnModel().getColumn(0);
        column.setPreferredWidth(20);

        column = table.getColumnModel().getColumn(1);
        column.setPreferredWidth(50);

        //Create the scroll pane and add the table to it.
        int vert = ScrollPaneConstants.VERTICAL_SCROLLBAR_ALWAYS;
        int horiz = ScrollPaneConstants.HORIZONTAL_SCROLLBAR_NEVER;
        JScrollPane scrollPane = new JScrollPane(table, vert, horiz);

        //Set up real input validation for the integer column.
        setUpIntegerEditor(table);

        panel_.add(scrollPane);
    }

    /**
     * Sets up the editor for the integer cells
     * @param table a JTable object
     * @pre table != null 
     * @post $none
     */
    private void setUpIntegerEditor(JTable table) 
    {
        //Set up the editor for the integer cells
        final WholeNumberField integerField = new WholeNumberField(0, 10);
        integerField.setHorizontalAlignment(WholeNumberField.RIGHT);

        DefaultCellEditor integerEditor = new DefaultCellEditor(integerField)
        {
            //Override DefaultCellEditor's getCellEditorValue method
            //to return an Integer, not a String:
            public Object getCellEditorValue() {
                return new Integer( integerField.getValue() );
            }
        };

        table.setDefaultEditor(Integer.class, integerEditor);
    }


    //////////////////////// Internal Class

    /**
     * Internal class to store a table of PEs (Processing Elements)
     */
    private class PETable extends AbstractTableModel 
    {
        private final String[] columnNames = {"PE Id", "MIPS Rating"};
        private Hashtable previousValue_;  // useful for reseting the values

        /**
         * Allocates a new PETable object
         * @pre $none
         * @post $none
         */
        private PETable() {
            previousValue_ = new Hashtable();
        }

        /**
         * Gets the number of columns in a table
         * @return number of columns
         * @pre $none
         * @post $result >= 0
         */
        public int getColumnCount() {
            return columnNames.length;
        }

        /**
         * Gets the number of rows in a table
         * @return number of rows
         * @pre $none
         * @post $result >= 0
         */
        public int getRowCount() {
            return pe_.size();
        }

        /**
         * Gets the column name specified by the column number
         * @param col   column number
         * @return column name
         * @pre col >= 0
         * @post $result != null
         */
        public String getColumnName(int col) {
            return columnNames[col];
        }

        /**
         * Gets the Object at a given row and column number
         * @param row   row number
         * @param col   column number
         * @return an Object
         * @pre row >= 0
         * @pre col >= 0
         * @post $result != null
         */
        public Object getValueAt(int row, int col)
        {
            if (col == 0) {
                return new Integer(row);
            }
            
            return pe_.get( new Integer(row) );
        }

        /**
         * JTable uses this method to determine the default renderer/
         * editor for each cell.  If we didn't implement this method,
         * then the numbers will be left-aligned rather than right-aligned.
         * @param col   column number
         * @return a Class object
         * @pre col >= 0
         * @post $result != null
         */
        public Class getColumnClass(int col) {
            return getValueAt(0, col).getClass();
        }

        /**
         * Checks whether the given row and column number cell is editable.
         * Don't need to implement this method unless your table is
         * editable.
         * @param row   row number
         * @param col   column number
         * @return <tt>true</tt> if the cell is editable, <tt>false</tt>
         *         otherwise
         * @pre row >= 0
         * @pre col >= 0
         * @post $none 
         */
        public boolean isCellEditable(int row, int col) 
        {
            // Note that the data/cell address is constant, no matter where 
            // the cell appears onscreen. Only the PE id cannot be changed.
            if (col < 1) {
                return false;
            }
            return true;
        }

        /**
         * Sets the value at a give table cell
         * @param value  an Object to be written
         * @param row    row number
         * @param col    column number
         * @pre value != null
         * @pre row >= 0
         * @pre col >= 0
         * @post $none
         */
        public void setValueAt(Object value, int row, int col)
        {
            Integer index, prev;

            index = new Integer(row);
            prev = (Integer) pe_.put(index, (Integer) value);
            previousValue_.put(index, prev);

            fireTableCellUpdated(row, col);
        }

        /**
         * Saves the changes made to the table
         * @pre $none
         * @post $none
         */
        protected void saveValue() {
            previousValue_.clear();
        }

        /**
         * Resets the table into previous values
         * @pre $none
         * @post $none
         */
        protected void resetValue()
        {
            Integer value, index;

            // loop to check whether there are any old values stored in
            // previousValue_ hashtable. If there are, then put back into pe_.
            for (int i = 0; i < pe_.size(); i++)
            {
                index = new Integer(i);
                if (previousValue_.containsKey(index) == false) {
                    continue;
                }

                value = (Integer) previousValue_.get(index);
                pe_.put(index, value);
            }

            previousValue_.clear();
        }

    } // end class

} // end class

