/*
 * Title:        Visual Modeler for GridSim Toolkit
 * Description:  This Visual Modeler enables the user to quickly create
 *               experiments on different Grid testbeds and generate the
 *               default Grid Broker source codes (in Java).
 *
 * $Id: MenuView.java,v 1.4 2003/06/13 04:24:31 anthony Exp $
 */

package visualmodeler;

import java.io.*;
import java.util.Observer;
import java.util.Observable;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.filechooser.*;
import java.awt.*;
import java.awt.event.*;


/**
 * MenuView setups the menu bar
 *
 * @author       Anthony Sulistio and Chee Shin Yeo
 * @version      1.1
 * @invariant $none
 */
public class MenuView implements ActionListener, Observer
{
    private JFrame frame_;
    private FileModel fileModel_;
    private JMenuItem saveItem_;

    
    /**
     * Constructs the menu bar
     * @param frame   Frame to setup the menu bar
     * @param model   File model for file menu functions
     * @pre frame != null
     * @pre model != null
     * @post $none
     */
    public MenuView(JFrame frame, FileModel model)
    {
        frame_ = frame;
        fileModel_ = model;

        initComponents();
    }

    /**
     * Action to be performed when event is triggered
     * @param evt  an ActionEvent object
     * @pre evt != null
     * @post $none
     */
    public void actionPerformed (ActionEvent evt)
    {
        String cmd = evt.getActionCommand();

        if (cmd.equals("new") == true) {
            fileModel_.newFile();
        }
        else if (cmd.equals("open") == true) {
            fileModel_.openFile();
        }
        else if (cmd.equals("close") == true) {
            fileModel_.closeFile();
        }
        else if (cmd.equals("save") == true) {
            fileModel_.saveFile();
        }
        else if (cmd.equals("save_as") == true) {
            fileModel_.saveAsFile();
        }
        else if (cmd.equals("exit") == true)
        {
            if (fileModel_.quitProgram() == true)
            {
                frame_.dispose();
                System.exit(0);
            }
        }
        else if (cmd.equals("code") == true) {
            fileModel_.generateCode();
        }
        else if (cmd.equals("about") == true) {
            new MenuAbout(frame_).show();
        }
    }

    /**
     * Update display of object
     * @param obs    Object that is observed
     * @param obj    Object to update
     * @pre obs != null
     * @pre obj != null
     * @post $none
     */
    public void update(Observable obs, Object obj)
    {
        if (obj == null) {
            saveItem_.setEnabled(true);
        }
        else {
            saveItem_.setEnabled(false);
        }
    }

    ////////////////////////////////// PRIVATE METHODS //////////////////////

    /**
     * This method is called from within the constructor to
     * initialize the form
     * @pre $none
     * @post $none
     */
    private void initComponents()
    {
        // create a menu bar and add it to frame
        JMenuBar menuBar = new JMenuBar();
        frame_.setJMenuBar(menuBar);

        // File Menu
        JMenu file = new JMenu("File");
        file.setMnemonic(KeyEvent.VK_F);

        addMenu(file, "New", "new", KeyEvent.VK_N);
        addMenu(file, "Open...", "open", KeyEvent.VK_O);
        addMenu(file, "Close", "close", KeyEvent.VK_C);

        saveItem_ = new JMenuItem("Save");
        addMenu(file, saveItem_, "save", KeyEvent.VK_S);
        addMenu(file, "Save As..", "save_as", KeyEvent.VK_F12);
        file.addSeparator();
        addMenu(file, "Exit", "exit", KeyEvent.VK_Q);

        // Project Menu
        JMenu project = new JMenu("Project");
        project.setMnemonic(KeyEvent.VK_P);

        addMenu(project, "Generate GridSim code", "code", KeyEvent.VK_G);

        // Help Menu
        JMenu help = new JMenu("Help");
        help.setMnemonic(KeyEvent.VK_H);

        addMenu(help, "About", "about", KeyEvent.VK_A);

        // Add menus into the menu bar
        menuBar.add(file);
        menuBar.add(project);
        menuBar.add(help);
    }

    /**
     * Add menu item to menu
     * @param file    Menu where item is to be added
     * @param name    Text to be displayed for menu item
     * @param cmd     command to be executed when menu item is activated
     * @param key     Key set for menu item
     * @pre file != null
     * @pre name != null
     * @pre cmd != null
     * @post $none
     */
    private void addMenu(JMenu file, String name, String cmd, int key)
    {
        JMenuItem item = new JMenuItem(name);
        item.setMnemonic(key);
        item.setAccelerator(KeyStroke.getKeyStroke(key,ActionEvent.CTRL_MASK));
        item.setActionCommand(cmd);
        item.addActionListener(this);

        file.add(item);
    }

    /**
     * Add menu item to menu
     * @param file    Menu where item is to be added
     * @param item    Menu Item to be added
     * @param cmd     command to be executed when menu item is activated
     * @param key     Key set for menu item
     * @pre file != null
     * @pre item != null
     * @pre cmd != null
     * @post $none
     */
    private void addMenu(JMenu file, JMenuItem item, String cmd, int key)
    {
        item.setEnabled(false);
        item.setActionCommand(cmd);
        item.setMnemonic(key);
        item.setAccelerator(KeyStroke.getKeyStroke(key,ActionEvent.CTRL_MASK));
        item.addActionListener(this);

        file.add(item);
    }

} // end class

