/*
 * Title:        Visual Modeler for GridSim Toolkit
 * Description:  This Visual Modeler enables the user to quickly create
 *               experiments on different Grid testbeds and generate the
 *               default Grid Broker source codes (in Java).
 *
 * $Id: IconWizard.java,v 1.7 2003/06/13 04:10:26 anthony Exp $
 */

package visualmodeler;

import javax.swing.*;
import javax.swing.event.*;
import java.awt.*;
import java.awt.event.*;

/**
 * IconWizard setups the dialog box for the wizard function
 *
 * @author       Anthony Sulistio and Chee Shin Yeo
 * @version      1.1
 * @invariant $none
 */
public class IconWizard extends JDialog implements ActionListener
{
    private final int USER = 0;
    private final int RESOURCE = 1;

    private UserModel userModel_;
    private ResourceModel resModel_;

    private JCheckBox check_;
    private JTextField userText_;
    private JTextField resText_;
    private JButton ok_, cancel_;

    private int numUser_;   // num of user
    private int numRes_;    // num of resource

    
    /**
     * Constructs the wizard dialog box
     * @param parent       Parent frame
     * @param user         User model
     * @param res          Resource model
     * @pre parent != null
     * @pre user != null
     * @pre res != null
     * @post $none
     */
    public IconWizard(JFrame parent, UserModel user, ResourceModel res)
    {
        super(parent, true);
        setSize(400, 250);
        setLocation(150, 150);
        setTitle("Wizard Dialog");

        userModel_ = user;
        resModel_ = res;

        numUser_ = 0;
        numRes_ = 0;

        initComponent();
    }

    /**
     * Shows the dialog box
     * @pre $none
     * @post $none
     */
    public void showDialog()
    {
        userText_.addActionListener(this);
        resText_.addActionListener(this);
        ok_.addActionListener(this);
        cancel_.addActionListener(this);

        this.show();
    }

    /**
     * Action to be performed when event is triggered
     * @param evt          Event
     * @pre evt != null
     * @post $none
     */
    public void actionPerformed (ActionEvent evt)
    {
        String cmd = evt.getActionCommand();

        if (cmd.equals("user") == true) {
            collectInputValue(USER, userText_.getText());
        }
        else if (cmd.equals("resource") == true) {
            collectInputValue(RESOURCE, resText_.getText());
        }
        else if (cmd.equals("ok") == true)
        {
            boolean userFlag = true;
            boolean resFlag = true;

            userFlag = collectInputValue(USER, userText_.getText());
            resFlag = collectInputValue(RESOURCE, resText_.getText());

            if (userFlag == true && resFlag == true)
            {
                if (numUser_ > 0) {
                    userModel_.createUser(numUser_, check_.isSelected());
                }
                
                if (numRes_ > 0) {
                    resModel_.createResource(numRes_, check_.isSelected());
                }
                resetValue();
                dispose();
            }
        }

        else
        {
            resetValue();
            dispose();
        }

    }

    ///////////////////////// PRIVATE METHODS //////////////////////////////

    /**
     * This method is called from within the constructor to
     * initialize the form.
     * @pre $none
     * @post $none
     */
    private void initComponent()
    {
        GridBagLayout gridbag = new GridBagLayout();
        GridBagConstraints c = new GridBagConstraints();
        getContentPane().setLayout(gridbag);

        // Create a label to display the user value
        int index = 0;
        c.gridy = index++;
        c.weighty = 0.5;
        c.insets = new Insets(10, 0, 5, 0);
        c.fill = GridBagConstraints.HORIZONTAL;
        Font font = new Font(null, Font.PLAIN, 14);
        JLabel userLabel = new JLabel("Number of users:   ");
        userLabel.setFont(font);
        getContentPane().add(userLabel, c);

        userText_ = new JTextField("" + numUser_, 5);
        userText_.setActionCommand("user");
        getContentPane().add(userText_, c);

        //////////////////////
        c.gridy = index++;
        JLabel resLabel = new JLabel("Number of resources:   ");
        resLabel.setFont(font);
        getContentPane().add(resLabel, c);

        resText_ = new JTextField("" + numRes_, 5);
        resText_.setActionCommand("resource");
        getContentPane().add(resText_, c);

        // Create a check box
        c.gridy = index++;
        c.gridwidth = 3;
        c.weighty = 1.0;   //request any extra vertical space
        check_ = new JCheckBox("Auto-generate random user and " +
                            "resource properties", true);
        getContentPane().add(check_, c);

        // reset values
        c.gridy = index++;
        c.gridwidth = 1;
        c.weighty = 0.5;
        c.fill = GridBagConstraints.NONE;
        ok_ = new JButton("OK");
        ok_.setActionCommand("ok");
        getContentPane().add(ok_, c);

        c.gridwidth = GridBagConstraints.REMAINDER;
        cancel_ = new JButton("Cancel");
        cancel_.setActionCommand("cancel");
        getContentPane().add(cancel_, c);
    }

    /**
     * Resets the values
     * @pre $none
     * @post $none
     */
    private void resetValue()
    {
        check_.setSelected(true);
        numUser_ = 0;
        numRes_ = 0;

        userText_.setText("" + numUser_);
        resText_.setText("" + numRes_);

        // remove all the listeners to prevent memory leakages
        userText_.removeActionListener(this);
        resText_.removeActionListener(this);
        ok_.removeActionListener(this);
        cancel_.removeActionListener(this);
    }

    /**
     * Whether the input value is collected
     * @param type       type of input, user or resource
     * @param value      integer value of input
     * @return           <code>true</code> if input value is collected
     *                   <code>false</code> otherwise.
     * @pre value != null
     * @post $none
     */
    private boolean collectInputValue(final int type, String value)
    {
        try
        {
            int num = new Integer(value).intValue();
            if (num < 0) {
                throw (new Exception());
            }
            
            if (type == USER) {
                numUser_ = num;
            }
            else {
                numRes_ = num;
            }
        }
        catch (Exception e)
        {
            JOptionPane.showMessageDialog(this,
                "Invalid number. It should be a positive integer.",
                "Set " + type + " number error", JOptionPane.ERROR_MESSAGE);

            return false;
        }

        return true;
    }

} // end class

