/*
 * Title:        Visual Modeler for GridSim Toolkit
 * Description:  This Visual Modeler enables the user to quickly create
 *               experiments on different Grid testbeds and generate the
 *               default Grid Broker source codes (in Java).
 *
 * $Id: IconView.java,v 1.12 2003/06/20 12:45:45 anthony Exp $
 */

package visualmodeler;

import java.awt.*;
import java.util.Observer;
import java.util.Observable;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import javax.swing.*;
import java.net.URL;


/**
 * IconView setup the icon toolbar
 *
 * @author       Anthony Sulistio and Chee Shin Yeo
 * @version      1.1
 * @invariant $none
 */
public class IconView extends JPanel implements ActionListener, Observer
{
    private JButton saveButton_;
    private IconWizard iconWiz_;
    private FileModel fileModel_;
    private JFrame frame_;


    /**
     * Construct the toolbar
     * @param frame         a JFrame object
     * @param orientation   the display layout
     * @param file          File model for toolbar buttons that deals with file
     * @param userModel     User model
     * @param resModel      Resource model
     * @pre frame != null
     * @pre orientation != null
     * @pre file != null
     * @pre userModel != null
     * @pre resModel != null
     * @post $none
     */
    public IconView(JFrame frame, final String orientation, FileModel file,
                UserModel userModel, ResourceModel resModel)
    {
        frame_ = frame;
        fileModel_ = file;
        iconWiz_ = new IconWizard(frame, userModel, resModel);

        createIconBar(orientation);
    }

    /**
     * Action to be performed when event is triggered
     * @param evt       an action event
     * @pre evt != null
     * @post $none
     */
    public void actionPerformed (ActionEvent evt)
    {
        String cmd = evt.getActionCommand();

        if (cmd.equals("new_file") == true) {
            fileModel_.newFile();
        }
        else if (cmd.equals("open_file") == true) {
            fileModel_.openFile();
        }
        else if (cmd.equals("close_file") == true) {
            fileModel_.closeFile();
        }
        else if (cmd.equals("save_file") == true) {
            fileModel_.saveFile();
        }
        else if (cmd.equals("wizard") == true) {
            iconWiz_.showDialog();
        }
        else if (cmd.equals("code") == true) {
            fileModel_.generateCode();
        }
        else if (cmd.equals("quit") == true) 
        {
            if (fileModel_.quitProgram() == true)
            {
                frame_.dispose();
                System.exit(0);
            }
        }
    }

    /**
     * Update display of object
     * @param obs    Object that is observed
     * @param obj    Object to update
     * @pre obs != null
     * @pre obj != null
     * @post $none
     */
    public void update(Observable obs, Object obj)
    {
        if (obj == null) {
            saveButton_.setEnabled(true);
        }
        else {
            saveButton_.setEnabled(false);
        }
    }


    //////////////////////////// PRIVATE METHODS //////////////////////////

    /**
     * Create icon toolbar
     * @param orientation   Display layout
     * @pre orientation != null
     * @post $none
     */
    private void createIconBar(final String orientation)
    {
        JToolBar toolBar = new JToolBar();
        toolBar.setOpaque(true);
        toolBar.setFloatable(false);
        toolBar.setBorderPainted(true);
        toolBar.setBorder(BorderFactory.createEtchedBorder());
        toolBar.setMargin(new Insets(0, 15, 0, 15));  

        String dir = "";
        try {
            // Searching for relative path with image directory in it
            dir = System.getProperty("user.dir") + "/image/";
        }
        catch (Exception e) 
        {
            System.out.println(
                    "VisualModeler : Error - Can't find image files.");
        }

        makeToolBarButton(toolBar, dir, "file_new.gif",
                "New File", "new_file");
        makeToolBarButton(toolBar, dir, "file_open.gif", "Open File",
                "open_file");
        makeToolBarButton(toolBar, dir, "file_close.gif", "Close File",
                "close_file");

        saveButton_ = new JButton();
        makeToolBarButton(toolBar, saveButton_, dir, "file_save.gif",
                "Save File", "save_file");
        toolBar.addSeparator();

        makeToolBarButton(toolBar, dir, "wizard.gif","Auto wizard","wizard");
        makeToolBarButton(toolBar, dir, "code.gif", "Generate GridSim code",
                "code");
        toolBar.addSeparator();
        makeToolBarButton(toolBar, dir, "exit.gif", "Quit Program", "quit");

        // make this tool bar visible to the main frame/window
        frame_.getContentPane().add(toolBar, orientation);
    }

    /**
     * Make button on the toolbar
     * @param toolBar       a JToolbar object
     * @param dir           the absolute directory path
     * @param icon          Icon to be shown for the button
     * @param name          Tooltip to be shown for the button
     * @param cmd           Command to be executed when button is activated
     * @pre toolBar != null
     * @pre dir != null
     * @pre icon != null
     * @pre name != null
     * @pre cmd != null
     * @post $none
     */
    private void makeToolBarButton(JToolBar toolBar, String dir, String icon,
                        String name, String cmd)
    {
        ImageIcon imageIcon = new ImageIcon(dir + icon);
        
        // If the user decided to run Visual Modeler directly from jar file
        // then look the images inside the file
        if (imageIcon.getIconWidth() == -1)
        {
            try 
            {
                // get the location from jar file
                URL url = super.getClass().getResource("image/" + icon); 
                Image image = Toolkit.getDefaultToolkit().getImage(url); 
                imageIcon.setImage(image);
            }
            catch (Exception e) 
            {
                System.out.println("Visual Modeler : Error - Can't find " +
                        "image files from visualmodeler.jar");
            }
        }
        
        JButton button = new JButton(imageIcon);
        button.setToolTipText(name);
        button.setBorderPainted(true);
        button.setActionCommand(cmd);
        button.addActionListener(this);

        toolBar.add(button);
    }

    /**
     * Make button on the toolbar
     * @param toolBar       a JToolbar object
     * @param button        a JButton object
     * @param dir           the absolute directory path
     * @param icon          Icon to be shown for the button
     * @param name          Tooltip to be shown for the button
     * @param cmd           Command to be executed when button is activated
     * @pre toolBar != null
     * @pre button != null
     * @pre icon != null
     * @pre name != null
     * @pre cmd != null
     * @post $none
     */
    private void makeToolBarButton(JToolBar toolBar, JButton button, 
                        String dir, String icon, String name, String cmd)
    {
        ImageIcon imageIcon = new ImageIcon(dir + icon);
        
        // If the user decided to run Visual Modeler directly from jar file
        // then look the images inside the file
        if (imageIcon.getIconWidth() == -1)
        {
            try 
            {
                // get the location from jar file
                URL url = super.getClass().getResource("image/" + icon); 
                Image image = Toolkit.getDefaultToolkit().getImage(url); 
                imageIcon.setImage(image);
            }
            catch (Exception e) 
            {
                System.out.println("Visual Modeler : Error - Can't find " +
                        "image files from visualmodeler.jar");
            }
        }
 
        button.setIcon(imageIcon);
        button.setEnabled(false);
        button.setToolTipText(name);
        button.setBorderPainted(true);
        button.setActionCommand(cmd);
        button.addActionListener(this);

        toolBar.add(button);
    }

} // end class

