/*
 * Title:        Visual Modeler for GridSim Toolkit
 * Description:  This Visual Modeler enables the user to quickly create
 *               experiments on different Grid testbeds and generate the
 *               default Grid Broker source codes (in Java).
 *
 * $Id: FileModel.java,v 1.6 2003/06/19 07:31:00 anthony Exp $
 */

package visualmodeler;

import java.awt.Toolkit;
import java.util.*;
import java.io.*;
import javax.swing.*;

// to handle loading XML. These packages exist since java 1.4
import org.w3c.dom.*;
import javax.xml.parsers.*;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;


/**
 * FileModel deals with the back-end operation of the VisualModeler system.
 * The operations handle by this class are:
 * <ul>
 * <li> saving new file in XML format
 * <li> opening new file in XML format
 * <li> generating Java code to be used in GridSim simulation
 * </ul>
 *
 * @author       Anthony Sulistio and Chee Shin Yeo
 * @version      1.1
 * @invariant $none
 */
public class FileModel extends Observable
{
    private DefaultFileFilter fileFilter_;
    private JFrame frame_;
    private UserModel user_;
    private ResourceModel res_;
    private final JFileChooser fc_;

    private boolean hasSaved_;
    private JOptionPane optionPane_;
    private String fileName_;
    private Toolkit toolkit_;


    /**
     * Allocates a new FileModel object
     * @param frame  a JFrame object
     * @param user   a UserModel object
     * @param res    a ResourceModel object
     * @pre frame != null
     * @pre user != null
     * @pre res != null
     * @post $none
     */
    public FileModel(JFrame frame, UserModel user, ResourceModel res)
    {
        frame_ = frame;
        user_ = user;
        res_ = res;

        fileName_ = null;
        hasSaved_ = false;

        toolkit_ = Toolkit.getDefaultToolkit();
        fc_ = new JFileChooser();
        fileFilter_ = new DefaultFileFilter("xml", "XML Files");

        optionPane_ = new JOptionPane("Empty message",
                                JOptionPane.QUESTION_MESSAGE,
                                JOptionPane.YES_NO_OPTION);
    }

    /**
     * Saves the Visual Modeler project file 
     * @pre $none
     * @post $none
     */
    public void saveFile()
    {
        // if no changes to both user and resource model, then exit
        if (user_.hasChanged() == false && res_.hasChanged() == false) {
            return;
        }

        if (fileName_ != null)
        {
            save();
            return;
        }

        fc_.setDialogTitle("Save");
        fc_.setFileFilter(fileFilter_);

        int returnVal = fc_.showSaveDialog(frame_);
        if (returnVal == JFileChooser.APPROVE_OPTION) {
            displaySaveDialog();
        }
    }

    /**
     * A method that asks the user to save the changes before exiting the
     * program 
     * @pre $none
     * @post $none
     */
    public void newFile()
    {
        // if the file has been saved before then open a new file and
        // reset the flag
        if (hasSaved_ == true)
        {
            hasSaved_ = false;
            user_.newValue();
            res_.newValue();
            fileName_ = null;
            return;
        }

        // if no changes to both user and resource model, then exit
        if (user_.hasChanged() == false && res_.hasChanged() == false) {
            return;
        }

        String msg="This file has not been saved yet.\nOpen a new file anyway?";
        if (confirmAction(msg, "New File") == true)
        {
            user_.newValue();
            res_.newValue();
            hasSaved_ = false;
            fileName_ = null;
        }
    }

    /**
     * A method that handles the dialog to open a new file 
     * @pre $none
     * @post $none
     */
    public void openFile()
    {
        // if no changes to both user and resource model, then open
        if (user_.hasChanged() == false && res_.hasChanged() == false)
        {
            displayOpenDialog();
            return;
        }

        String msg="This file has not been saved yet.\nOpen a new file anyway?";
        if (confirmAction(msg, "Open File") == true) {
            displayOpenDialog();
        }
    }

    /**
     * A method that shows a close file dialog 
     * @pre $none
     * @post $none
     */
    public void closeFile()
    {
        // if no changes to both user and resource model, then exit
        if (user_.hasChanged() == false && res_.hasChanged() == false) {
            return;
        }

        String msg = "The file has not been saved yet.\nClose anyway?";
        if (confirmAction(msg, "Close File") == true)
        {
            user_.newValue();
            res_.newValue();
            fileName_ = null;
        }
    }

    /**
     * A method that shows a Save As dialog 
     * @pre $none
     * @post $none
     */
    public void saveAsFile()
    {
        fc_.setDialogTitle("Save As");
        fc_.setFileFilter(fileFilter_);

        int returnVal = fc_.showSaveDialog(frame_);
        if (returnVal == JFileChooser.APPROVE_OPTION) {
            displaySaveDialog();
        }
    }

    /**
     * A method that asks the user whether to quit the program or not
     * @return <tt>true</tt> if the user wants to quit, <tt>false</tt> otherwise
     * @pre $none
     * @post $none
     */
    public boolean quitProgram()
    {
        // if no changes to both user and resource model, then exit
        if (user_.hasChanged() == false && res_.hasChanged() == false) {
            return true;
        }

        String msg = "The file has not been saved yet.\nQuit anyway?";
        if (confirmAction(msg, "Exit Program") == true)
        {
            user_.newValue();
            res_.newValue();

            return true;
        }

        return false;
    }

    /**
     * Generates a Java source code 
     * @pre $none
     * @post $none
     */
    public void generateCode()
    {
        // if no changes to both user and resource model, then exit
        /******
        if (user_.hasChanged() == false && res_.hasChanged() == false)
        {
            toolkit_.beep();
            JOptionPane.showMessageDialog(frame_,
                "Cannot generate GridSim code.\n" +
                "No user or resource has been created.",
                "Generate GridSim code",
                JOptionPane.ERROR_MESSAGE);

            return;
        }
        *************/

        DefaultFileFilter filter = new DefaultFileFilter("java", "Java Files");

        fc_.setDialogTitle("Generate GridSim code");
        fc_.setFileFilter(filter);

        int returnVal = fc_.showSaveDialog(frame_);
        if (returnVal != JFileChooser.APPROVE_OPTION) {
            return;
        }

        // get the name of the file and use it as a java class name
        File file = fc_.getSelectedFile();
        String className = file.getName();
        className = fileFilter_.getFileNameOnly(className);

        // Creates a string buffer with long chars to increase performance
        StringBuffer buffer = new StringBuffer(1000); 
        
        // this will get the absolute directory path + file name
        buffer.append( file.getAbsolutePath() );

        // if the filename has no extension, then attach
        if (fileFilter_.getExtension(file) == null) {
            buffer.append(".java");
        }

        // Copy the file name
        String name = buffer.toString();

        // if a file already exists, then double check with user
        File newFile = new File(name);
        if (newFile.exists() == true)
        {
            // reset the buffer
            buffer.delete( 0, buffer.length() );
            buffer.append(name);
            buffer.append(" already exists.\nDo you want to replace it?");
            
            if ( confirmAction(buffer.toString(), "Generate GridSim code") 
                    == false ) 
            {
                return;
            }
        }

        String indent = "    ";     // 4 spaces
        
        // reset the buffer
        buffer.delete( 0, buffer.length() );

        // Writes the Java file header
        buffer.append("// This file is auto-generated by VisualModeler.\n");
        buffer.append("// Created on ");

        Date date = Calendar.getInstance().getTime();
        buffer.append( date.toString() );
        buffer.append("\n\n");
        buffer.append("import java.util.*;\n");
        buffer.append("import gridsim.*;\n");
        buffer.append("import gridbroker.*;\n");

        // writes the class name and static method
        buffer.append("\npublic class ");
        buffer.append(className);
        buffer.append(" \n{\n\n");
        buffer.append(indent);
        buffer.append("public static void main(String[] args)\n");
        buffer.append(indent);
        buffer.append("{\n\n");

        // writes statements inside the main method
        String doubleIndent = "        ";   // 8 spaces
        buffer.append(doubleIndent);
        buffer.append("try\n");
        buffer.append(doubleIndent);
        buffer.append("{\n");

        String tripleIndent = "            ";   // now become 12 spaces
        buffer.append(tripleIndent);
        buffer.append("Calendar calendar = Calendar.getInstance();\n");
        buffer.append(tripleIndent);
        buffer.append("Random random;\n");
        buffer.append(tripleIndent);
        buffer.append("long seed = 11L*13*17*19*23+1;\n");
        buffer.append(tripleIndent);
        buffer.append("seed = seed * 97 + 1;\n\n");

        buffer.append(tripleIndent);
        buffer.append("String[] eff = {\"\"};\n");
        buffer.append(tripleIndent);
        buffer.append("String[] efp = {\"\"};\n");
        buffer.append(tripleIndent);
        buffer.append("String[] category = { \"*.USER.GridletCompletion\",\n");
        buffer.append(indent);
        buffer.append(tripleIndent);
        buffer.append("\"*.USER.TimeUtilization\", ");
        buffer.append("\"*.USER.BudgetUtilization\" };\n\n");
        buffer.append(tripleIndent);
        buffer.append("String ReportWriterName = \"ReportWriter");
        buffer.append(className);
        buffer.append("\";\n\n");

        // writes gridsim initialization
        int userNum = user_.getTotalUser();
        int resNum = res_.getTotalResource();

        buffer.append(tripleIndent);
        buffer.append("// Initializing GridSim for ");
        buffer.append(userNum);
        buffer.append(" grid users and ");
        buffer.append(resNum);
        buffer.append(" grid resources\n");
        buffer.append(tripleIndent);
        buffer.append("GridSim.init(");
        buffer.append(userNum);
        buffer.append(", calendar, true, eff, efp, ReportWriterName);\n\n");

        buffer.append(tripleIndent);
        buffer.append("// Creates Report Writer entity\n");
        buffer.append(tripleIndent);
        buffer.append("new ReportWriter(ReportWriterName, ");
        buffer.append(userNum);
        buffer.append(", ");
        buffer.append(resNum);
        buffer.append(", \"reportFile");
        buffer.append(className);
        buffer.append("\",\n"); 
        buffer.append(indent);
        buffer.append(tripleIndent);
        buffer.append("category, false, null, true, 10000);\n\n");
        
        // writes grid resources code
        buffer.append( res_.generateCode(tripleIndent) );

        // writes grid users code 
        buffer.append( user_.generateCode(tripleIndent) );

        buffer.append("\n");
        buffer.append(tripleIndent);
        buffer.append("// Starts grid simulation\n");
        buffer.append(tripleIndent);
        buffer.append("GridSim.startGridSimulation();\n");

        // writes catch clause
        buffer.append(doubleIndent);
        buffer.append("}\n");
        buffer.append(doubleIndent);
        buffer.append("catch (Exception e) {\n");
        buffer.append(tripleIndent);
        buffer.append("e.printStackTrace();\n");
        buffer.append(doubleIndent);
        buffer.append("}\n    } \n} \n\n");  // end of main and class

        try
        {
            FileOutputStream out = new FileOutputStream(name);
            BufferedOutputStream outBuffer = new BufferedOutputStream(out);
            outBuffer.write( buffer.toString().getBytes() );
            outBuffer.close();
        }

        catch(Exception e)
        {
            toolkit_.beep();
            JOptionPane.showMessageDialog(frame_,
                    "Sorry, cannot generate Java file.",
                    "Generate GridSim code",
                    JOptionPane.ERROR_MESSAGE);
        }

        toolkit_.beep();
        JOptionPane.showMessageDialog(frame_,
                "Finish generating Java code.",
                "Generate GridSim code",
                JOptionPane.INFORMATION_MESSAGE);
    }


    /////////////////////////////// PRIVATE METHODS ////////////////////////

    /**
     * @param msg   A message 
     * @param type  type of action
     * @return <tt>true</tt> if user wants to save, <tt>false</tt> otherwise
     * @pre msg != null
     * @pre type != null
     * @post $none
     */
    private boolean confirmAction(String msg, String type)
    {
        optionPane_.setMessage(msg);
        JDialog dialog = optionPane_.createDialog(frame_, type);
        toolkit_.beep();
        dialog.show();

        // put try and catch, just in-case the something wrong with casting
        try
        {
            int value = ( (Integer) optionPane_.getValue() ).intValue();
            if (value == JOptionPane.YES_OPTION) {
                return true;
            }
        }
        catch (Exception e) {
            System.out.println("FileModel.confirmAction() : " + e.getMessage());
        }

        return false;
    }

    /**
     * A method that displays the open dialog 
     * @pre $none
     * @post $none
     */
    private void displayOpenDialog()
    {
        fc_.setDialogTitle("Open");
        fc_.setFileFilter(fileFilter_);

        int returnVal =  fc_.showOpenDialog(frame_);
        if (returnVal == JFileChooser.APPROVE_OPTION)
        {
            File file = fc_.getSelectedFile();

            // clear everything
            user_.newValue();
            res_.newValue();
            fileName_ = file.getAbsolutePath();

            // if the filename has no extension, then attach
            String ext = fileFilter_.getExtension(file);
            if (ext == null) {
                fileName_ += ".xml";
            }

            loadXmlFile(fileName_);
        }
    }

    /**
     * A method that reads the given XML (eXtensible Markup Language) file 
     * @param fileName  A file name
     * @pre fileName != null
     * @post $none
     */
    private void loadXmlFile(String fileName)
    {
        // Create an instance of the DocumentBuilderFactory
        DocumentBuilderFactory docFac = DocumentBuilderFactory.newInstance();

        try
        {
            //Get the DocumentBuilder from the factory that we just got above.
            DocumentBuilder docBuilder = docFac.newDocumentBuilder();

            // turn it into an in-memory object
            Document doc = docBuilder.parse(fileName);

            // root element, i.e <gridsimProject>
            Element root = doc.getDocumentElement();
            NodeList nodeList = root.getChildNodes();
            
            Node node;
            String name;

            int length = nodeList.getLength();
            for (int i = 0; i < length; i++)
            {
                node = nodeList.item(i);

                if (node.getNodeType() == Node.ELEMENT_NODE)
                {
                    name = node.getNodeName();
                    NodeList childList = node.getChildNodes();
                    if (name.equals("gridUser") == true) {
                        user_.loadXml(childList);
                    }

                    else if (name.equals("gridResource") == true) {
                        res_.loadXml(childList);
                    }
                }
            }
        }
        catch (Exception ex) {
            System.out.println("FileModel.loadXMLFile() : Error - unable to " +
                    "open and/or read \"" + fileName + "\"");        
        }
    }

    /**
     * A method that displays a save dialog 
     * @pre $none
     * @post $none
     */
    private void displaySaveDialog()
    {
        File file = fc_.getSelectedFile();

        // this will get the absolute directory path + file name
        fileName_ = file.getAbsolutePath();

        // if the filename has no extension, then attach
        String ext = fileFilter_.getExtension(file);
        if (ext == null) {
            fileName_ += ".xml";
        }

        // if a file already exists, then double check with user
        File newFile = new File(fileName_);
        if (newFile.exists() == true)
        {
            StringBuffer msg = new StringBuffer();
            msg.append(fileName_);
            msg.append(" already exists.\n");
            msg.append("Do you want to replace it?");
            
            if (confirmAction(msg.toString(), "Save File") == false)
            {
                fileName_ = null;
                return;
            }
        }

        save();
    }

    /**
     * Saves the Visual Modeler project file into XML format 
     * @pre $none
     * @post $none
     */
    private void save()
    {
        String indent = "    ";  // 4 spaces
        StringBuffer xml = new StringBuffer(1000);
        xml.append("<?xml version=\"1.0\" standalone=\"yes\"?>\n\n");
        
        xml.append("<!-- This file is auto-generated by VisualModeler.\n");
        xml.append("     Created on ");

        Date date = Calendar.getInstance().getTime();
        xml.append( date.toString() );
        xml.append("\n-->\n\n");
        
        xml.append("\n<gridsimProject>");

        // saving grid user
        xml.append( user_.saveFile(indent) );

        // saving grid resource
        xml.append( res_.saveFile(indent) );
        xml.append("\n</gridsimProject>\n");

        try
        {
            FileOutputStream out = new FileOutputStream(fileName_);
            out.write( xml.toString().getBytes() );
            out.close();

            hasSaved_ = true;
            setChanged();
            notifyObservers("save");
        }
        catch(Exception e)
        {
            toolkit_.beep();
            JOptionPane.showMessageDialog(frame_,
                    "Sorry, cannot save this file.",
                    "Saving error",
                    JOptionPane.ERROR_MESSAGE);
        }

        toolkit_.beep();
        JOptionPane.showMessageDialog(frame_,
                "Finish saving file.",
                "Saving file success",
                JOptionPane.INFORMATION_MESSAGE);

    }

} // end class

