/*
 * Title:        Visual Modeler for GridSim Toolkit
 * Description:  This Visual Modeler enables the user to quickly create
 *               experiments on different Grid testbeds and generate the
 *               default Grid Broker source codes (in Java).
 *
 * $Id: DisplayView.java,v 1.17 2003/06/20 12:45:45 anthony Exp $
 */

package visualmodeler;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.border.*;
import java.awt.*;
import java.awt.event.*;
import java.util.*;
import java.net.URL;


/**
 * DisplayView sets the visual display for the Visual Modeler
 *
 * @author       Anthony Sulistio and Chee Shin Yeo
 * @version      1.1
 * @invariant $none
 */
public class DisplayView extends JPanel
                         implements ActionListener, Observer
{
    private final int USER = 0;
    private final int RESOURCE = 1;

    private JFrame frame_;

    private UserModel user_;
    private ResourceModel res_;

    private JList userList_,    resList_;
    private JLabel userLabel_,  resLabel_;


    /**
     * Constructs the display view for the frame with the orientation
     * @param frame         a JFrame object
     * @param orientation   Display layout
     * @param model         User model
     * @param res           Resource model
     * @pre frame != null
     * @pre orientation != null
     * @pre model != null
     * @pre res != null
     * @post $none
     */
    public DisplayView(JFrame frame, final String orientation,
                UserModel model, ResourceModel res)
    {
        frame_ = frame;
        user_ = model;
        res_ = res;

        createDisplayPanel(orientation);
    }

    /**
     * Action to be performed when event is triggered
     * @param evt   Event
     * @pre evt != null
     * @post $none
     */
    public void actionPerformed(ActionEvent evt)
    {
        String cmd = evt.getActionCommand();
        int index = 0;

        if (cmd.equals("add_user") == true) {
            user_.createUser(1, true);
        }
        else if (cmd.equals("add_res") == true) {
            res_.createResource(1, true);
        }
        else if (cmd.equals("del_user") == true)
        {
            index = userList_.getSelectedIndex();
            if (index == -1) {
                return;
            }

            user_.removeUser(index);
        }
        else if (cmd.equals("del_res") == true)
        {
            index = resList_.getSelectedIndex();
            if (index == -1) {
                return;
            }

            res_.removeResource(index);
        }
        else if (cmd.equals("view_user") == true)
        {
            index = userList_.getSelectedIndex();
            if (index == -1) {
                return;
            }

            user_.showProperty(index);
        }

        else if (cmd.equals("view_res") == true)
        {
            index = resList_.getSelectedIndex();
            if (index == -1) {
                return;
            }

            res_.showProperty(index);
        }
    }

    /**
     * Updates display of object
     * @param obs    Object that is observed
     * @param obj    Object to update
     * @pre obs != null
     * @pre obj != null
     * @post $none
     */
    public void update(Observable obs, Object obj)
    {
        int size = 0;
        if (obs == user_)
        {
            size = user_.getTotalUser();
            userLabel_.setText("          Total Users:  " + size);

            userList_.clearSelection();
            userList_.setListData( user_.getAllUserName() );
            userList_.setSelectedIndex(size - 1);
        }

        else if (obs == res_)
        {
            size = res_.getTotalResource();
            resLabel_.setText("          Total Resources:  " + size);

            resList_.clearSelection();
            resList_.setListData( res_.getAllResourceName() );
            resList_.setSelectedIndex(size - 1);
        }

        revalidate();
        repaint();
    }

    /////////////////////////// PRIVATE METHODS /////////////////////////////
    
    /**
     * Creates the display panel with the orientation
     * @param orientation     Display layout
     * @pre orientation != null
     * @post $none
     */
    private void createDisplayPanel(final String orientation)
    {
        // row - col - horiz - vertical
        int HORIZ = 30;
        int VERT = 20;
        JPanel panel = new JPanel(new GridLayout(0, 1, HORIZ, HORIZ));
        JPanel userPanel = new JPanel(new BorderLayout(HORIZ, VERT));
        JPanel resPanel = new JPanel(new BorderLayout(HORIZ, VERT));

        Font font = new Font(null, Font.BOLD, 16);

        TitledBorder title1 = BorderFactory.createTitledBorder("User");
        title1.setTitleFont(font);
        userPanel.setBorder(title1);
        userPanel.setSize(50, 100);

        userList_ = new JList();
        userLabel_ = new JLabel("          Total Users:  0");
        createList(userPanel, userList_, userLabel_, USER);

        TitledBorder title2 = BorderFactory.createTitledBorder("Resource");
        title2.setTitleFont(font);
        resPanel.setBorder(title2);

        resList_ = new JList();
        resLabel_ = new JLabel("          Total Resources:  0");
        createList(resPanel, resList_, resLabel_, RESOURCE);

        panel.add(userPanel);
        panel.add(resPanel);

        // Add panel to a scroll pane
        int vert = ScrollPaneConstants.VERTICAL_SCROLLBAR_AS_NEEDED;
        int horiz = ScrollPaneConstants.HORIZONTAL_SCROLLBAR_AS_NEEDED;
        JScrollPane scroll = new JScrollPane(panel, vert, horiz);

        frame_.getContentPane().add(scroll, orientation);
    }

    /**
     * Creates the list for User or Resource
     * @param panel     Panel to place the list in
     * @param list      a JList object
     * @param label     a JLabel object
     * @param type      Type of list, User or Resource
     * @pre panel != null
     * @pre list != null
     * @pre label != null
     * @post $none
     */
    private void createList(JPanel panel, JList list, JLabel label,
                        final int type)
    {
        Font font = new Font(null, Font.PLAIN, 14);
        label.setFont(font);
        panel.add(label, BorderLayout.NORTH);

        JLabel tempLabel = new JLabel("   ");
        panel.add(tempLabel, BorderLayout.WEST);

        list.setSize(20, 20);  // width and height
        list.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);

        MouseListener mouseListener = new MouseAdapter() 
                {
                    public void mouseClicked(MouseEvent e)
                    {
                        int click = e.getClickCount();
                        Point point = e.getPoint();

                        if (type == USER) {
                            userMouseClicked(click, point);
                        }
                        else {
                            resourceMouseClicked(click, point);
                        }
                    }
                };

        list.addMouseListener(mouseListener);

        int vert = ScrollPaneConstants.VERTICAL_SCROLLBAR_AS_NEEDED;
        int horiz = ScrollPaneConstants.HORIZONTAL_SCROLLBAR_AS_NEEDED;
        JScrollPane listScrollPane = new JScrollPane(list, vert, horiz);
        panel.add(listScrollPane, BorderLayout.CENTER);

        JPanel viewPanel = new JPanel();
        JButton button1 = new JButton("View Property");
        button1.addActionListener(this);
        viewPanel.add(button1);
        panel.add(viewPanel, BorderLayout.EAST);

        JPanel buttonPanel = new JPanel(new FlowLayout(FlowLayout.LEFT,80,10));
        JButton button2 = new JButton("Add");
        button2.addActionListener(this);
        buttonPanel.add(button2);

        JButton button3 = new JButton("Delete");
        button3.addActionListener(this);
        buttonPanel.add(button3);
        panel.add(buttonPanel, BorderLayout.SOUTH);

        String dir = "";
        try {
            // Searching for relative path with image directory in it
            dir = System.getProperty("user.dir") + "/image/";
        }
        catch (Exception e) 
        {
            System.out.println(
                    "VisualModeler : Error - Can't find image files.");
        }

        if (type == USER)
        {
            setButtonDisplay(button1, "  View User", "view_user",
                    dir, "view_user.gif");

            setButtonDisplay(button2, "  Add User", "add_user",
                    dir, "add_user.gif");

            setButtonDisplay(button3, "  Delete User", "del_user",
                    dir, "del_user.gif");
        }
        else
        {
            setButtonDisplay(button1, "  View Resource", "view_res",
                    dir, "view_res.gif");

            setButtonDisplay(button2, "  Add Resource", "add_res",
                    dir, "add_res.gif");

            setButtonDisplay(button3, "  Delete Resource", "del_res",
                    dir, "del_res.gif");
        }
    }

    /**
     * Sets the display of the button
     * @param button     a JButton object
     * @param name       Text to be shown on the button
     * @param cmd        Command to be executed when the button is activated
     * @param dir           the absolute directory path
     * @param iconName   Icon to be displayed on the button
     * @pre button != null
     * @pre name != null
     * @pre cmd != null
     * @pre dir != null
     * @pre iconName != null
     * @post $none
     */
    private void setButtonDisplay(JButton button, String name, String cmd,
                        String dir, String iconName)
    {
        ImageIcon icon = new ImageIcon(dir + iconName);
        
        // If the user decided to run Visual Modeler directly from jar file
        // then look the images inside the file
        if (icon.getIconWidth() == -1)
        {
            try 
            {
                // get the location from jar file
                URL url = super.getClass().getResource("image/" + iconName); 
                Image image = Toolkit.getDefaultToolkit().getImage(url); 
                icon.setImage(image);
            }
            catch (Exception e) 
            {
                System.out.println("Visual Modeler : Error - Can't find " +
                        "image files from visualmodeler.jar");
            }
        }

        button.setActionCommand(cmd);
        button.setText(name);
        button.setIcon(icon);
    }

    /**
     * User list has mouse click
     * @param clickCount   The number of mouse clicks
     * @param point        Point when the mouse click is activated
     * @pre point != null
     * @post $none
     */
    private void userMouseClicked(int clickCount, Point point)
    {
        if (clickCount == 2)
        {
            int index = userList_.locationToIndex(point);
            user_.showProperty(index);
        }
    }

    /**
     * Resource list has mouse click
     * @param clickCount   The number of mouse clicks
     * @param point        Point when the mouse click is activated
     * @pre point != null
     * @post $none
     */
    private void resourceMouseClicked(int clickCount, Point point)
    {
        if (clickCount == 2)
        {
            int index = resList_.locationToIndex(point);
            res_.showProperty(index);
        }
    }

    /**
     * Sets Action to be performed for User Property when event is triggered
     * @param evt      Event
     * @pre evt != null
     * @post $none
     */
    private void userPropertyActionPerformed(ActionEvent evt)
    {
        String cmd = evt.getActionCommand();
        int index = new Integer(cmd).intValue();
        user_.showProperty(index - 1);
    }


    /**
     * Sets action to be performed for Resource Property when event is triggered
     * @param evt      Event
     * @pre evt != null
     * @post $none
     */
    private void resourcePropertyActionPerformed(ActionEvent evt)
    {
        String cmd = evt.getActionCommand();
        int index = new Integer(cmd).intValue();
        res_.showProperty(index - 1);
    }

} // end class

