/*
 * Title:        Visual Modeler for GridSim Toolkit
 * Description:  This Visual Modeler enables the user to quickly create
 *               experiments on different Grid testbeds and generate the
 *               default Grid Broker source codes (in Java).
 *
 * $Id: DefaultFileFilter.java,v 1.3 2003/06/11 06:33:57 anthony Exp $
 */

package visualmodeler;

import java.io.File;
import java.util.Hashtable;
import java.util.Enumeration;
import javax.swing.*;
import javax.swing.filechooser.*;


/**
 * A DefaultFileFilter, once implemented, can be set on a JFileChooser to keep
 * unwanted files from appearing in the directory listing.
 *
 * @author       Anthony Sulistio and Chee Shin Yeo
 * @version      1.1
 * @invariant $none
 */
public class DefaultFileFilter extends FileFilter
{
    private Hashtable filters_ = new Hashtable() ;
    private String desc_ = null;
    private String fullDesc_ = null;

    /**
     * Constructs a default file filter to filter a file extension
     * @param ext   File extension to filter
     * @param desc  Description of this filter, eg. "JPG and GIF Images"
     * @pre ext != null
     * @pre desc != null
     * @post $none
     */
    public DefaultFileFilter(String ext, String desc)
    {
        if (ext != null) {
            addExtension(ext);
        }

        if (desc != null) {
            setDescription(desc);
        }
    }

    /**
     * Overloaded method. 
     * Constructs a default file filter to filter file extensions.
     * @param ext   File extensions to filter
     * @param desc  Description of this filter, eg. "JPG and GIF Images"
     * @pre ext != null
     * @pre desc != null
     * @post $none
     */
    public DefaultFileFilter(String[] ext, String desc)
    {
        if (ext != null) 
        {
            int length = ext.length;
            for (int i = 0; i < length; i++) {
                addExtension(ext[i]);   // add filters one by one
            }
        }

        if (desc != null) {
            setDescription(desc);
        }
    }

    /**
     * Checks whether the given file is accepted by this filter or not
     * @param file  A File object
     * @return <tt>true</tt> if it is accepted, 
     *         <tt>false</tt> otherwise.
     * @pre $none
     * @post $none
     */
    public boolean accept(File file)
    {
        if (file != null)
        {
            if ( file.isDirectory() ) {
                return true;
            }

            String ext = getExtension(file);
            if (ext != null && filters_.get(ext) != null) {
                return true;
            }
        }

        return false;
    }

    /**
     * Gets the file extension of a given File object
     * @param file  A File extension
     * @return file extension or <tt>null</tt> if the given file is empty.
     * @pre $none
     * @post $none
     */
    public String getExtension(File file)
    {
        if (file != null)
        {
            String filename = file.getName();
            if (filename == null) {
                return null;
            }

            int i = filename.lastIndexOf('.');
            if (i > 0 && i < filename.length()-1) {
                return filename.substring(i+1).toLowerCase();
            }
        }

        return null;
    }

    /**
     * Gets the file name without its file extension
     * @param fileName  A filename with file extension
     * @return a filename without file extension
     * @pre $none
     * @post $none
     */
    public String getFileNameOnly(String fileName)
    {
        int i = fileName.lastIndexOf('.');
        if (i > 0 && i < fileName.length()-1) {
            return fileName.substring(0, i);
        }

        return fileName;
    }

    /**
     * Gets the description of this filter
     * @return description
     * @pre $none
     * @post $none
     */
    public String getDescription()
    {
        if (fullDesc_ == null)
        {
            if (desc_ == null)
            {
                StringBuffer buffer = new StringBuffer();
                buffer.append( (desc_ == null) ? "(" : desc_ + " (" );

                // build the description from the extension list
                Enumeration extensions = filters_.keys();
                if (extensions != null)
                {
                    buffer.append( "." + (String) extensions.nextElement() );

                    while (extensions.hasMoreElements())
                    {
                        buffer.append(", .");
                        buffer.append( (String) extensions.nextElement() );
                    }
                }

                buffer.append(")");
                fullDesc_ = buffer.toString();
            }
            else {
                fullDesc_ = desc_;
            }
        }

        return fullDesc_;
    }

    ////////////////////////////////// PRIVATE METHODS //////////////////////

    /**
     * Set the description for this filter
     * @param desc  A filter description
     * @pre $none
     * @post $none
     */
    private void setDescription(String desc)
    {
        desc_ = desc;
        fullDesc_ = null;
    }

    /**
     * Adds the file extension
     * @param extension  A file extension
     * @pre $none
     * @post $none
     */
    private void addExtension(String extension)
    {
        if (extension == null) {
            return;
        }
        
        if (filters_ == null) {
            filters_ = new Hashtable(5);
        }

        filters_.put(extension.toLowerCase(), this);
        fullDesc_ = null;
    }

} // end class

