/*
 * Title:        GridSim Toolkit
 * Description:  GridSim (Grid Simulation) Toolkit for Modeling and Simulation 
 *               of Parallel and Distributed Systems such as Clusters and Grids
 * Licence:      GPL - http://www.gnu.org/copyleft/gpl.html
 *
 * $Id: Stat.java,v 1.7 2003/12/13 15:54:12 anthony Exp $
 */

package gridsim;

import java.util.*;
import java.io.*;
import eduni.simjava.*;

/**
 * A class representing contents of a statistic object
 *
 * @author       Manzur Murshed and Rajkumar Buyya
 * @version      2.2, December 2003
 * @invariant $none
 */
public class Stat
{
    private double time_;
    private String category_;
    private String name_;
    private String data_;


    /**
     * Allocates a new Stat object
     * @param time the time at which Statistic info was recorded. 
     *             Generally this value is taken from <tt>GridSim.Clock()</tt>
     * @param category user-defined name for data type
     * @param name of the entity that want to store this data
     * @param data data to be recorded
     * @see gridsim.GridSim#Clock()
     * @pre time >= 0.0
     * @pre category != null
     * @pre name != null
     * @pre data != null
     * @post $none
     */
    public Stat(double time, String category, String name, String data) 
    {
        this.time_ = time;
        this.category_ = category;
        this.name_ = name;
        this.data_ = data;
    }

    /**
     * Allocates a new Stat object
     * @param time the time at which Statistic info was recorded. 
     *             Generally this value is taken from <tt>GridSim.Clock()</tt>
     * @param category user-defined name for data type
     * @param name of the entity that want to store this data
     * @param data data to be recorded
     * @see gridsim.GridSim#Clock()
     * @pre time >= 0.0
     * @pre category != null
     * @pre name != null
     * @pre data >= 0
     * @post $none
     */
    public Stat(double time, String category, String name, int data) 
    {
        this.time_ = time;
        this.category_ = category;
        this.name_ = name;
        this.data_ = "" + data;
    }

    /**
     * Allocates a new Stat object
     * @param time the time at which Statistic info was recorded. 
     *             Generally this value is taken from <tt>GridSim.Clock()</tt>
     * @param category user-defined name for data type
     * @param name of the entity that want to store this data
     * @param data data to be recorded
     * @see gridsim.GridSim#Clock()
     * @pre time >= 0.0
     * @pre category != null
     * @pre name != null
     * @pre data >= 0.0
     * @post $none
     */
    public Stat(double time, String category, String name, double data) 
    {
        this.time_ = time;
        this.category_ = category;
        this.name_ = name;
        this.data_ = "" + data;
    }

    /**
     * Allocates a new Stat object
     * @param time the time at which Statistic info was recorded. 
     *             Generally this value is taken from <tt>GridSim.Clock()</tt>
     * @param category user-defined name for data type
     * @param name of the entity that want to store this data
     * @param data data to be recorded
     * @see gridsim.GridSim#Clock()
     * @pre time >= 0.0
     * @pre category != null
     * @pre name != null
     * @post $none
     */
    public Stat(double time, String category, String name, boolean data) 
    {
        this.time_ = time;
        this.category_ = category;
        this.name_ = name;
        this.data_ = "" + data;
    }

    /**
     * Gets the time at which Statistic info was recorded
     * @return the time at which Statistic info was recorded
     * @deprecated As of GridSim 2.1, replaced by {@link #getTime()}
     * @pre $none
     * @post $result >= 0.0
     */
    public double get_time() {
        return this.getTime();
    }

    /**
     * Gets the time at which Statistic info was recorded
     * @return the time at which Statistic info was recorded
     * @pre $none
     * @post $result >= 0.0
     */
    public double getTime() {
        return time_;
    }

    /**
     * Gets the user-defined name for data type
     * @return the user-defined name for data type
     * @deprecated As of GridSim 2.1, replaced by {@link #getCategory()}
     * @pre $none
     * @post $result != null
     */
    public String get_category() {
        return this.getCategory();
    }

    /**
     * Gets the user-defined name for data type
     * @return the user-defined name for data type
     * @pre $none
     * @post $result != null
     */
    public String getCategory() {
        return category_;
    }

    /**
     * Gets the name of the entity that want to store this data
     * @return the name of the entity that want to store this data
     * @deprecated As of GridSim 2.1, replaced by {@link #getName()}
     * @pre $none
     * @post $result != null
     */
    public String get_name() {
        return this.getName();
    }

    /**
     * Gets the name of the entity that want to store this data
     * @return the name of the entity that want to store this data
     * @pre $none
     * @post $result != null
     */
    public String getName() {
        return name_;
    }

    /**
     * Gets the the data to be recorded
     * @return the data to be recorded
     * @deprecated As of GridSim 2.1, replaced by {@link #getData()}
     * @pre $none
     * @post $result != null
     */
    public String get_data() {
        return this.getData();
    }

    /**
     * Gets the the data to be recorded
     * @return the data to be recorded
     * @pre $none
     * @post $result != null
     */
    public String getData() {
        return data_;
    }

    /**
     * Gets the the concatenated value of all items as a string
     * @return the concatenated value of all items as a string
     * @pre $none
     * @post $result != null
     */
    public String toString() {
        return "" + time_ + "\t" + category_ + "\t" + name_ + "\t" + data_;
    }
    
} // end class

