/*
 * Title:        GridSim Toolkit
 * Description:  GridSim (Grid Simulation) Toolkit for Modeling and Simulation
 *               of Parallel and Distributed Systems such as Clusters and Grids
 * Licence:      GPL - http://www.gnu.org/copyleft/gpl.html
 *
 * $Id: ResourceCharacteristics.java,v 1.16 2003/12/13 22:49:57 anthony Exp $
 */

package gridsim;

/**
 * GridSim ResourceCharacteristics class represents static properties of a
 * resource such as resource architecture, Operating System (OS), management
 * policy (time- or space-shared), cost and time zone at which the resource
 * is located along resource configuration.
 *
 * @author       Manzur Murshed and Rajkumar Buyya
 * @version      2.2, December 2003
 * @invariant $none
 */
public class ResourceCharacteristics
{
    private int id_;     // resource id--setup when Resource is created
    private String architecture_;
    private String OS_;
    private MachineList machineList_;
    private double timeZone_;    // difference from GMT

    // Price/CPU-unit if unit = sec., then G$/CPU-sec.
    private double costPerSec_;

    // Resource Types -- allocation policy
    private int allocationPolicy_;

    /** Time-shared system */
    public static final int TIME_SHARED = 0;

    /** Spaced-shared system */
    public static final int SPACE_SHARED = 1;

    /** Assuming all PEs in all given Machines have the same rating.
     * the difference will be in getMIPSRating()
     */
    public static final int OTHER_POLICY_SAME_RATING = 2;

    /** Assuming all PEs in a given Machine have the same rating.
     * But different Machines in a Cluster can have different rating.
     * the difference will be in getMIPSRating()
     */
    public static final int OTHER_POLICY_DIFFERENT_RATING = 3;

    /** Advanced-reservation system <b>(NOT YET IMPLEMENTED)</b> */
    public static final int ADVANCE_RESERVATION = 4;

    /**
     * Allocates a new ResourceCharacteristics object
     * @param architecture  the architecture of a resource
     * @param OS            the operating system used
     * @param machineList   list of machines in a resource
     * @param allocationPolicy     the resource allocation policy
     * @param timeZone      the time zone this resource is located
     * @param costPerSec    the cost per sec to use this resource
     * @pre architecture != null
     * @pre OS != null
     * @pre machineList != null
     * @pre allocationPolicy >= 0 && allocationPolicy <= 3
     * @pre costPerSec >= 0.0
     * @post $none
     */
    public ResourceCharacteristics(String architecture, String OS,
                MachineList machineList, int allocationPolicy,
                double timeZone, double costPerSec)
    {
        this.id_ = -1;
        this.architecture_ = architecture;
        this.OS_ = OS;
        this.machineList_ = machineList;
        this.allocationPolicy_ = allocationPolicy;
        this.timeZone_ = timeZone;
        this.costPerSec_ = costPerSec;
    }

    /**
     * Sets the resource ID
     * @param id    the resource ID
     * @deprecated As of GridSim 2.1, replaced by {@link #setResourceID(int)}
     * @pre id >= 0
     * @post $none
     */
    public void SetID(int id) {
        this.setResourceID(id);
    }

    /**
     * Sets the resource ID
     * @param id    the resource ID
     * @pre id >= 0
     * @post $none
     */
    public void setResourceID(int id) {
        this.id_ = id;
    }

    /**
     * Gets the resource ID
     * @return the resource ID
     * @deprecated As of GridSim 2.1, replaced by {@link #getResourceID()}
     * @pre $none
     * @post $result >= 0
     */
    public int GetID() {
        return this.getResourceID();
    }

    /**
     * Gets the resource ID
     * @return the resource ID
     * @pre $none
     * @post $result >= 0
     */
    public int getResourceID() {
        return id_;
    }

    /**
     * Gets the name of a resource
     * @return the resource name
     * @deprecated As of GridSim 2.1, replaced by
     *             {@link #getResourceName()}
     * @pre $none
     * @post $result != null
     */
    public String GetName() {
        return this.getResourceName();
    }

    /**
     * Gets the name of a resource
     * @return the resource name
     * @pre $none
     * @post $result != null
     */
    public String getResourceName() {
        return GridSim.getEntityName(id_);
    }

    /**
     * Gets the resource architecture name
     * @return the architecture name
     * @deprecated As of GridSim 2.1, replaced by
     *             {@link #getResourceArch()}
     * @pre $none
     * @post $result != null
     */
    public String GetArch() {
        return this.getResourceArch();
    }

    /**
     * Gets the resource architecture name
     * @return the architecture name
     * @pre $none
     * @post $result != null
     */
    public String getResourceArch() {
        return architecture_;
    }

    /**
     * Gets the Operating System (OS) this resource is used
     * @return the name of OS
     * @deprecated As of GridSim 2.1, replaced by {@link #getResourceOS()}
     * @pre $none
     * @post $result != null
     */
    public String GetOS() {
        return this.getResourceOS();
    }

    /**
     * Gets the Operating System (OS) this resource is used
     * @return the name of OS
     * @pre $none
     * @post $result != null
     */
    public String getResourceOS() {
        return OS_;
    }

    /**
     * Gets the list of machines in a resouce
     * @return a MachineList object
     * @see gridsim.MachineList
     * @deprecated As of GridSim 2.1, replaced by {@link #getMachineList()}
     * @pre $none
     * @post $result != null
     */
    public MachineList GetMachines() {
        return this.getMachineList();
    }

    /**
     * Gets the list of machines in a resouce
     * @return a MachineList object
     * @see gridsim.MachineList
     * @pre $none
     * @post $result != null
     */
    public MachineList getMachineList() {
        return machineList_;
    }

    /**
     * Gets a Machine with at least one empty PE
     * @return a Machine object or <tt>null</tt> if not found
     * @pre $none
     * @post $none
     */
    public Machine getMachineWithFreePE() {
        return machineList_.getMachineWithFreePE();
    }

    /**
     * Gets the resource allocation policy
     * @return the allocation policy
     * @deprecated As of GridSim 2.1, replaced by
     *             {@link #getResourceAllocationPolicy()}
     * @pre $none
     * @post $result >= 0 && $result <= 2
     */
    public int GetAllocationPolicy() {
        return this.getResourceAllocationPolicy();
    }

    /**
     * Gets the resource allocation policy
     * @return the allocation policy
     * @pre $none
     * @post $result >= 0 && $result <= 2
     */
    public int getResourceAllocationPolicy() {
        return allocationPolicy_;
    }

    /**
     * Gets the resource time zone
     * @return the time zone
     * @deprecated As of GridSim 2.1, replaced by
     *             {@link #getResourceTimeZone()}
     * @pre $none
     * @post $none
     */
    public double GetTimeZone() {
        return this.getResourceTimeZone();
    }

    /**
     * Gets the resource time zone
     * @return the time zone
     * @pre $none
     * @post $none
     */
    public double getResourceTimeZone() {
        return timeZone_;
    }

    /**
     * Gets Millions Instructions Per Second (MIPS) Rating of a Processing
     * Element (PE). It is assumed all PEs' rating is same in a given machine.
     * @return the MIPS Rating or <tt>-1</tt> if no PEs are exists.
     * @deprecated As of GridSim 2.1, replaced by
     *             {@link #getMIPSRatingOfOnePE()}
     * @pre $none
     * @post $result >= -1
     */
    public int GetMIPSRatingOfOnePE() {
        return this.getMIPSRatingOfOnePE();
    }

    /**
     * Gets Millions Instructions Per Second (MIPS) Rating of a Processing
     * Element (PE). It is assumed all PEs' rating is same in a given machine.
     * @return the MIPS Rating or <tt>-1</tt> if no PEs are exists.
     * @pre $none
     * @post $result >= -1
     */
    public int getMIPSRatingOfOnePE()
    {
        if (machineList_.size() == 0) {
            return -1;
        }

        return machineList_.getMachine(0).getPEList().getMIPSRating(0);
    }

    /**
     * Gets Millions Instructions Per Second (MIPS) Rating of a Processing
     * Element (PE).
     * It is essential to use this method when a resource is made up
     * of heterogenous PEs/machines.
     * @param id    the machine ID
     * @param peID  the PE ID
     * @return the MIPS Rating or <tt>-1</tt> if no PEs are exists.
     * @deprecated As of GridSim 2.1, replaced by
     *             {@link #getMIPSRatingOfOnePE(int, int)}
     * @pre id >= 0
     * @pre peID >= 0
     * @post $result >= -1
     */
    public int GetMIPSRatingOfOnePE(int id, int peID) {
        return this.getMIPSRatingOfOnePE(id, peID);
    }

    /**
     * Gets Millions Instructions Per Second (MIPS) Rating of a Processing
     * Element (PE).
     * It is essential to use this method when a resource is made up
     * of heterogenous PEs/machines.
     * @param id        the machine ID
     * @param peID      the PE ID
     * @return the MIPS Rating or <tt>-1</tt> if no PEs are exists.
     * @pre id >= 0
     * @pre peID >= 0
     * @post $result >= -1
     */
    public int getMIPSRatingOfOnePE(int id, int peID)
    {
        if (machineList_.size() == 0) {
            return -1;
        }

        return machineList_.getMachine(id).getPEList().getMIPSRating(peID);
    }

    /**
     * Gets the total MIPS rating, which is the sum of MIPS rating of all
     * machines in a resource
     * @return the sum of MIPS ratings
     * @deprecated As of GridSim 2.1, replaced by {@link #getMIPSRating()}
     * @pre $none
     * @post $result >= 0
     */
    public int GetMIPSRating() {
        return this.getMIPSRating();
    }

    /**
     * Gets the total MIPS rating, which is the sum of MIPS rating of all
     * machines in a resource.
     * For TimeShared scheduling = 1 Rating of a PE * Total number of PEs
     * For SpaceShared schedulin = different !!!
     * @return the sum of MIPS ratings
     * @pre $none
     * @post $result >= 0
     */
    public int getMIPSRating()
    {
        int rating = 0;
        switch (allocationPolicy_)
        {
            // Assuming all PEs in all Machine have same rating.
            case ResourceCharacteristics.TIME_SHARED:
            case ResourceCharacteristics.OTHER_POLICY_SAME_RATING:
                rating = getMIPSRatingOfOnePE() * machineList_.getNumPE();
                break;

            // Assuming all PEs in a given Machine have the same rating.
            // But different machines in a Cluster can have different rating
            case ResourceCharacteristics.SPACE_SHARED:
            //case ResourceCharacteristics.ADVANCE_RESERVATION:
            case ResourceCharacteristics.OTHER_POLICY_DIFFERENT_RATING:
                for (int i = 0; i < machineList_.size(); i++) {
                    rating += ((Machine) machineList_.get(i)).getMIPSRating();
                }
                break;

            default:
                break;
        }

        return rating;
    }

    /**
     * Gets the CPU time given the specified parameters
     * @param gridletLength     the length of a Gridlet
     * @param load              the load of a Gridlet
     * @return the CPU time
     * @deprecated As of GridSim 2.1, replaced by
     *             {@link #getCPUTime(double, double)}
     * @pre gridletLength >= 0.0
     * @pre load >= 0.0
     * @post $result >= 0.0
     */
    public double CPU_time(double gridletLength, double load) {
        return this.getCPUTime(gridletLength, load);
    }

    /**
     * Gets the CPU time given the specified parameters (only for TIME_SHARED).
     * <tt>NOTE:</tt> The CPU time for SPACE_SHARED and ADVANCE_RESERVATION
     *                are not yet implemented.
     * @param gridletLength     the length of a Gridlet
     * @param load              the load of a Gridlet
     * @return the CPU time
     * @pre gridletLength >= 0.0
     * @pre load >= 0.0
     * @post $result >= 0.0
     */
    public double getCPUTime(double gridletLength, double load)
    {
        double cpuTime = 0.0;

        switch (allocationPolicy_)
        {
            case ResourceCharacteristics.TIME_SHARED:
                cpuTime = gridletLength / ( getMIPSRatingOfOnePE()*(1.0-load) );
                break;

            default:
                break;
        }

        return cpuTime;
    }

    /**
     * Gets the total number of PEs for all Machines
     * @return number of PEs
     * @deprecated As of GridSim 2.1, replaced by {@link #getNumPE()}
     * @pre $none
     * @post $result >= 0
     */
    public int GetNoOfPEs() {
        return this.getNumPE();
    }

    /**
     * Gets the total number of PEs for all Machines
     * @return number of PEs
     * @pre $none
     * @post $result >= 0
     */
    public int getNumPE() {
        return machineList_.getNumPE();
    }

    /**
     * Gets the total number of <tt>FREE</tt> or non-busy PEs for all Machines
     * @return number of PEs
     * @pre $none
     * @post $result >= 0
     */
    public int getNumFreePE() {
        return machineList_.getNumFreePE();
    }

    /**
     * Gets the total number of <tt>BUSY</tt> PEs for all Machines
     * @return number of PEs
     * @pre $none
     * @post $result >= 0
     */
    public int getNumBusyPE() {
        return machineList_.getNumBusyPE();
    }

    /**
     * Sets the particular PE status on a Machine
     * @param status   PE status, either <tt>PE.FREE</tt> or <tt>PE.BUSY</tt>
     * @param machineID    Machine ID
     * @param peID     PE id
     * @return <tt>true</tt> if the PE status has changed, <tt>false</tt>
     * otherwise (Machine id or PE id might not be exist)
     * @pre machineID >= 0
     * @pre peID >= 0
     * @post $none
     */
    public boolean setStatusPE(boolean status, int machineID, int peID) {
        return machineList_.setStatusPE(status, machineID, peID);
    }

    /**
     * Sets the cost per second associated with a resource
     * @param costPerSec   the cost using a resource
     * @deprecated As of GridSim 2.1, replaced by {@link #setCostPerSec(double)}
     * @pre costPerSec >= 0.0
     * @post $none
     */
    public void SetCostPerSec(double costPerSec) {
        this.setCostPerSec(costPerSec);
    }

    /**
     * Sets the cost per second associated with a resource
     * @param costPerSec   the cost using a resource
     * @pre costPerSec >= 0.0
     * @post $none
     */
    public void setCostPerSec(double costPerSec) {
        this.costPerSec_ = costPerSec;
    }

    /**
     * Gets the cost per second associated with a resource
     * @return the cost using a resource
     * @deprecated As of GridSim 2.1, replaced by {@link #getCostPerSec()}
     * @pre $none
     * @post $result >= 0.0
     */
    public double GetCostPerSec() {
        return this.getCostPerSec();
    }

    /**
     * Gets the cost per second associated with a resource
     * @return the cost using a resource
     * @pre $none
     * @post $result >= 0.0
     */
    public double getCostPerSec() {
        return costPerSec_;
    }

    /**
     * Gets the cost per Millions Instruction (MI) associated with a resource
     * @return the cost using a resource
     * @deprecated As of GridSim 2.1, replaced by {@link #getCostPerMI()}
     * @pre $none
     * @post $result >= 0.0
     */
    public double GetCostPerMI() {
        return this.getCostPerMI();
    }

    /**
     * Gets the cost per Millions Instruction (MI) associated with a resource
     * @return the cost using a resource
     * @pre $none
     * @post $result >= 0.0
     */
    public double getCostPerMI() {
        return costPerSec_ / getMIPSRatingOfOnePE();
    }

    /**
     * Gets the byte size of this class
     * @return the byte size
     * @deprecated As of GridSim 2.1, replaced by {@link #getByteSize()}
     * @pre $none
     * @post $result > 0
     */
    public int ByteSize() {
        return this.getByteSize();
    }

    /**
     * Gets the byte size of this class
     * @return the byte size
     * @pre $none
     * @post $result > 0
     */
    public int getByteSize()
    {
        // this class overall has: 2 ints, 2 Strings, 1 MachineList and
        //                         2 doubles.
        // NOTE: static attributes do not count
        int totalInt = 2 * 4;
        int totalDouble = 2 * 8;
        int totalSize = architecture_.length() + OS_.length() +
                   machineList_.getByteSize() + totalInt + totalDouble ;

        return totalSize;
    }

} // end class

