/*
 * Title:        GridSim Toolkit
 * Description:  GridSim (Grid Simulation) Toolkit for Modeling and Simulation
 *               of Parallel and Distributed Systems such as Clusters and Grids
 * Licence:      GPL - http://www.gnu.org/copyleft/gpl.html
 *
 * $Id: ResGridlet.java,v 1.13 2003/12/13 22:12:07 anthony Exp $
 */

package gridsim;

import java.util.Calendar;


/**
 * GridSim ResGridlet represents a Gridlet submitted to GridResource for
 * processing. This class keeps track the time for all activities in the
 * GridResource for a specific Gridlet. Before a Gridlet exits the
 * GridResource, it is RECOMMENDED to call this method
 * {@link #finalizeGridlet()}.
 * <p>
 * It contains a Gridlet object along with its arrival time and
 * the ID of the machine and the PE (Processing Element) allocated to it.
 * It acts as a placeholder for maintaining the amount of resource share
 * allocated at various times for simulating any
 * scheduling using internal events.
 *
 * @author       Manzur Murshed and Rajkumar Buyya
 * @version      2.2, December 2003
 * @invariant $none
 */
public class ResGridlet
{
    private Gridlet gridlet_;       // a Gridlet object
    private double arrivalTime_;    // Gridlet arrival time for the first time
    private double finishedTime_;   // estimation of Gridlet finished time
    private double gridletFinishedSoFar_;  // length of Gridlet finished so far

    // Gridlet execution start time. This attribute will only hold the latest
    // time since a Gridlet can be cancel, paused or resumed.
    private double startExecTime_;
    private double totalCompletionTime_;  // total time to complete this Gridlet
    private int gridletID_;   // this Gridlet id
    private int userID_;      // the Gridlet's user id

    // The below attributes are only be used by the SpaceShared policy
    private int machineID_;   // machine id this Gridlet is assigned to
    private int peID_;        // PE id this Gridlet is assigned to

    /**
     * Allocates a new ResGridlet object upon the arrival of a Gridlet object.
     * The arriving time is determined by GridSim.Clock()
     * @param gridlet a gridlet object
     * @see GridSim#Clock()
     * @pre gridlet != null
     * @post $none
     */
    public ResGridlet(Gridlet gridlet)
    {
        // when a new ResGridlet is created, then it will automatically set
        // the submission time and other properties, such as remaining length
        this.gridlet_ = gridlet;
        this.arrivalTime_ = GridSim.clock();
        this.gridlet_.setSubmissionTime(arrivalTime_);

        // default values
        this.finishedTime_ = -1;  // Cannot finish in this hourly slot.
        this.machineID_ = -1;
        this.peID_ = -1;
        this.startExecTime_ = 0.0;
        this.totalCompletionTime_ = 0.0;

        // set the IDs of this Gridlet
        this.gridletID_ = gridlet_.getGridletID();
        this.userID_ = gridlet_.getUserID();

        // In case a Gridlet has been executed partially by some other grid
        // resources.
        this.gridletFinishedSoFar_ = gridlet_.getGridletFinishedSoFar();
    }

    /**
     * Gets this Gridlet entity Id
     * @return the Gridlet entity Id
     * @pre $none
     * @post $none
     */
    public int getGridletID() {
        return gridletID_;
    }

    /**
     * Gets the user or owner of this Gridlet
     * @return the Gridlet's user Id
     * @pre $none
     * @post $none
     */
    public int getUserID() {
        return userID_;
    }

    /**
     * Gets the Gridlet's length
     * @return Gridlet's length
     * @pre $none
     * @post $none
     */
    public double getGridletLength() {
        return gridlet_.getGridletLength();
    }

    /**
     * Sets the Gridlet status.
     * @param status  the Gridlet status
     * @return <tt>true</tt> if the new status has been set, <tt>false</tt>
     *         otherwise
     * @pre status >= 0
     * @post $none
     */
    public boolean setGridletStatus(int status)
    {
        // gets Gridlet's previous status
        int prevStatus = gridlet_.getGridletStatus();

        // if the status of a Gridlet is the same as last time, then ignore
        if (prevStatus == status) {
            return false;
        }

        boolean success = true;
        try
        {
            double clock = GridSim.clock();   // gets the current clock

            // sets Gridlet's current status
            gridlet_.setGridletStatus(status);

            // if a previous Gridlet status is INEXEC
            if (prevStatus == Gridlet.INEXEC)
            {
                // and current status is either CANCELED, PAUSED or SUCCESS
                if (status == Gridlet.CANCELED || status == Gridlet.PAUSED ||
                    status == Gridlet.SUCCESS)
                {
                    // then update the Gridlet completion time
                    totalCompletionTime_ += (clock - startExecTime_);
                    return true;
                }
            }

            if (prevStatus == Gridlet.RESUMED && status == Gridlet.SUCCESS)
            {
                // then update the Gridlet completion time
                totalCompletionTime_ += (clock - startExecTime_);
                return true;
            }

            // if a Gridlet is now in execution
            if (status == Gridlet.INEXEC ||
                (prevStatus == Gridlet.PAUSED && status == Gridlet.RESUMED) )
            {
                startExecTime_ = clock;
                gridlet_.setExecStartTime(startExecTime_);
            }
        }
        catch(Exception e) {
            success = false;
        }

        return success;
    }

    /**
     * Gets the Gridlet's execution start time
     * @return Gridlet's execution start time
     * @pre $none
     * @post $none
     */
    public double getExecStartTime() {
        return gridlet_.getExecStartTime();
    }

    /**
     * Sets this Gridlet's execution parameters. These parameters are set by
     * the GridResource before departure or sending back to the original
     * Gridlet's owner.
     *
     * @param wallClockTime    the time of this Gridlet resides in
     *                         a GridResource (from arrival time until
     *                         departure time).
     * @param actualCPUTime    the total execution time of this Gridlet in a
     *                         GridResource.
     * @pre wallClockTime >= 0.0
     * @pre actualCPUTime >= 0.0
     * @post $none
     */
    public void setExecParam(double wallClockTime, double actualCPUTime) {
        gridlet_.setExecParam(wallClockTime, actualCPUTime);
    }

    /**
     * Set the machine and PE (Processing Element) ID
     * @param machineID   machine ID
     * @param peID        PE ID
     * @deprecated As of GridSim 2.1, replaced by
     *             {@link #setMachineAndPEID(int, int)}
     * @pre machineID >= 0
     * @pre peID >= 0
     * @post $none
     */
    public void SetIDs(int machineID, int peID) {
        this.setMachineAndPEID(machineID, peID);
    }

    /**
     * Set the machine and PE (Processing Element) ID
     * @param machineID   machine ID
     * @param peID        PE ID
     * @pre machineID >= 0
     * @pre peID >= 0
     * @post $none
     */
    public void setMachineAndPEID(int machineID, int peID)
    {
        this.machineID_ = machineID;
        this.peID_ = peID;
    }

    /**
     * Get machine ID
     * @return machine ID or <tt>-1</tt> if it is not specified before
     * @deprecated As of GridSim 2.1, replaced by {@link #getMachineID()}
     * @pre $none
     * @post $result >= -1
     */
    public int GetMachineID() {
        return this.getMachineID();
    }

    /**
     * Get machine ID
     * @return machine ID or <tt>-1</tt> if it is not specified before
     * @pre $none
     * @post $result >= -1
     */
    public int getMachineID() {
        return machineID_;
    }

    /**
     * Get PE ID
     * @return PE ID or <tt>-1</tt> if it is not specified before
     * @deprecated As of GridSim 2.1, replaced by {@link #getPEID()}
     * @pre $none
     * @post $result >= -1
     */
    public int GetPEID() {
        return this.getPEID();
    }

    /**
     * Get PE ID
     * @return PE ID or <tt>-1</tt> if it is not specified before
     * @pre $none
     * @post $result >= -1
     */
    public int getPEID() {
        return peID_;
    }

    /**
     * Get the remaining gridlet length
     * @return gridlet length
     * @deprecated As of GridSim 2.1, replaced by
     *             {@link #getRemainingGridletLength()}
     * @pre $none
     * @post $result >= 0
     */
    public double GetRemainingLength() {
        return this.getRemainingGridletLength();
    }

    /**
     * Get the remaining gridlet length
     * @return gridlet length
     * @pre $none
     * @post $result >= 0
     */
    public double getRemainingGridletLength()
    {
        double length = gridlet_.getGridletLength() - gridletFinishedSoFar_;

        // Remaining Gridlet length can't be negative number. This can be
        // happening when this.updateGridletFinishedSoFar() keep calling.
        if (length < 0.0) {
            length = 0.0;
        }

        return length;
    }

    /**
     * Finalizes all relevant information before <tt>exiting</tt> the
     * GridResource entity. This method sets the final data of:
     * <ul>
     *     <li> wall clock time, i.e. the time of this Gridlet resides in
     *          a GridResource (from arrival time until departure time).
     *     <li> actual CPU time, i.e. the total execution time of this
     *          Gridlet in a GridResource.
     *     <li> Gridlet's finished so far
     * </ul>
     * @pre $none
     * @post $none
     */
    public void finalizeGridlet()
    {
        // Sets the wall clock time and actual CPU time
        double wallClockTime = GridSim.clock() - arrivalTime_;
        gridlet_.setExecParam(wallClockTime, totalCompletionTime_);

        double finished = 0.0;
        if (gridlet_.getGridletLength() < gridletFinishedSoFar_) {
            finished = gridlet_.getGridletLength();
        }
        else {
            finished = gridletFinishedSoFar_;
        }

        gridlet_.setGridletFinishedSoFar(finished);
    }

    /**
     * A method that updates the length of gridlet that has been completed
     * @param miLength gridlet length in Million Instructions (MI)
     * @deprecated As of GridSim 2.1, replaced by
     *             {@link #updateGridletFinishedSoFar(double)}
     * @pre miLength >= 0.0
     * @post $none
     */
    public void UpdateGridletFinishedSoFar(double miLength) {
        this.updateGridletFinishedSoFar(miLength);
    }

    /**
     * A method that updates the length of gridlet that has been completed
     * @param miLength gridlet length in Million Instructions (MI)
     * @pre miLength >= 0.0
     * @post $none
     */
    public void updateGridletFinishedSoFar(double miLength) {
        gridletFinishedSoFar_ += miLength;
    }

    /**
     * Get arrival time of a gridlet
     * @return arrival time
     * @deprecated As of GridSim 2.1, replaced by
     *             {@link #getGridletArrivalTime()}
     * @pre $none
     * @post $result >= 0.0
     */
    public double GetArrivalTime() {
        return this.getGridletArrivalTime();
    }

    /**
     * Get arrival time of a gridlet
     * @return arrival time
     * @pre $none
     * @post $result >= 0.0
     */
    public double getGridletArrivalTime() {
        return arrivalTime_;
    }

    /**
     * Determine Time by which this Gridlet can finish.
     * <br>
     * <b>NOTE:</b> To be used by Time shared resources only.
     * @param availableRating  the available MIPS considering current hour
     *                         load
     * @param rCalendar        Resourcce Calendar
     * @throws NullPointerException  if ResourceCalendar object is <tt>null</tt>
     * @deprecated As of GridSim 2.2, replaced by
     *             {@link #setFinishTime(double)}. This method is
     *             <b>OBSOLETE</b> since it is the responsibility of individual
     *             scheduling to come up with the finish time for each Gridlet.
     * @deprecated As of GridSim 2.1, replaced by
     *             {@link #setFinishTimeForTimeSharedResource(double,
     *             ResourceCalendar)}
     * @pre availableRating > 0.0
     * @pre rCalendar != null
     * @post $none
     */
    public void SetFinishTimeForTimeSharedResource(double availableRating,
                    ResourceCalendar rCalendar) throws NullPointerException
    {
        this.setFinishTimeForTimeSharedResource(availableRating, rCalendar);
    }

    /**
     * Determine Time by which this Gridlet can finish.
     * <br>
     * <b>NOTE:</b> To be used by Time shared resources only.
     * @param availableRating  the available MIPS considering current hour
     *                         load
     * @param rCalendar        Resourcce Calendar
     * @throws NullPointerException  if ResourceCalendar object is <tt>null</tt>
     * @deprecated As of GridSim 2.2, replaced by
     *             {@link #setFinishTime(double)}. This method is
     *             <b>OBSOLETE</b> since it is the responsibility of individual
     *             scheduling to come up with the finish time for each Gridlet.
     * @pre availableRating > 0.0
     * @pre rCalendar != null
     * @post $none
     */
    public void setFinishTimeForTimeSharedResource(double availableRating,
                    ResourceCalendar rCalendar) throws NullPointerException
    {
        // ... obsolete method
    }

    /**
     * Determine Time by which this Gridlet can finish.
     * <br>
     * <b>NOTE:</b> To be used by Space shared resources only.
     * @param availableRating  this is 100% availability of PE MIPS since
     *                         it runs in dedicated mode
     * @deprecated As of GridSim 2.2, replaced by
     *             {@link #setFinishTime(double)}. This method is
     *             <b>OBSOLETE</b> since it is the responsibility of individual
     *             scheduling to come up with the finish time for each Gridlet.
     * @deprecated As of GridSim 2.1, replaced by
     *             {@link #setFinishTimeForSpaceSharedResource(double)}
     * @pre availableRating > 0.0
     * @post $none
     */
    public void SetFinishTimeForSpaceSharedResource(double availableRating)
    {
        this.setFinishTimeForSpaceSharedResource(availableRating);
    }

    /**
     * Determine Time by which this Gridlet can finish.
     * <br>
     * <b>NOTE:</b> To be used by Space shared resources only.
     * @param availableRating  this is 100% availability of PE MIPS since
     *                         it runs in dedicated mode
     * @deprecated As of GridSim 2.2, replaced by
     *             {@link #setFinishTime(double)}. This method is
     *             <b>OBSOLETE</b> since it is the responsibility of individual
     *             scheduling to come up with the finish time for each Gridlet.
     * @pre availableRating > 0.0
     * @post $none
     */
    public void setFinishTimeForSpaceSharedResource(double availableRating)
    {
        // ... obsolete method
    }

    /**
     * Sets the finish time for this Gridlet. If time is negative, then it is
     * being ignored.
     * @param time   finish time
     * @pre time >= 0.0
     * @post $none
     */
    public void setFinishTime(double time)
    {
        if (time < 0.0) {
            return;
        }

        finishedTime_ = time;
    }

    /**
     * Get the Gridlet's finish time
     * @return finish time of a gridlet or <tt>-1.0</tt> if
     *         it cannot finish in this hourly slot
     * @deprecated As of GridSim 2.1, replaced by
     *             {@link #getGridletFinishTime()}
     * @pre $none
     * @post $result >= -1.0
     */
    public double GetFinishTime() {
        return this.getGridletFinishTime();
    }

    /**
     * Get the Gridlet's finish time
     * @return finish time of a gridlet or <tt>-1.0</tt> if
     *         it cannot finish in this hourly slot
     * @pre $none
     * @post $result >= -1.0
     */
    public double getGridletFinishTime() {
        return finishedTime_;
    }

    /**
     * Get this Gridlet object
     * @return gridlet object
     * @deprecated As of GridSim 2.1, replaced by {@link #getGridlet()}
     * @pre $none
     * @post $result != null
     */
    public Gridlet GetGridlet() {
        return this.getGridlet();
    }

    /**
     * Get this Gridlet object
     * @return gridlet object
     * @pre $none
     * @post $result != null
     */
    public Gridlet getGridlet() {
        return gridlet_;
    }

    /**
     * Gets the Gridlet status
     * @return Gridlet status
     * @pre $none
     * @post $none
     */
    public int getGridletStatus() {
        return gridlet_.getGridletStatus();
    }

} // end class

