/*
 * Title:        GridSim Toolkit
 * Description:  GridSim (Grid Simulation) Toolkit for Modeling and Simulation
 *               of Parallel and Distributed Systems such as Clusters and Grids
 * Licence:      GPL - http://www.gnu.org/copyleft/gpl.html
 *
 * $Id: PEList.java,v 1.9 2003/12/13 16:09:13 anthony Exp $
 */

package gridsim;

import java.util.LinkedList;
import java.util.Iterator;

/**
 * GridSim PEList maintains a list of PEs (Processing Elements) that make up
 * a machine.
 *
 * @author       Manzur Murshed and Rajkumar Buyya
 * @version      2.2, December 2003
 * @invariant $none
 */
public class PEList extends LinkedList
{
    /**
     * Allocates a new PEList object
     * @pre $none
     * @post $none
     */
    public PEList() {
        super();
    }

    /**
     * Gets MIPS Rating for a specified PE ID
     * @param id    the PE ID
     * @return the MIPS rating if exists, otherwise returns -1
     * @deprecated As of GridSim 2.1, replaced by {@link #getMIPSRating(int)}
     * @pre id >= 0
     * @post $none
     */
    public int GetMIPSRating(int id) {
        return this.getMIPSRating(id);
    }

    /**
     * Gets MIPS Rating for a specified PE ID
     * @param id    the PE ID
     * @return the MIPS rating if exists, otherwise returns -1
     * @pre id >= 0
     * @post $none
     */
    public int getMIPSRating(int id)
    {
        PE obj = null;
        Iterator it = super.iterator();
        while ( it.hasNext() )
        {
            obj = (PE) it.next();
            if (obj.getID() == id) {
                return obj.getMIPSRating();
            }
        }

        return -1;  // no PE with given id
    }

    /**
     * Gets a PE ID which is FREE
     * @return a PE ID if it is FREE, otherwise returns -1
     * @deprecated As of GridSim 2.1, replaced by {@link #getFreePEID()}
     * @pre $none
     * @post $none
     */
    public int GetFreePEID() {
        return this.getFreePEID();
    }

    /**
     * Gets a PE ID which is FREE
     * @return a PE ID if it is FREE, otherwise returns -1
     * @pre $none
     * @post $none
     */
    public int getFreePEID()
    {
        PE obj = null;
        Iterator it = super.iterator();
        while ( it.hasNext() )
        {
            obj = (PE) it.next();
            if (obj.getStatus() == PE.FREE) {
                return obj.getID();
            }
        }

        return -1;
    }

    /**
     * Gets the number of <tt>FREE</tt> or non-busy PE.
     * @return number of PE
     * @pre $none
     * @post $result >= 0
     */
    public int getNumFreePE()
    {
        int counter = 0;
        PE obj = null;

        // a loop that counts the number of free PE
        Iterator it = super.iterator();
        while ( it.hasNext() )
        {
            obj = (PE) it.next();
            if (obj.getStatus() == PE.FREE) {
                counter++;
            }
        }

        return counter;
    }

    /**
     * Sets the PE status
     * @param status   PE status, either <tt>PE.FREE</tt> or <tt>PE.BUSY</tt>
     * @param peID     PE id
     * @return <tt>true</tt> if the PE status has changed, <tt>false</tt>
     * otherwise (PE id might not be exist)
     * @pre peID >= 0
     * @post $none
     */
    public boolean setStatusPE(boolean status, int peID)
    {
        boolean found = false;
        PE obj = null;

        // a loop that counts the number of free PE
        Iterator it = super.iterator();
        while ( it.hasNext() )
        {
            obj = (PE) it.next();
            if (obj.getID() == peID)
            {
                obj.setStatus(status);
                found = true;
                break;
            }
        }

        return found;
    }

    /**
     * Gets the number of <tt>BUSY</tt> PE
     * @return number of PE
     * @pre $none
     * @post $result >= 0
     */
    public int getNumBusyPE(){
        return super.size() - getNumFreePE();
    }

    /**
     * Gets the byte size of PEList internal data members
     * @return the byte size
     * @deprecated As of GridSim 2.1, replaced by {@link #getByteSize()}
     * @pre $none
     * @post $result >= 0
     */
    public int ByteSize() {
        return this.getByteSize();
    }

    /**
     * Gets the byte size of PEList internal data members
     * @return the byte size
     * @pre $none
     * @post $result >= 0
     */
    public int getByteSize() {
        return super.size() * PE.getByteSize();
    }

} // end class

