/*
 * Title:        GridSim Toolkit
 * Description:  GridSim (Grid Simulation) Toolkit for Modeling and Simulation
 *               of Parallel and Distributed Systems such as Clusters and Grids
 * Licence:      GPL - http://www.gnu.org/copyleft/gpl.html
 *
 * $Id: Output.java,v 1.9 2003/12/13 22:41:53 anthony Exp $
 */

package gridsim;

import eduni.simjava.*;

/**
 * GridSim Output defines a port through which a simulation entity sends
 * data to the simulated network.
 * <p>
 * It maintains an event queue to serialize
 * the data-out-flow and delivers to the destination entity.
 * It works along with Input entity to simulate network
 * communication delay. Simultaneous outputs can be modeled by using multiple
 * instances of this class
 *
 * @author       Manzur Murshed and Rajkumar Buyya
 * @version      2.2, December 2003
 * @invariant $none
 */
public class Output extends Sim_entity
{
    private Sim_port outPort_;
    private double baudRate_;

    /**
     * Allocates a new Output object
     * @param name         the name of this object
     * @param baudRate     the communication speed
     * @throws NullPointerException This happens when creating this entity
     *                  before initializing GridSim package or this entity name
     *                  is <tt>null</tt> or empty
     * @pre name != null
     * @pre baudRate >= 0.0
     * @post $none
     */
    public Output(String name, double baudRate) throws NullPointerException
    {
        super(name);
        this.baudRate_ = baudRate;
        outPort_ = new Sim_port("output_buffer");
        super.add_port(outPort_);
    }

    /**
     * Gets the baud rate
     * @return the baud rate
     * @deprecated As of GridSim 2.1, replaced by {@link #getBaudRate()}
     * @pre $none
     * @post $result >= 0.0
     */
    public double GetBaudRate() {
        return this.getBaudRate();
    }

    /**
     * Gets the baud rate
     * @return the baud rate
     * @pre $none
     * @post $result >= 0.0
     */
    public double getBaudRate() {
        return baudRate_;
    }

    /**
     * Gets the I/O real number based on a given value
     * @param value  the specified value
     * @return real number
     * @deprecated As of GridSim 2.1, replaced by {@link #realIO(double)}
     * @pre $none
     * @post $result >= 0.0
     */
    public double real_io(double value) {
        return this.realIO(value);
    }

    /**
     * Gets the I/O real number based on a given value
     * @param value  the specified value
     * @return real number
     * @pre $none
     * @post $result >= 0.0
     */
    public double realIO(double value) {
        return GridSimRandom.realIO(value);
    }

    /**
     * A method that gets one process event at one time until the end
     * of a simulation, then delivers an event to the entity (its parent)
     * @pre $none
     * @post $none
     */
    public void body()
    {
        // Process events
        Sim_event ev = new Sim_event();
        for ( sim_get_next(ev); ev.get_tag() != GridSimTags.END_OF_SIMULATION;
             sim_get_next(ev) )
        {
            if (ev.get_data() != null)
            {
                IO_data io = (IO_data) ev.get_data();

                // Identify ID of an entity which acts as Input/Buffer entity
                // of destination entity
                int id = GridSim.getEntityId("Input_" +
                           Sim_system.get_entity( io.getDestID() ).get_name());

                // Send first and then hold
                super.sim_schedule(id, GridSimTags.SCHEDULE_NOW,
                        ev.get_tag(), io);

                double receiverBaudRate = ( (Input)
                            Sim_system.get_entity(id) ).getBaudRate();

                double minBaudRate = Math.min(baudRate_, receiverBaudRate);
                double communicationDelay = GridSimRandom.realIO(
                            io.getByteSize() / minBaudRate );

                // TODO: Below is deprecated for SimJava 2
                //super.sim_hold(communicationDelay);
                super.sim_process(communicationDelay);
            }
        }
    }

} // end class
