/*
 * Title:        GridSim Toolkit
 * Description:  GridSim (Grid Simulation) Toolkit for Modeling and Simulation
 *               of Parallel and Distributed Systems such as Clusters and Grids
 * Licence:      GPL - http://www.gnu.org/copyleft/gpl.html
 *
 * $Id: MachineList.java,v 1.9 2003/12/13 16:09:13 anthony Exp $
 */

package gridsim;

import java.util.LinkedList;
import java.util.Iterator;


/**
 * GridSim MachineList simulates a collection of machines. It is up to the
 * GridSim users to define the connectivity among the machines in a collection.
 * Therefore, this class can be instantiated to model a simple LAN to cluster
 * to WAN.
 *
 * @author       Manzur Murshed and Rajkumar Buyya
 * @version      2.2, December 2003
 * @invariant $none
 */
public class MachineList extends LinkedList
{
    /**
     * Allocates a new MachineList object
     * @pre $none
     * @post $none
     */
    public MachineList() {
        super();
    }

    /**
     * Gets the Machine object for a particular ID
     * @param id    the machine ID
     * @return the Machine object or <tt>null</tt> if no machine exists
     * @see gridsim.Machine
     * @deprecated As of GridSim 2.1, replaced by {@link #getMachine(int)}
     * @pre id > 0
     * @post $none
     */
    public Machine GetMachine(int id) {
        return this.getMachine(id);
    }

    /**
     * Gets the Machine object for a particular ID
     * @param id    the machine ID
     * @return the Machine object or <tt>null</tt> if no machine exists
     * @see gridsim.Machine
     * @pre id >= 0
     * @post $none
     */
    public Machine getMachine(int id)
    {
        Machine obj = null;

        Iterator it = super.iterator();
        while ( it.hasNext() )
        {
            obj = (Machine) it.next();
            if (obj.getMachineID() == id) {
                return obj;
            }
        }

        return null;    // no machine with given id
    }

    /**
     * Gets the total number of PEs for all Machines
     * @return number of PEs
     * @deprecated As of GridSim 2.1, replaced by {@link #getNumPE()}
     * @pre $none
     * @post $result >= 0
     */
    public int GetNoOfPEs() {
        return this.getNumPE();
    }

    /**
     * Gets the total number of PEs for all Machines
     * @return number of PEs
     * @pre $none
     * @post $result >= 0
     */
    public int getNumPE()
    {
        int totalSize = 0;
        Machine obj = null;

        Iterator it = super.iterator();
        while ( it.hasNext() )
        {
            obj = (Machine) it.next();
            totalSize += obj.getNumPE();
        }

        return totalSize;
    }

    /**
     * Gets the total number of <tt>FREE</tt> or non-busy PEs for all Machines
     * @return number of PEs
     * @pre $none
     * @post $result >= 0
     */
    public int getNumFreePE()
    {
        int free = 0;
        Machine obj = null;

        Iterator it = super.iterator();
        while ( it.hasNext() )
        {
            obj = (Machine) it.next();
            free += obj.getNumFreePE();
        }

        return free;
    }

    /**
     * Gets the total number of <tt>BUSY</tt> PEs for all Machines
     * @return number of PEs
     * @pre $none
     * @post $result >= 0
     */
    public int getNumBusyPE() {
        return this.getNumPE() - this.getNumFreePE();
    }

    /**
     * Gets a Machine with free PE
     * @return a machine object or <tt>null</tt> if not found
     * @deprecated As of GridSim 2.1, replaced by
     *             {@link #getMachineWithFreePE()}
     * @pre $none
     * @post $none
     */
    public Machine GetMachineWithFreePE() {
        return this.getMachineWithFreePE();
    }

    /**
     * Gets a Machine with free PE
     * @return a machine object or <tt>null</tt> if not found
     * @pre $none
     * @post $none
     */
    public Machine getMachineWithFreePE()
    {
        Machine obj = null;

        Iterator it = super.iterator();
        while ( it.hasNext() )
        {
            obj = (Machine) it.next();
            PEList myPElist = obj.getPEList();

            if (myPElist.getFreePEID() != -1) {
                return obj;   // a machine with Free ID is found.
            }
        }

        return null;    // none of the machines have free PE.
    }

    /**
     * Sets the particular PE status on a Machine
     * @param status   PE status, either <tt>PE.FREE</tt> or <tt>PE.BUSY</tt>
     * @param machineID    Machine ID
     * @param peID     PE id
     * @return <tt>true</tt> if the PE status has changed, <tt>false</tt>
     * otherwise (Machine id or PE id might not be exist)
     * @pre machineID >= 0
     * @pre peID >= 0
     * @post $none
     */
    public boolean setStatusPE(boolean status, int machineID, int peID)
    {
        Machine obj = getMachine(machineID);
        if (obj == null) {
            return false;
        }

        return obj.setStatusPE(status, peID);
    }

    /**
     * Gets the byte size of this class
     * @return the byte size
     * @deprecated As of GridSim 2.1, replaced by {@link #getByteSize()}
     * @pre $none
     * @post $result >= 0
     */
    public int ByteSize() {
        return this.getByteSize();
    }

    /**
     * Gets the byte size of this class
     * @return the byte size
     * @pre $none
     * @post $result >= 0
     */
    public int getByteSize()
    {
        int bSize = 0;
        Machine obj = null;

        Iterator it = super.iterator();
        while ( it.hasNext() )
        {
            obj = (Machine) it.next();
            bSize += obj.getByteSize();
        }

        return bSize;
    }

} // end class

