/*
 * Title:        GridSim Toolkit
 * Description:  GridSim (Grid Simulation) Toolkit for Modeling and Simulation
 *               of Parallel and Distributed Systems such as Clusters and Grids
 * Licence:      GPL - http://www.gnu.org/copyleft/gpl.html
 *
 * $Id: Machine.java,v 1.9 2003/12/13 16:09:12 anthony Exp $
 */


package gridsim;

import java.util.Iterator;


/**
 * GridSim Machine class represents an uniprocessor or shared memory
 * multiprocessor machine. It can contain one or more Processing Elements (PEs).
 *
 * @author       Manzur Murshed and Rajkumar Buyya
 * @version      2.2, December 2003
 * @invariant $none
 */
public class Machine
{
    // |PEs| > 1 is SMP (Shared Memory Multiprocessors)
    private PEList PEList_;
    private int id_;


    /**
     * Allocates a new Machine object
     * @param id    the machine ID
     * @param list  list of PEs
     * @pre id > 0
     * @pre list != null
     * @post $none
     */
    public Machine(int id, PEList list)
    {
        this.id_ = id;
        this.PEList_ = list;
    }

    /**
     * Gets the machine ID
     * @return the machine ID
     * @deprecated As of GridSim 2.1, replaced by {@link #getMachineID()}
     * @pre $none
     * @post $result > 0
     */
    public int GetID() {
        return this.getMachineID();
    }

    /**
     * Gets the machine ID
     * @return the machine ID
     * @pre $none
     * @post $result > 0
     */
    public int getMachineID() {
        return id_;
    }

    /**
     * Gets the number of PEs
     * @return the number of PEs
     * @deprecated As of GridSim 2.1, replaced by {@link #getSize()}
     * @pre $none
     * @post $result >= 0
     */
    public int GetSize() {
        return this.getSize();
    }

    /**
     * Gets the number of PEs
     * @return the number of PEs
     * @pre $none
     * @post $result >= 0
     */
    public int getSize() {
        return PEList_.size();
    }

    /**
     * Gets the linked-list of all PEs
     * @return the linked-list of all PEs
     * @see gridsim.PEList
     * @deprecated As of GridSim 2.1, replaced by {@link #getPEList()}
     * @pre $none
     * @post $result != null
     */
    public PEList GetPEs() {
        return this.getPEList();
    }

    /**
     * Gets the linked-list of all PEs
     * @return the linked-list of all PEs
     * @see gridsim.PEList
     * @pre $none
     * @post $result != null
     */
    public PEList getPEList() {
        return PEList_;
    }

    /**
     * Gets the Millions Instruction Per Second (MIPS) Rating.
     * However, for Shared Memory Multiprocessors (SMPs), it is is
     * generally assumed that all PEs have the same rating.
     * @return the sum of MIPS rating of all PEs in a machine.
     * @deprecated As of GridSim 2.1, replaced by {@link #getMIPSRating()}
     * @pre $none
     * @post $result >= 0
     */
    public int GetMIPSRating() {
        return this.getMIPSRating();
    }

    /**
     * Gets the Millions Instruction Per Second (MIPS) Rating.
     * However, for Shared Memory Multiprocessors (SMPs), it is is
     * generally assumed that all PEs have the same rating.
     * @return the sum of MIPS rating of all PEs in a machine.
     * @pre $none
     * @post $result >= 0
     */
    public int getMIPSRating()
    {
        int rating = 0;
        PE obj = null;

        // a loop that adds all the PE's MIPS rating
        Iterator it = PEList_.iterator();
        while ( it.hasNext() )
        {
            obj = (PE) it.next();
            rating += obj.getMIPSRating();
        }

        return rating;
    }

    /**
     * Sets the particular PE status on this Machine
     * @param status   PE status, either <tt>PE.FREE</tt> or <tt>PE.BUSY</tt>
     * @param peID     PE id
     * @return <tt>true</tt> if the PE status has changed, <tt>false</tt>
     * otherwise (PE id might not be exist)
     * @pre peID >= 0
     * @post $none
     */
    public boolean setStatusPE(boolean status, int peID) {
        return PEList_.setStatusPE(status, peID);
    }

    /**
     * Gets the number of PE for this Machine
     * @return number of PE
     * @pre $none
     * @post $result >= 0
     */
    public int getNumPE() {
        return PEList_.size();
    }

    /**
     * Gets the number of <tt>FREE</tt> or non-busy PE for this Machine
     * @return number of PE
     * @pre $none
     * @post $result >= 0
     */
    public int getNumFreePE(){
        return PEList_.getNumFreePE();
    }

    /**
     * Gets the number of <tt>BUSY</tt> PE for this Machine
     * @return number of PE
     * @pre $none
     * @post $result >= 0
     */
    public int getNumBusyPE() {
        return PEList_.getNumBusyPE();
    }

    /**
     * Gets the byte size of this class
     * @return the byte size
     * @deprecated As of GridSim 2.1, replaced by {@link #getByteSize()}
     * @pre $none
     * @post $result > 0
     */
    public int ByteSize() {
        return this.getByteSize();
    }

    /**
     * Gets the byte size of this class
     * @return the byte size
     * @pre $none
     * @post $result > 0
     */
    public int getByteSize()
    {
        int totalInt = 4;
        return totalInt + PEList_.getByteSize();
    }

} // end class

