/*
 * Title:        GridSim Toolkit
 * Description:  GridSim (Grid Simulation) Toolkit for Modeling and Simulation
 *               of Parallel and Distributed Systems such as Clusters and Grids
 * Licence:      GPL - http://www.gnu.org/copyleft/gpl.html
 *
 * $Id: Input.java,v 1.8 2003/12/13 22:41:53 anthony Exp $
 */

package gridsim;

import eduni.simjava.*;

/**
 * GridSim Input class defines a port through which a simulation entity
 * receives data from the simulated network.
 * <p>
 * It maintains an event queue
 * to serialize the data-in-flow and delivers to its parent entity.
 * It accepts messages that comes from GridSim entities 'Output' entity
 * and passes the same to the GridSim entity.
 * It simulates Network communication delay depending on Baud rate
 * and data length. Simultaneous inputs can be modeled using multiple
 * instances of this class.
 *
 * @author       Manzur Murshed and Rajkumar Buyya
 * @version      2.2, December 2003
 * @invariant $none
 */
public class Input extends Sim_entity
{
    private Sim_port inPort_;
    private double baudRate_;

    /**
     * Allocates a new Input object
     * @param name         the name of this object
     * @param baudRate     the communication speed
     * @throws NullPointerException This happens when creating this entity
     *                  before initializing GridSim package or this entity name
     *                  is <tt>null</tt> or empty
     * @pre name != null
     * @pre baudRate >= 0.0
     * @post $none
     */
    public Input(String name, double baudRate) throws NullPointerException
    {
        super(name);
        this.baudRate_ = baudRate;

        inPort_ = new Sim_port("input_buffer");
        super.add_port(inPort_);
    }

    /**
     * Gets the baud rate
     * @return the baud rate
     * @deprecated As of GridSim 2.1, replaced by {@link #getBaudRate()}
     * @pre $none
     * @post $result >= 0.0
     */
    public double GetBaudRate() {
        return this.getBaudRate();
    }

    /**
     * Gets the baud rate
     * @return the baud rate
     * @pre $none
     * @post $result >= 0.0
     */
    public double getBaudRate() {
        return baudRate_;
    }

    /**
     * Gets the I/O real number based on a given value
     * @param value   the specified value
     * @return real number
     * @deprecated As of GridSim 2.1, replaced by {@link #realIO(double)}
     * @pre value >= 0.0
     * @post $result >= 0.0
     */
    public double real_io(double value) {
        return this.realIO(value);
    }

    /**
     * Gets the I/O real number based on a given value
     * @param value   the specified value
     * @return real number
     * @pre value >= 0.0
     * @post $result >= 0.0
     */
    public double realIO(double value) {
        return GridSimRandom.realIO(value);
    }

    /**
     * A method that gets one process event at one time until the end
     * of a simulation, then delivers an event to the entity (its parent)
     * @pre $none
     * @post $none
     */
    public void body()
    {
        // Process events
        Sim_event ev = new Sim_event();
        for ( sim_get_next(ev); ev.get_tag() != GridSimTags.END_OF_SIMULATION;
             sim_get_next(ev) )
        {
            if (ev.get_data() != null)
            {
                IO_data io = (IO_data) ev.get_data();

                // Simulate Transmission Time after Receiving
                // Hold first then dispatch
                double senderBaudRate = ( (Output)
                        Sim_system.get_entity(ev.get_src()) ).getBaudRate();

                double minBaudRate = Math.min(baudRate_, senderBaudRate);
                double communicationDelay = GridSimRandom.realIO(
                                io.getByteSize() / minBaudRate);

                // TODO: Below is deprecated for SimJava 2
                //super.sim_hold(communicationDelay);
                super.sim_process(communicationDelay);   // receiving time


                // Deliver Event to the entity (its parent) to which
                // it is acting as buffer
                super.sim_schedule( inPort_, GridSimTags.SCHEDULE_NOW,
                        ev.get_tag(), io.getData() );
            }
        }
    }

} // end class

