/*
 * Title:        GridSim Toolkit
 * Description:  GridSim (Grid Simulation) Toolkit for Modeling and Simulation
 *               of Parallel and Distributed Systems such as Clusters and Grids
 * Licence:      GPL - http://www.gnu.org/copyleft/gpl.html
 *
 * $Id: Gridlet.java,v 1.22 2003/12/14 00:16:08 anthony Exp $
 */

package gridsim;

import java.util.*;
import java.text.DecimalFormat;


/**
 * A Gridlet is a package that contains  all the information related to the job
 * and its execution management details such as job length expressed in MI
 * (Millions Instruction), disk I/O operations, the size of input and output
 * files, and the job originator.
 * <p>
 * These basic parameters help in determining
 * execution time, the time required to transport input and output files between
 * users and remote resources, and returning the processed Gridlets back to the
 * originator along with the results.
 * <p>
 * Individual users model their application by creating Gridlets for
 * processing them
 * on Grid resources assigned by scheduling entities (resource brokers).
 *
 * @author       Manzur Murshed and Rajkumar Buyya
 * @version      2.2, December 2003
 * @invariant $none
 */
public class Gridlet
{
    // the User or Broker ID. It is advisable that broker set this ID
    // with its own ID, so that GridResource returns to it after the execution
    private int userID_;

    // the size of this Gridlet to be executed in a GridResource (unit: in MI)
    private double gridletLength_;

    // the input file size of this Gridlet before execution (unit: in byte)
    private long gridletFileSize_;   // in byte = program + input data size

    // the output file size of this Gridlet after execution (unit in byte)
    private long gridletOutputSize_;

    private int gridletID_;          // this Gridlet ID
    private int status_;             // status of this Gridlet
    private DecimalFormat num_;      // to format the decimal number
    private double finishTime_;      // the time where this Gridlet completes

    // start time of executing this Gridlet.
    // With new functionalities, such as CANCEL, PAUSED and RESUMED, this
    // attribute only stores the latest execution time. Previous execution time
    // are ignored.
    private double execStartTime_;   // in simulation time

    // records the transaction history for this Gridlet
    private StringBuffer history_;
    private ArrayList resList_;
    private int index_;

    ////////////////////////////////////////////
    // Below are CONSTANTS attributes
    /** The Gridlet has been created and added to the GridletList object */
    public static final int CREATED = 0;

    /** The Gridlet has been assigned to a GridResource object as planned */
    public static final int READY = 1;

    /** The Gridlet has moved to a Grid node */
    public static final int QUEUED = 2;

    /** The Gridlet is in execution in a Grid node */
    public static final int INEXEC = 3;

    /** The Gridlet has been executed successfully */
    public static final int SUCCESS = 4;

    /** The Gridlet is failed */
    public static final int FAILED = 5;

    /** The Gridlet has been canceled.  */
    public static final int CANCELED = 6;

    /** The Gridlet has been paused. It can be resumed by changing the status
     * into <tt>RESUMED</tt>.
     */
    public static final int PAUSED = 7;

    /** The Gridlet has been resumed from <tt>PAUSED</tt> state. */
    public static final int RESUMED = 8;

    /////////////////////////////////////////////////////////////

    /**
     * Allocates a new Gridlet object. The Gridlet length, input and output
     * file sizes should be greater than or equal to 1.
     * @param gridletID            the unique ID of this Gridlet
     * @param gridletLength        the length or size of this Gridlet to be
     *                             executed in a GridResource
     * @param gridletFileSize      the input file size of this Gridlet
     *                             <tt>BEFORE</tt> submitting to a GridResource
     * @param gridletOutputSize    the output file size of this Gridlet
     *                             <tt>AFTER</tt> submitting and executing to
     *                             a GridResource
     * @pre gridletID >= 0
     * @pre gridletLength >= 0.0
     * @pre gridletFileSize >= 1
     * @pre gridletOutputSize >= 1
     * @post $none
     */
    public Gridlet(int gridletID, double gridletLength,
                long gridletFileSize, long gridletOutputSize)
    {
        // to be set by the Broker or other scheduler
        this.userID_ = -1;
        this.status_ = CREATED;

        // initial default values
        this.gridletID_ = gridletID;

        // Gridlet length, Input and Output size should be at least 1 byte.
        this.gridletLength_ = Math.max(1, gridletLength);
        this.gridletFileSize_ = Math.max(1, gridletFileSize);
        this.gridletOutputSize_ = Math.max(1, gridletOutputSize);

        // initial default values
        this.execStartTime_ = 0.0;
        this.finishTime_ = -1.0;    // meaning this Gridlet hasn't finished yet

        this.resList_ = new ArrayList();
        this.index_ = -1;
        this.num_ = new DecimalFormat("#0.00#");   // with 3 decimal spaces

        // Creates the history or transactions of this Gridlet
        history_ = new StringBuffer(1000);
        history_.append("Time below denotes the simulation time.");
        history_.append( System.getProperty("line.separator") );
        history_.append("Time (sec)       Description Gridlet #" + gridletID);
        history_.append( System.getProperty("line.separator") );
        history_.append("------------------------------------------");
        history_.append( System.getProperty("line.separator") );
        write("Creates Gridlet ID #" + gridletID);
    }


    //////////////////////// INTERNAL CLASS ///////////////////////////////////

    /**
     * Internal class that keeps track Gridlet's movement in different
     * GridResources
     */
    private class Resource
    {
        /** Gridlet's submission time to a GridResource */
        public double submissionTime;

        /** The time of this Gridlet resides in a GridResource
         * (from arrival time until departure time).
         */
        public double wallClockTime;

        /** The total execution time of this Gridlet in a GridResource. */
        public double actualCPUTime;

        /** Cost per second a GridResource charge to execute this Gridlet */
        public double costPerSec;

        /** Gridlet's length finished so far */
        public double finishedSoFar;

        /** a GridResource id */
        public int resourceId;

        /** a GridResource name */
        public String resourceName;

        /** Constructor */
        public Resource()
        {
            submissionTime = 0.0;
            wallClockTime = 0.0;
            actualCPUTime = 0.0;
            costPerSec = 0.0;
            finishedSoFar = 0.0;
            resourceId = -1;
            resourceName = null;
        }

    } // end of internal class

    //////////////////////// End of Internal Class //////////////////////////


    /**
     * Gets the history of this Gridlet. The layout of this history is in a
     * readable table column with <tt>time</tt> and <tt>description</tt>
     * as headers.
     * @return a String containing the history of this Gridlet object.
     * @pre $none
     * @post $result != null
     */
    public String getGridletHistory() {
        return history_.toString();
    }

    /**
     * Gets the length of this Gridlet that has been executed so far
     * from the latest GridResource. This
     * method is useful when trying to move this Gridlet into different
     * GridResources or to cancel it.
     * @return the length of a partially executed Gridlet or the full Gridlet
     *         length if it is completed
     * @pre $none
     * @post $result >= 0.0
     */
    public double getGridletFinishedSoFar()
    {
        if (index_ == -1) {
            return gridletLength_;
        }

        double finish = ( (Resource) resList_.get(index_) ).finishedSoFar;
        if (finish > gridletLength_) {
            return gridletLength_;
        }

        return finish;
    }

    /**
     * Checks whether this Gridlet has finished execution or not
     * @return <tt>true</tt> if this Gridlet has finished execution,
     *         <tt>false</tt> otherwise
     * @pre $none
     * @post $none
     */
    public boolean isFinished()
    {
        if (index_ == -1) {
            return false;
        }

        boolean completed = false;

        // if result is 0 or -ve then this Gridlet has finished
        double finish = ( (Resource) resList_.get(index_) ).finishedSoFar;
        double result = gridletLength_ - finish;
        if (result <= 0.0) {
            completed = true;
        }

        return completed;
    }

    /**
     * Sets the length of this Gridlet that has been executed so far.
     * This method is used by ResGridlet class when an application
     * is decided to cancel or to move this Gridlet into different
     * GridResources.
     * @param length    length of this Gridlet
     * @see gridsim.AllocPolicy
     * @see gridsim.ResGridlet
     * @pre length >= 0.0
     * @post $none
     */
    public void setGridletFinishedSoFar(double length)
    {
        // if length is -ve then ignore
        if (length < 0.0 || index_ < 0) {
            return;
        }

        Resource res = (Resource) resList_.get(index_);
        res.finishedSoFar = length;

        write("Sets the length's finished so far to " + length);
    }

    /**
     * Gets the Gridlet ID
     * @return the Gridlet ID
     * @deprecated As of GridSim 2.1, replaced by {@link #getGridletID()}
     * @pre $none
     * @post $result >= 0
     */
    public int GetGridletID() {
        return this.getGridletID();
    }

    /**
     * Gets the Gridlet ID
     * @return the Gridlet ID
     * @pre $none
     * @post $result >= 0
     */
    public int getGridletID() {
        return gridletID_;
    }

    /**
     * Sets the user or owner ID of this Gridlet. It is <tt>VERY</tt> important
     * to set the user ID, otherwise this Gridlet will not be executed in a
     * GridResource.
     * @param id  the user ID
     * @deprecated As of GridSim 2.1, replaced by {@link #setUserID(int)}
     * @pre id >= 0
     * @post $none
     */
    public void SetUserID(int id) {
        this.setUserID(id);
    }

    /**
     * Sets the user or owner ID of this Gridlet. It is <tt>VERY</tt> important
     * to set the user ID, otherwise this Gridlet will not be executed in a
     * GridResource.
     * @param id  the user ID
     * @pre id >= 0
     * @post $none
     */
    public void setUserID(int id)
    {
        userID_ = id;
        write("Assigns the Gridlet to " + GridSim.getEntityName(id) +
              " (ID #" + id + ")");
    }

    /**
     * Gets the user or owner ID of this Gridlet
     * @return the user ID or <tt>-1</tt> if the user ID has not been set before
     * @deprecated As of GridSim 2.1, replaced by {@link #getUserID()}
     * @pre $none
     * @post $result >= -1
     */
    public int GetUserID() {
        return this.getUserID();
    }

    /**
     * Gets the user or owner ID of this Gridlet
     * @return the user ID or <tt>-1</tt> if the user ID has not been set before
     * @pre $none
     * @post $result >= -1
     */
    public int getUserID() {
        return userID_;
    }

    /**
     * Gets the latest resource ID that processes this Gridlet
     * @return the resource ID or <tt>-1</tt> if none
     * @deprecated As of GridSim 2.1, replaced by {@link #getResourceID()}
     * @pre $none
     * @post $result >= -1
     */
    public int GetResourceID() {
        return this.getResourceID();
    }

    /**
     * Gets the latest resource ID that processes this Gridlet
     * @return the resource ID or <tt>-1</tt> if none
     * @pre $none
     * @post $result >= -1
     */
    public int getResourceID()
    {
        if (index_ == -1) {
            return -1;
        }

        return ( (Resource) resList_.get(index_) ).resourceId;
    }

    /**
     * Gets the input file size of this Gridlet <tt>BEFORE</tt>
     * submitting to a GridResource
     * @return the input file size of this Gridlet
     * @deprecated As of GridSim 2.1, replaced by {@link #getGridletFileSize()}
     * @pre $none
     * @post $result >= 1
     */
    public long GetGridletFileSize() {
        return this.getGridletFileSize();
    }

    /**
     * Gets the input file size of this Gridlet <tt>BEFORE</tt>
     * submitting to a GridResource
     * @return the input file size of this Gridlet
     * @pre $none
     * @post $result >= 1
     */
    public long getGridletFileSize() {
        return gridletFileSize_;
    }

    /**
     * Gets the output size of this Gridlet <tt>AFTER</tt> submitting and
     * executing to a GridResource
     * @return the Gridlet output file size
     * @deprecated As of GridSim 2.1, replaced by
     *             {@link #getGridletOutputSize()}
     * @pre $none
     * @post $result >= 1
     */
    public long GetGridletOutputSize() {
        return this.getGridletOutputSize();
    }

    /**
     * Gets the output size of this Gridlet <tt>AFTER</tt> submitting and
     * executing to a GridResource
     * @return the Gridlet output file size
     * @pre $none
     * @post $result >= 1
     */
    public long getGridletOutputSize() {
        return gridletOutputSize_;
    }

    /**
     * Sets the resource parameters for which this Gridlet is going to be
     * executed.
     * @param resourceID   the GridResource ID
     * @param costPerSec   the cost running this GridResource per second
     * @deprecated As of GridSim 2.1, replaced by
     *             {@link #setResourceParameter(int, double)}
     * @pre resourceID >= 0
     * @pre costPerSec > 0.0
     * @post $none
     */
    public void SetResParam(int resourceID, double costPerSec) {
        this.setResourceParameter(resourceID, costPerSec);
    }

    /**
     * Sets the resource parameters for which this Gridlet is going to be
     * executed.
     * @param resourceID   the GridResource ID
     * @param cost   the cost running this GridResource per second
     * @pre resourceID >= 0
     * @pre cost > 0.0
     * @post $none
     */
    public void setResourceParameter(int resourceID, double cost)
    {
        Resource res = new Resource();
        res.resourceId = resourceID;
        res.costPerSec = cost;
        res.resourceName = GridSim.getEntityName(resourceID);

        // add into a list if moving to a new grid resource
        resList_.add(res);

        if (index_ == -1)
        {
            write("Allocates this Gridlet to " + res.resourceName +
                  " (ID #" + resourceID + ") with cost = $" + cost + "/sec");
        }
        else
        {
            int id = ( (Resource) resList_.get(index_) ).resourceId;
            String name = ( (Resource) resList_.get(index_) ).resourceName;
            write("Moves Gridlet from " + name + " (ID #" + id + ") to " +
                  res.resourceName + " (ID #" + resourceID +
                  ") with cost = $" + cost + "/sec");
        }

        index_++;  // initially, index_ = -1
    }

    /**
     * Sets the submission or arrival time of this Gridlet into a GridResource
     * @param clockTime     the submission time
     * @deprecated As of GridSim 2.1, replaced by
     *             {@link #setSubmissionTime(double)}
     * @pre clockTime >= 0.0
     * @post $none
     */
    public void SetSubmissionTime(double clockTime) {
        this.setSubmissionTime(clockTime);
    }

    /**
     * Sets the submission or arrival time of this Gridlet into a GridResource
     * @param clockTime     the submission time
     * @pre clockTime >= 0.0
     * @post $none
     */
    public void setSubmissionTime(double clockTime)
    {
        if (clockTime < 0.0 || index_ < 0) {
            return;
        }

        Resource res = (Resource) resList_.get(index_);
        res.submissionTime = clockTime;

        write( "Sets the submission time to " + num_.format(clockTime) );
    }

    /**
     * Gets the submission or arrival time of this Gridlet from
     * the latest GridResource
     * @return the submission time or <tt>0.0</tt> if none
     * @pre $none
     * @post $result >= 0.0
     */
    public double getSubmissionTime()
    {
        if (index_ == -1) {
            return 0.0;
        }

        return ( (Resource) resList_.get(index_) ).submissionTime;
    }

    /**
     * Sets the execution start time of this Gridlet inside a GridResource.
     * <b>NOTE:</b> With new functionalities, such as being able to cancel /
     * to pause / to resume this Gridlet, the execution start time only holds
     * the latest one. Meaning, all previous execution start time are ignored.
     *
     * @param clockTime     the latest execution start time
     * @deprecated As of GridSim 2.1, replaced by
     *             {@link #setExecStartTime(double)}
     * @pre clockTime >= 0.0
     * @post $none
     */
    public void SetExecStartTime(double clockTime) {
        this.setExecStartTime(clockTime);
    }

    /**
     * Sets the execution start time of this Gridlet inside a GridResource.
     * <b>NOTE:</b> With new functionalities, such as being able to cancel /
     * to pause / to resume this Gridlet, the execution start time only holds
     * the latest one. Meaning, all previous execution start time are ignored.
     *
     * @param clockTime     the latest execution start time
     * @pre clockTime >= 0.0
     * @post $none
     */
    public void setExecStartTime(double clockTime)
    {
        execStartTime_ = clockTime;
        write( "Sets the execution start time to " + num_.format(clockTime) );
    }

    /**
     * Gets the latest execution start time
     * @return the latest execution start time
     * @pre $none
     * @post $result >= 0.0
     */
    public double getExecStartTime() {
        return execStartTime_;
    }

    /**
     * Sets this Gridlet's execution parameters. These parameters are set by
     * the GridResource before departure or sending back to the original
     * Gridlet's owner.
     *
     * @param wallClockTime    the time of this Gridlet resides in
     *                         a GridResource (from arrival time until
     *                         departure time).
     * @param actualCPUTime    the total execution time of this Gridlet in a
     *                         GridResource.
     *
     * @deprecated As of GridSim 2.1, replaced by
     *             {@link #setExecParam(double, double)}
     * @pre wallClockTime >= 0.0
     * @pre actualCPUTime >= 0.0
     * @post $none
     */
    public void SetExecParam(double wallClockTime, double actualCPUTime) {
        this.setExecParam(wallClockTime, actualCPUTime);
    }

    /**
     * Sets this Gridlet's execution parameters. These parameters are set by
     * the GridResource before departure or sending back to the original
     * Gridlet's owner.
     *
     * @param wallTime    the time of this Gridlet resides in
     *                         a GridResource (from arrival time until
     *                         departure time).
     * @param actualTime    the total execution time of this Gridlet in a
     *                         GridResource.
     *
     * @pre wallTime >= 0.0
     * @pre actualTime >= 0.0
     * @post $none
     */
    public void setExecParam(double wallTime, double actualTime)
    {
        if (wallTime < 0.0 || actualTime < 0.0 || index_ < 0) {
            return;
        }

        Resource res = (Resource) resList_.get(index_);
        res.wallClockTime = wallTime;
        res.actualCPUTime = actualTime;

        write("Sets the wall clock time to "+ num_.format(wallTime)+
            " and the actual CPU time to " + num_.format(actualTime));
    }

    /**
     * Sets the status code of this Gridlet
     * @param status    the status code of this Gridlet
     * @deprecated As of GridSim 2.1, replaced by {@link #setGridletStatus(int)}
     * @throws Exception Invalid Gridlet status
     * @pre status >= 0 && status <= 5
     * @post $none
     */
    public void SetStatus(int status) throws Exception {
        this.setGridletStatus(status);
    }

    /**
     * Sets the status code of this Gridlet
     * @param newStatus    the status code of this Gridlet
     * @throws Exception   Invalid range of Gridlet status
     * @pre newStatus >= 0 && newStatus <= 8
     * @post $none
     */
    public void setGridletStatus(int newStatus) throws Exception
    {
        // if the new status is same as current one, then ignore the rest
        if (status_ == newStatus) {
            return;
        }

        // throws an exception if the new status is outside the range
        if (newStatus < Gridlet.CREATED || newStatus > Gridlet.RESUMED)
        {
            throw new Exception("Gridlet.setGridletStatus() : Error - " +
                    "Invalid integer range for Gridlet status.");
        }

        if (newStatus == Gridlet.SUCCESS) {
            finishTime_ = GridSim.clock();
        }

        String obj = "Sets Gridlet status from " + getGridletStatusString();
        this.status_ = newStatus;
        write( obj + " to " + getGridletStatusString() );
    }

    /**
     * Gets the status code of this Gridlet
     * @return the status code of this Gridlet
     * @deprecated As of GridSim 2.1, replaced by {@link #getGridletStatus()}
     * @pre $none
     * @post $result >= 0
     */
    public int GetStatus() {
        return this.getGridletStatus();
    }

    /**
     * Gets the status code of this Gridlet
     * @return the status code of this Gridlet
     * @pre $none
     * @post $result >= 0
     */
    public int getGridletStatus() {
        return status_;
    }

    /**
     * Gets the string representation of the current Gridlet status code
     * @return the Gridlet status code as a string or <tt>null</tt> if the
     *         status code is unknown
     * @deprecated As of GridSim 2.1, replaced by
     *             {@link #getGridletStatusString()}
     * @pre $none
     * @post $none
     */
    public String GetStatusString() {
        return this.getGridletStatusString();
    }

    /**
     * Gets the string representation of the current Gridlet status code
     * @return the Gridlet status code as a string or <tt>null</tt> if the
     *         status code is unknown
     * @pre $none
     * @post $none
     */
    public String getGridletStatusString() {
        return Gridlet.getStatusString(status_);
    }

    /**
     * Gets the string representation of the given Gridlet status code
     * @param status    the Gridlet status code
     * @return the Gridlet status code as a string or <tt>null</tt> if the
     *         status code is unknown
     * @pre $none
     * @post $none
     */
    public static String getStatusString(int status)
    {
        String statusString = null;
        switch (status)
        {
            case Gridlet.CREATED:
                statusString = "Created";
                break;

            case Gridlet.READY:
                statusString = "Ready";
                break;

            case Gridlet.INEXEC:
                statusString = "InExec";
                break;

            case Gridlet.SUCCESS:
                statusString = "Success";
                break;

            case Gridlet.QUEUED:
                statusString = "Queued";
                break;

            case Gridlet.FAILED:
                statusString = "Failed";
                break;

            case Gridlet.CANCELED:
                statusString = "Canceled";
                break;

            case Gridlet.PAUSED:
                statusString = "Paused";
                break;

            case Gridlet.RESUMED:
                statusString = "Resumed";
                break;

            default:
                break;
        }

        return statusString;
    }

    /**
     * Gets the length of this Gridlet
     * @return the length of this Gridlet
     * @deprecated As of GridSim 2.1, replaced by {@link #getGridletLength()}
     * @pre $none
     * @post $result >= 0.0
     */
    public double GetGridletLength() {
        return this.getGridletLength();
    }

    /**
     * Gets the length of this Gridlet
     * @return the length of this Gridlet
     * @pre $none
     * @post $result >= 0.0
     */
    public double getGridletLength() {
        return gridletLength_;
    }

    /**
     * Gets the total execution time of this Gridlet from the latest
     * GridResource
     * @return the the total execution time of this Gridlet in a GridResource
     * @deprecated As of GridSim 2.1, replaced by {@link #getActualCPUTime()}
     * @pre $none
     * @post $result >= 0.0
     */
    public double GetActualCPUTime() {
        return this.getActualCPUTime();
    }

    /**
     * Gets the total execution time of this Gridlet from the latest
     * GridResource
     * @return the total execution time of this Gridlet in a GridResource
     * @pre $none
     * @post $result >= 0.0
     */
    public double getActualCPUTime()
    {
        if (index_ == -1) {
            return 0.0;
        }

        return ( (Resource) resList_.get(index_) ).actualCPUTime;
    }

    /**
     * Gets the cost running this Gridlet in the latest GridResource
     * @return the cost associated with running this Gridlet
     *         or <tt>0.0</tt> if none
     * @deprecated As of GridSim 2.1, replaced by {@link #getCostPerSec()}
     * @pre $none
     * @post $result >= 0.0
     */
    public double GetCostPerSec() {
        return this.getCostPerSec();
    }

    /**
     * Gets the cost running this Gridlet in the latest GridResource
     * @return the cost associated with running this Gridlet
     *         or <tt>0.0</tt> if none
     * @pre $none
     * @post $result >= 0.0
     */
    public double getCostPerSec()
    {
        if (index_ == -1) {
            return 0.0;
        }

        return ( (Resource) resList_.get(index_) ).costPerSec;
    }

    /**
     * Gets the total cost of processing or executing this Gridlet.
     * <tt>Processing Cost = actual CPU Time * cost per sec</tt>
     * @return the total cost of processing Gridlet
     * @deprecated As of GridSim 2.1, replaced by {@link #getProcessingCost()}
     * @see gridsim.Gridlet#getActualCPUTime()
     * @see gridsim.Gridlet#getCostPerSec()
     * @pre $none
     * @post $result >= 0.0
     */
    public double GetProcessingCost() {
        return this.getProcessingCost();
    }

    /**
     * Gets the total cost of processing or executing this Gridlet
     * <tt>Processing Cost = actual CPU Time * cost per sec</tt>
     * @return the total cost of processing Gridlet
     * @see gridsim.Gridlet#getActualCPUTime()
     * @see gridsim.Gridlet#getCostPerSec()
     * @pre $none
     * @post $result >= 0.0
     */
    public double getProcessingCost()
    {
        if (index_ == -1) {
            return 0.0;
        }

        Resource res = null;
        double cost = 0.0;
        for (int i = 0; i <= index_; i++)
        {
            res = (Resource) resList_.get(i);
            cost += (res.actualCPUTime * res.costPerSec);
        }

        return cost;
    }

    /**
     * Gets the time of this Gridlet resides in the latest GridResource
     * (from arrival time until departure time).
     * @return the time of this Gridlet resides in a GridResource
     * @deprecated As of GridSim 2.1, replaced by {@link #getWallClockTime()}
     * @pre $none
     * @post $result >= 0.0
     */
    public double GetWallClockTime() {
        return this.getWallClockTime();
    }

    /**
     * Gets the time of this Gridlet resides in the latest GridResource
     * (from arrival time until departure time).
     * @return the time of this Gridlet resides in a GridResource
     * @pre $none
     * @post $result >= 0.0
     */
    public double getWallClockTime()
    {
        if (index_ == -1) {
            return 0.0;
        }

        return ( (Resource) resList_.get(index_) ).wallClockTime;
    }

    /**
     * Gets all the GridResource names that executed this Gridlet
     * @return an array of GridResource names or <tt>null</tt> if it has none
     * @pre $none
     * @post $none
     */
    public String[] getAllResourceName()
    {
        int size = resList_.size();
        String[] data = null;

        if (size > 0)
        {
            data = new String[size];
            for (int i = 0; i < size; i++) {
                data[i] = ( (Resource) resList_.get(i)).resourceName;
            }
        }

        return data;
    }

    /**
     * Gets all the GridResource IDs that executed this Gridlet
     * @return an array of GridResource IDs or <tt>null</tt> if it has none
     * @pre $none
     * @post $none
     */
    public int[] getAllResourceID()
    {
        int size = resList_.size();
        int[] data = null;

        if (size > 0)
        {
            data = new int[size];
            for (int i = 0; i < size; i++) {
                data[i] = ( (Resource) resList_.get(i)).resourceId;
            }
        }

        return data;
    }

    /**
     * Gets the total execution time of this Gridlet in a given GridResource ID
     * @param resId  a GridResource entity ID
     * @return the total execution time of this Gridlet in a GridResource
     *         or <tt>0.0</tt> if not found
     * @pre resId >= 0
     * @post $result >= 0.0
     */
    public double getActualCPUTime(int resId)
    {
        Resource res = null;
        int size = resList_.size();
        for (int i = 0; i < size; i++)
        {
            res = (Resource) resList_.get(i);
            if (resId == res.resourceId) {
                return res.actualCPUTime;
            }
        }

        return 0.0;
    }

    /**
     * Gets the cost running this Gridlet in a given GridResource ID
     * @param resId  a GridResource entity ID
     * @return the cost associated with running this Gridlet
     *         or <tt>0.0</tt> if not found
     * @pre resId >= 0
     * @post $result >= 0.0
     */
    public double getCostPerSec(int resId)
    {
        Resource res = null;
        int size = resList_.size();
        for (int i = 0; i < size; i++)
        {
            res = (Resource) resList_.get(i);
            if (resId == res.resourceId) {
                return res.costPerSec;
            }
        }

        return 0.0;
    }

    /**
     * Gets the length of this Gridlet that has been executed so far in a given
     * GridResource ID. This method is useful when trying to move this Gridlet
     * into different GridResources or to cancel it.
     * @param resId  a GridResource entity ID
     * @return the length of a partially executed Gridlet or the full Gridlet
     *         length if it is completed or <tt>0.0</tt> if not found
     * @pre resId >= 0
     * @post $result >= 0.0
     */
    public double getGridletFinishedSoFar(int resId)
    {
        Resource res = null;
        int size = resList_.size();
        for (int i = 0; i < size; i++)
        {
            res = (Resource) resList_.get(i);
            if (resId == res.resourceId) {
                return res.finishedSoFar;
            }
        }

        return 0.0;
    }

    /**
     * Gets the submission or arrival time of this Gridlet in the
     * given GridResource ID
     * @param resId  a GridResource entity ID
     * @return the submission time or <tt>0.0</tt> if not found
     * @pre resId >= 0
     * @post $result >= 0.0
     */
    public double getSubmissionTime(int resId)
    {
        Resource res = null;
        int size = resList_.size();
        for (int i = 0; i < size; i++)
        {
            res = (Resource) resList_.get(i);
            if (resId == res.resourceId) {
                return res.submissionTime;
            }
        }

        return 0.0;
    }

    /**
     * Gets the time of this Gridlet resides in a given GridResource ID
     * (from arrival time until departure time).
     * @param resId  a GridResource entity ID
     * @return the time of this Gridlet resides in the GridResource
     *         or <tt>0.0</tt> if not found
     * @pre resId >= 0
     * @post $result >= 0.0
     */
    public double getWallClockTime(int resId)
    {
        Resource res = null;
        int size = resList_.size();
        for (int i = 0; i < size; i++)
        {
            res = (Resource) resList_.get(i);
            if (resId == res.resourceId) {
                return res.wallClockTime;
            }
        }

        return 0.0;
    }

    /**
     * Gets the GridResource name based on its ID
     * @param resId  a GridResource entity ID
     * @return the GridResource name or <tt>null</tt> if not found
     * @pre resId >= 0
     * @post $none
     */
    public String getResourceName(int resId)
    {
        Resource res = null;
        int size = resList_.size();
        for (int i = 0; i < size; i++)
        {
            res = (Resource) resList_.get(i);
            if (resId == res.resourceId) {
                return res.resourceName;
            }
        }

        return null;
    }

    /**
     * Gets the finish time of this Gridlet in a GridResource
     * @return the finish or completion time of this Gridlet or <tt>-1</tt> if
     *         not finished yet.
     * @deprecated As of GridSim 2.1, replaced by {@link #getFinishTime()}
     * @pre $none
     * @post $result >= -1
     */
    public double GetFinishTime() {
        return this.getFinishTime();
    }

    /**
     * Gets the finish time of this Gridlet in a GridResource
     * @return the finish or completion time of this Gridlet or <tt>-1</tt> if
     *         not finished yet.
     * @pre $none
     * @post $result >= -1
     */
    public double getFinishTime() {
        return finishTime_;
    }

    /////////////////////////// PRIVATE METHODS //////////////////////////////

    /**
     * Writes this particular history transaction of this Gridlet into a log
     * @param str   a history transaction of this Gridlet
     * @pre str != null
     * @post $none
     */
    private void write(String str)
    {
        history_.append( num_.format(GridSim.clock()) );
        history_.append( "   " + str + System.getProperty("line.separator") );
    }

} // end class

