/*
 * Title:        GridSim Toolkit
 * Description:  GridSim (Grid Simulation) Toolkit for Modeling and Simulation 
 *               of Parallel and Distributed Systems such as Clusters and Grids
 * Licence:      GPL - http://www.gnu.org/copyleft/gpl.html
 *
 * $Id: GridSimShutdown.java,v 1.8 2003/12/13 15:54:12 anthony Exp $
 */

package gridsim;

import java.lang.*;
import gridsim.*;
import java.util.LinkedList;
import eduni.simjava.Sim_event;

/**
 * GridimShutdown waits for termination of all user entities to determine
 * the end of simulation. 
 * <p>
 * This class signals the user-defined report-writer 
 * entity to interact with the GridStatistics entity to generate a report.
 * Finally, it signals the end of simulation to other GridSim core
 * entities.
 * <p>
 * This class wraps up the simulation by signaling the <tt>END</tt> 
 * to the entities.
 * It sends <tt>END_OF_SIMULATION</tt> message to various GridSim entities on
 * receiving <tt>END_OF_SIMULATION</tt> from all User entities.
 *
 * @author       Manzur Murshed and Rajkumar Buyya
 * @version      2.2, December 2003
 * @invariant $none
 */
public class GridSimShutdown extends GridSim
{
    private int numUser_;
    private String reportWriterName_;

    /**
     * Allocates a new GridSimShutdown object
     * @param name       the name to be associated with this entity (as 
     *                   required by Sim_entity class from simjava package)
     * @param numUser       number of User entities
     * @param reportWriterName  the name of the file to be written
     * @throws Exception This happens when creating this entity before 
     *                   initializing GridSim package or this entity name is
     *                   <tt>null</tt> or empty
     * @see gridsim.GridSim#Init(int, Calendar, boolean, String[], String[],
     *          String)
     * @see eduni.simjava.Sim_entity
     * @pre name != null
     * @pre numUser >= 0
     * @post $none
     */
    public GridSimShutdown(String name, int numUser, 
                String reportWriterName) throws Exception
    {
        super(name, GridSimTags.DEFAULT_BAUD_RATE);
        this.numUser_ = numUser;
        this.reportWriterName_ = reportWriterName;
    }

    /**
     * Discovers resources and sends <tt>END_OF_SIMULATION</tt> 
     * message to them
     * @deprecated As of GridSim 2.1, replaced by {@link #shutdownResources()}
     * @pre $none
     * @post $none
     */
    public void ShutdownResources() {
        this.shutdownResources();
    }

    /**
     * Discovers resources and sends <tt>END_OF_SIMULATION</tt> 
     * message to them
     * @pre $none
     * @post $none
     */
    public void shutdownResources()
    {
        LinkedList resIDList = null;

        // Requesting the GridInformationService to send a list of resources
        resIDList = super.getGridResourceList();

        // Send END_OF_SIMULATION event to all Resources
        for (int i = 0; i < resIDList.size(); i++)
        {
            // Get Resource ID
            int resID = ( (Integer) resIDList.get(i) ).intValue();  
            super.send(resID, GridSimTags.SCHEDULE_NOW, 
                    GridSimTags.END_OF_SIMULATION);
        }
    }

    /**
     * The main method that shuts down resources and Grid Information 
     * Service (GIS). In addition, this method writes down a report at the
     * end of a simulation based on <tt>reportWriterName</tt> defined in 
     * the Constructor.
     * <br>
     * <b>NOTE:</b> This method shuts down grid resources and GIS entities 
     *              <tt>AFTER</tt> all grid users have been shut down. 
     *              Therefore, the number of grid users given in the 
     *              Constructor <tt>must</tt> be correct. Otherwise, GridSim
     *              package hangs forever or it does not terminate properly.
     * @pre $none
     * @post $none
     */
    public void body()
    {
        Sim_event ev = new Sim_event();

        // wait for shutdown message from all users.
        for (int i = 0; i < numUser_; i++) {
            super.sim_get_next(ev);
        }

        // Shutdown resources and GIS
        shutdownResources();
        super.send(GridSim.getGridInfoServiceEntityId(), 
                GridSimTags.SCHEDULE_NOW, GridSimTags.END_OF_SIMULATION);

        // Invoke report Writter and shutdown
        if (reportWriterName_ != null)
        {
            int repWriterID = GridSim.getEntityId(reportWriterName_);
            if (repWriterID != -1)
            {
                super.send(repWriterID, GridSimTags.SCHEDULE_NOW, 
                        GridSimTags.END_OF_SIMULATION);
            }
            else
            {
                System.out.println(super.get_name() +
                        ": User defined Report Writer entity " +
                        reportWriterName_ + " does not exist.");
            }
        }
        else {
            super.shutdownGridStatisticsEntity();
        }

        super.terminateIOEntities();
    }
    
} // end class

