/*
 * Title:        GridSim Toolkit
 * Description:  GridSim (Grid Simulation) Toolkit for Modeling and Simulation
 *               of Parallel and Distributed Systems such as Clusters and Grids
 * Licence:      GPL - http://www.gnu.org/copyleft/gpl.html
 *
 * $Id: GridSim.java,v 1.28 2003/12/13 22:41:53 anthony Exp $
 */

package gridsim;

import java.io.*;
import java.util.*;
import eduni.simjava.*;


/**
 * The main class of the GridSim package that must be extended by GridSim
 * entities. It inherits event management and threaded entity features from
 * the <b>eduni.simjava.Sim_entity</b> class. This class adds networking and
 * event delivery features, which allow synchronous or asynchronous
 * communication for service access or delivery.
 * <p>
 * All classes that extend this class must implement a method called
 * <tt>body()</tt>, which is automatically invoked since it is expected to be
 * responsible for simulating entity behavior.
 * <p>
 * The entities that extend this class can be instantiated <i>with</i> or
 * <i>without</i> networked I/O ports. A networked GridSim entity gains
 * communication capability via the objects of GridSim's I/O entity classes,
 * <b>gridsim.Input</b> and <b>gridsim.Output</b> classes. Each I/O entity
 * will have a unique name assuming each GridSim entity that the user creates
 * has a unique name. For example, a resource entity with the name
 * <b>`Resource2'</b>,
 * will have an input entity whose name is prefixed with <b>`Input_'</b>,
 * making the input entity's full name <b>`Input_Resource2'</b>, which is
 * expected to be unique. The I/O entities are concurrent entities, but they
 * are visible within the GridSim entity and are able to communicate with
 * other GridSim entities by sending messages.
 * <p>
 * This class supports methods for simulation initilization, management and
 * flow control. The GridSim environment must be initialized to set-up the
 * simulation environment before creating any other GridSim entities at the
 * user level. This method also prepares the system for simulation by creating
 * three GridSim internal entities - <b>gridsim.GridInformationService</b>,
 * <b>gridsim.GridSimShutdown</b>,<b>gridsim.GridStatistics</b>. Invoking the
 * <tt>startGridSimulation()</tt> method starts the Grid simulation.
 * All the resource and
 * user entities must be instantiated in between invoking the above two
 * methods.
 * <p>
 * This class supports static methods for sending and receiving messages
 * between entities directly or via network entities, managing and accessing
 * handles to various GridSim core entities, and recording statistics.
 *
 * @author       Manzur Murshed and Rajkumar Buyya
 * @version      2.2, December 2003
 * @see eduni.simjava.Sim_entity
 * @see gridsim.Output
 * @see gridsim.Input
 * @see gridsim.GridInformationService
 * @see gridsim.GridSimShutdown
 * @see gridsim.GridStatistics
 * @invariant $none
 */
public class GridSim extends Sim_entity
{
    // a temp array: [0] = gridlet id, [1] = user Id and [2] = dest resource id
    private int[] dataArray_ = new int[3];
    private Sim_event ev_;           // a temp variable
    private boolean networkedFlag_;  // true, if networked entity, other false

    // false means NOT invoked
    private boolean terminateIOEntitiesFlag_ = false;

    /** Reading data received via input port */
    protected Sim_port input;

    /** Sending data via output port to external entities */
    protected Sim_port output;

    ///////////// STATIC variables ////////////////////

    /**
     * Simulation start date with respect to GMT 0. This object is initialized
     * during the call to {@link #init(int, Calendar, boolean, String[],
     *             String[], String)}
     */
    public static Date SimulationStartDate = null;

    /**
     * A Random object. This object is initialized during the call to
     * {@link #init(int, Calendar, boolean, String[], String[], String)}
     */
    public static GridSimRandom rand = null;

    private static Calendar calendar_ = null;    // a Calendar object
    private static GridInformationService gis_ = null;   // a GIS object
    private final static int NOT_FOUND = -1;     // a constant


    /**
     * Allocates a new GridSim object
     * <b>without</b> NETWORK communication channels: "input" and
     * "output" Sim_port. In summary, this object has <tt>NO</tt>
     * network communication or bandwidth speed.
     * @param name       the name to be associated with this entity (as
     *                   required by Sim_entity class from simjava package)
     * @throws Exception This happens when creating this entity before
     *                   initializing GridSim package or this entity name is
     *                   <tt>null</tt> or empty
     * @see gridsim.GridSim#init(int, Calendar, boolean, String[], String[],
     *          String)
     * @see eduni.simjava.Sim_entity
     * @pre name != null
     * @post $none
     */
    public GridSim(String name) throws Exception
    {
        super(name);
        networkedFlag_ = false;
        input = null;
        output = null;
        ev_ = new Sim_event();
    }

    /**
     * Allocates a new GridSim object
     * <b>with</b> NETWORK communication channels: "input" and
     * "output" Sim_port. In addition, this method will create <tt>Input</tt>
     * and <tt>Output</tt> object.
     * @param name       the name to be associated with this entity (as
     *                   required by Sim_entity class from simjava package)
     * @param baudRate   network communication or bandwidth speed
     * @throws Exception This happens when creating this entity before
     *                   initializing GridSim package or this entity name is
     *                   <tt>null</tt> or empty
     * @see gridsim.GridSim#init(int, Calendar, boolean, String[], String[],
     *          String)
     * @see eduni.simjava.Sim_entity
     * @see gridsim.Input
     * @see gridsim.Output
     * @pre name != null
     * @pre baudRate > 0.0
     * @post $none
     */
    public GridSim(String name, double baudRate) throws Exception
    {
        super(name);
        networkedFlag_ = true;

        input = new Sim_port("input");
        output = new Sim_port("output");
        ev_ = new Sim_event();

        super.add_port(input);
        super.add_port(output);

        // Every GridSim entity with network has its own input/output channels.
        // Connect this entity "input" port to its input buffer "in_port"
        new Input("Input_" + name, baudRate);
        Sim_system.link_ports(name, "input", "Input_" + name, "input_buffer");

        new Output("Output_" + name, baudRate);
        Sim_system.link_ports(name, "output", "Output_"+name, "output_buffer");
    }

    /**
     * It terminates Entities managing NETWORK communication channels.
     * It can be invoked explicity to shutdown NETWORK communication channels.
     * It is advisable for all entities extending GridSim class, explicitly
     * invoke this method to terminate <tt>Input</tt> and
     * <tt>Output</tt> entities created
     * by the constructor: {@link GridSim#GridSim(String, double)}
     * @deprecated As of GridSim 2.1, replaced by
     *             {@link #terminateIOEntities()}
     * @pre $none
     * @post $none
     */
    protected void TerminateInputOutputEntities() {
        terminateIOEntities();
    }

    /**
     * It terminates Entities managing NETWORK communication channels.
     * It can be invoked explicity to shutdown NETWORK communication channels.
     * It is advisable for all entities extending GridSim class, explicitly
     * invoke this method to terminate <tt>Input</tt> and
     * <tt>Output</tt> entities created
     * by the constructor: {@link GridSim#GridSim(String, double)}
     * @pre $none
     * @post $none
     */
    protected void terminateIOEntities()
    {
        // If it is Networked entity and Not yet terminated, then terminate.
        if ( isNetworked() && !terminateIOEntitiesFlag_ )
        {
            // Send END_OF_SIMULATION to Input entity
            send(input, 0.0, GridSimTags.END_OF_SIMULATION);

            // Send END_OF_SIMULATION to Output entity
            send(output, 0.0, GridSimTags.END_OF_SIMULATION);

            terminateIOEntitiesFlag_ = true;
        }
    }

    /**
     * It terminates the entities of this object that manage <tt>NETWORK</tt>
     * communication channels
     * @deprecated As of GridSim 2.1, replaced by
     *      {@link #finalizeGridSimulation()}
     * @see gridsim.GridSim#terminateIOEntities()
     * @pre $none
     * @post $none
     */
    protected void finalize() {
        finalizeGridSimulation();
    }

    /**
     * It terminates the entities of this object that manage <tt>NETWORK</tt>
     * communication channels
     * @see gridsim.GridSim#terminateIOEntities()
     * @pre $none
     * @post $none
     */
    protected void finalizeGridSimulation() {
        terminateIOEntities();
    }

    /**
     * Check type of entity
     * @return true if entity has NETWORK communication channel, otherwise
     *          it returns false
     * @deprecated As of GridSim 2.1, replaced by {@link #isNetworked()}
     * @pre $none
     * @post $result == true || false
     */
    public boolean IsNetworked() {
        return isNetworked();
    }

    /**
     * Check type of entity
     * @return true if entity has NETWORK communication channel, otherwise
     *          it returns false
     * @pre $none
     * @post $result == true || false
     */
    public boolean isNetworked() {
        return networkedFlag_;
    }


    //////////// SIMULATION CONTROL METHODS START ////////////

    /**
     * Gets simulation start date with respect to GMT 0. If the return object is
     * <tt>null</tt>, then need to initialize it by calling
     * {@link #init(int, Calendar, boolean, String[], String[], String)}
     * @return an object of Date or <tt>null</tt> if it is empty
     * @deprecated As of GridSim 2.1, replaced by
     *             {@link #getSimulationStartDate()}
     * @see java.util.Date
     * @pre $none
     * @post $none
     */
    public static Date GetSimulationStartDate() {
        return getSimulationStartDate();
    }

    /**
     * Gets simulation start date with respect to GMT 0. If the return object
     * is <tt>null</tt>, then need to initialize it by calling
     * {@link #init(int, Calendar, boolean, String[], String[], String)}
     * @return an object of Date
     * @see java.util.Date
     * @pre $none
     * @post $none
     */
    public static Date getSimulationStartDate() {
        return SimulationStartDate;
    }

    /**
     * Gets the current simulation Calendar
     * @return a Calendar object or <tt>null</tt> if GridSim hasn't been
     *         initialized
     * @see gridsim.GridSim#init(int, Calendar, boolean, String[], String[],
     *      String)
     * @pre $none
     * @post $none
     */
    public static Calendar getSimulationCalendar() {
        return calendar_;
    }

    /**
     * Initializes GridSim Parameters.
     * This should be called before creating any entities.
     * @param numUser  the number of User Entities created.
     *                 This parameters indicates that
     *                 <b>gridsim.GridSimShutdown</b> first waits for
     *                 User Entities's END_OF_SIMULATION signal before
     *                 issuing terminate signal to other entities
     * @param cal          the Calendar object for storing peaktime, holiday,
     *                     etc
     * @param traceFlag    true if GridSim trace need to be written
     * @param excludeFromFile  an array of String containing list of files to
     *                         be excluded from statistics
     * @param excludeFromProcessing   an array of String containing list of
     *                                processings to be excluded from writing
     *                                into a file
     * @param reportWriterName     the name of the entities which is signaled
     *                             to write any specific report just before
     *                             termination of the simulation
     * @deprecated As of GridSim 2.1, replaced by
     *             {@link #init(int, Calendar, boolean, String[],
     *             String[], String)}
     * @see gridsim.GridSimShutdown
     * @pre numUser >= 0
     * @post $none
     */
    public static void Init(int numUser, Calendar cal, boolean traceFlag,
            String[] excludeFromFile, String[] excludeFromProcessing,
            String reportWriterName)
    {
        init(numUser, cal, traceFlag, excludeFromFile,
                excludeFromProcessing, reportWriterName);
    }

    /**
     * Initializes GridSim Parameters.
     * This should be called before creating any entities.
     * @param numUser  the number of User Entities created.
     *                 This parameters indicates that
     *                 <b>gridsim.GridSimShutdown</b> first waits for
     *                 User Entities's END_OF_SIMULATION signal before
     *                 issuing terminate signal to other entities
     * @param cal          the Calendar object for storing peaktime, holiday,
     *                     etc
     * @param traceFlag    true if GridSim trace need to be written
     * @param excludeFromFile  an array of String containing list of files to
     *                         be excluded from statistics
     * @param excludeFromProcessing   an array of String containing list of
     *                                processings to be excluded from writing
     *                                into a file
     * @param reportWriterName     the name of the entities which is signaled
     *                             to write any specific report just before
     *                             termination of the simulation
     * @see gridsim.GridSimShutdown
     * @pre numUser >= 0
     * @post $none
     */
    public static void init(int numUser, Calendar cal, boolean traceFlag,
            String[] excludeFromFile, String[] excludeFromProcessing,
            String reportWriterName)
    {
        Sim_system.initialise();
        Sim_system.set_auto_trace(traceFlag);
        Sim_system.set_trc_level(1);

        // Set the current Wall clock time as the starting time of simulation
        calendar_ = cal;
        if (cal == null) {
            calendar_ = Calendar.getInstance();
        }

        SimulationStartDate = calendar_.getTime();
        rand = new GridSimRandom();

        try
        {
            new GridStatistics("GridStatistics", "GridSim_stat.txt", true,
                    excludeFromFile, excludeFromProcessing);

            new GridSimShutdown("GridSimShutdown", numUser, reportWriterName);

            gis_ = new GridInformationService("GridInformationService",
                                              GridSimTags.DEFAULT_BAUD_RATE);
        }
        catch (Exception e)
        {
            System.out.println("GridSim.init(): Unwanted errors happen");
            System.out.println( e.getMessage() );
        }
    }

    /**
     * Starts the execution of GridSim simulation.
     * It waits for complete execution of all entitities, i.e. until
     * all entities threads reach non-RUNNABLE state by exiting from the
     * <tt>body()</tt> method. Then, it kills threads of all entities.
     * <p>
     * <b>Note</b>: This method should be called after all the entities
     *              have been setup and added, and their ports are linked.
     * @deprecated As of GridSim 2.1, replaced by {@link #startGridSimulation()}
     * @throws NullPointerException This happens when creating this entity
     *               before initializing GridSim package or this entity name is
     *               <tt>null</tt> or empty
     * @see gridsim.GridSim#init(int, Calendar, boolean, String[], String[],
     *          String)
     * @pre $none
     * @post $none
     */
    public static void Start() throws NullPointerException {
        startGridSimulation();
    }

    /**
     * Starts the execution of GridSim simulation.
     * It waits for complete execution of all entitities, i.e. until
     * all entities threads reach non-RUNNABLE state by exiting from the
     * <tt>body()</tt> method. Then, it kills threads of all entities.
     * <p>
     * <b>Note</b>: This method should be called after all the entities
     *              have been setup and added, and their ports are linked.
     * @throws NullPointerException This happens when creating this entity
     *              before initializing GridSim package or this entity name is
     *              <tt>null</tt> or empty
     * @see gridsim.GridSim#init(int, Calendar, boolean, String[], String[],
     *          String)
     * @pre $none
     * @post $none
     */
    public static void startGridSimulation() throws NullPointerException
    {
        System.out.println("Starting GridSim version 2.2");
        try {
            Sim_system.run();
        }
        catch (Sim_exception e)
        {
            throw new NullPointerException("GridSim.startGridSimulation() :" +
                    " Error - you haven't initialized GridSim.");
        }
    }

    /**
     * Gets the current simulation time (based on SimJava simulation clock)
     * @return The current simulation time from the simulation clock
     * @deprecated As of GridSim 2.1, replaced by {@link #clock()}
     * @see eduni.simjava.Sim_system#clock()
     * @pre $none
     * @post $result >= 0.0
     */
    public static double Clock() {
        return clock();
    }

    /**
     * Gets the current simulation time (based on SimJava simulation clock)
     * @return The current simulation time from the simulation clock
     * @see eduni.simjava.Sim_system#clock()
     * @pre $none
     * @post $result >= 0.0
     */
    public static double clock() {
        return Sim_system.clock();
    }

    /**
     * Causes the entity to hold for <tt>duration</tt> units of simulation time
     * @param duration the amount of time to hold
     * @deprecated As of GridSim 2.1, replaced by {@link #gridSimHold(double)}
     * @pre $none
     * @post $none
     */
    public void GridSimHold(double duration) {
        gridSimHold(duration);
    }

    /**
     * Causes the entity to hold for <tt>duration</tt> units of simulation time
     * @param duration the amount of time to hold
     * @pre $none
     * @post $none
     */
    public void gridSimHold(double duration)
    {
        // if duration is -ve, then no use to hold -ve time.
        if (duration < 0.0) {
            return;
        }

        // TODO: Below is deprecated for SimJava 2
        //super.sim_hold(duration);
        super.sim_process(duration);
    }

    /**
     * Stops Grid Simulation (based on SimJava Sim_system.run_stop()).
     * This should be ony called if any of the user defined entities
     * <b>explicitly</b> want
     * to terminate simulation during execution.
     * @see eduni.simjava.Sim_system#run_stop()
     * @deprecated As of GridSim 2.1, replaced by {@link #stopGridSimulation()}
     * @throws NullPointerException This happens when creating this entity
     *               before initializing GridSim package or this entity name is
     *               <tt>null</tt> or empty
     * @see gridsim.GridSim#init(int, Calendar, boolean, String[], String[],
     *          String)
     * @pre $none
     * @post $none
     */
    public static void Stop() throws NullPointerException {
        stopGridSimulation();
    }

    /**
     * Stops Grid Simulation (based on SimJava Sim_system.run_stop()).
     * This should be ony called if any of the user defined entities
     * <b>explicitly</b> want
     * to terminate simulation during execution.
     * @throws NullPointerException This happens when creating this entity
     *                before initializing GridSim package or this entity name is
     *                <tt>null</tt> or empty
     * @see gridsim.GridSim#init(int, Calendar, boolean, String[], String[],
     *          String)
     * @see eduni.simjava.Sim_system#run_stop()
     * @pre $none
     * @post $none
     */
    public static void stopGridSimulation() throws NullPointerException
    {
        try {
            Sim_system.run_stop();
        }
        catch (Sim_exception e)
        {
            throw new NullPointerException("GridSim.stopGridSimulation() : " +
                    "Error - can't stop Grid Simulation.");
        }
    }

    ////////////  SIMULATION CONTROL METHODS END ////////////


    ////////////  ENTITY COMMUNICATION METHODS START ////////////

    /**
     * Sends an event/message to another entity by <tt>delaying</tt>
     * the simulation time
     * from the current time, with a tag representing the event type.
     * <p>
     * It is <tt>recommended</tt> to use <tt>send()</tt> method with
     * an output port if the network bandwidth plays an important role in
     * this simulation. However, the entity must have the network entities, i.e.
     * <tt>Input</tt> and <tt>Output</tt> port (specified during the creation
     * of the entity by giving a baud rate or bandwidth speed).
     * Below is an example on how to do:
     * <p>
     * <code>
     * ... // other code <br>
     * // object is the entity or message you want to send <br>
     * // size is the object size in bytes (rough estimation) <br>
     * // destination id is the entity ID you want to send the object to <br>
     * IO_data data = new IO_data(object, size, destinationID); <br>
     * <br> <br>
     * // If this entity extends from GridSim class, then you should use it <br>
     * // otherwise need to create a new <tt>Sim_port</tt> object <br>
     * Sim_port port = super.output; <br> <br>
     * // delay is the simulation time delay <br>
     * // tag is the event type (user-defined or choose one from
     *    <tt>GridSimTags</tt> class)<br>
     * send(port, delay, tag, data);<br>
     * ... // remaining other code <br>
     * </code>
     *
     * @param entityName the name of the destination entity
     * @param delay      how long the event/message should be delivered
     *                   from the current simulation time.
     *                   If delay is a negative number, then it will be
     *                   changed to 0.0
     * @param gridSimTag an user-defined number representing the type of
     *                   an event/message
     * @deprecated As of GridSim 2.1, replaced by {@link #send(String, double,
     *             int)}
     * @pre entityName != null
     * @pre delay >= 0.0
     * @post $none
     */
    protected void Send(String entityName, double delay, int gridSimTag) {
        send(entityName, delay, gridSimTag);
    }

    /**
     * Sends an event/message to another entity by <tt>delaying</tt>
     * the simulation time
     * from the current time, with a tag representing the event type.
     * <p>
     * It is <tt>recommended</tt> to use <tt>send()</tt> method with
     * an output port if the network bandwidth plays an important role in
     * this simulation. However, the entity must have the network entities, i.e.
     * <tt>Input</tt> and <tt>Output</tt> port (specified during the creation
     * of the entity by giving a baud rate or bandwidth speed).
     * Below is an example on how to do:
     * <p>
     * <code>
     * ... // other code <br>
     * // object is the entity or message you want to send <br>
     * // size is the object size in bytes (rough estimation) <br>
     * // destination id is the entity ID you want to send the object to <br>
     * IO_data data = new IO_data(object, size, destinationID); <br>
     * <br> <br>
     * // If this entity extends from GridSim class, then you should use it <br>
     * // otherwise need to create a new <tt>Sim_port</tt> object <br>
     * Sim_port port = super.output; <br> <br>
     * // delay is the simulation time delay <br>
     * // tag is the event type (user-defined or choose one from
     *    <tt>GridSimTags</tt> class)<br>
     * send(port, delay, tag, data);<br>
     * ... // remaining other code <br>
     * </code>
     *
     * @param entityName the name of the destination entity
     * @param delay      how long the event/message should be delivered
     *                   from the current simulation time
     *                   If delay is a negative number, then it will be
     *                   changed to 0.0
     * @param gridSimTag an user-defined number representing the type of
     *                   an event/message
     * @pre entityName != null
     * @pre delay >= 0.0
     * @post $none
     */
    protected void send(String entityName, double delay, int gridSimTag)
    {
        if (entityName == null) {
            return;
        }

        // if delay is -ve, then it doesn't make sense. So resets to 0.0
        if (delay < 0.0) {
            delay = 0.0;
        }

        super.sim_schedule(GridSim.getEntityId(entityName), delay, gridSimTag);
    }

    /**
     * Sends an event/message to another entity by <tt>delaying</tt>
     * the simulation time
     * from the current time, with a tag representing the event type.
     * <p>
     * It is <tt>recommended</tt> to use <tt>send()</tt> method with
     * an output port if the network bandwidth plays an important role in
     * this simulation. However, the entity must have the network entities, i.e.
     * <tt>Input</tt> and <tt>Output</tt> port (specified during the creation
     * of the entity by giving a baud rate or bandwidth speed).
     * Below is an example on how to do:
     * <p>
     * <code>
     * ... // other code <br>
     * // object is the entity or message you want to send <br>
     * // size is the object size in bytes (rough estimation) <br>
     * // destination id is the entity ID you want to send the object to <br>
     * IO_data data = new IO_data(object, size, destinationID); <br>
     * <br> <br>
     * // If this entity extends from GridSim class, then you should use it <br>
     * // otherwise need to create a new <tt>Sim_port</tt> object <br>
     * Sim_port port = super.output; <br> <br>
     * // delay is the simulation time delay <br>
     * // tag is the event type (user-defined or choose one from
     *    <tt>GridSimTags</tt> class)<br>
     * send(port, delay, tag, data);<br>
     * ... // remaining other code <br>
     * </code>
     *
     * @param entityName the name of the destination entity
     * @param delay      how long the event/message should be delivered
     *                   from the current simulation time
     *                   If delay is a negative number, then it will be
     *                   changed to 0.0
     * @param gridSimTag an user-defined number representing the type of
     *                   an event/message
     * @param data       A reference to data to be sent with the event
     * @deprecated As of GridSim 2.1, replaced by {@link #send(String, double,
     *             int, Object)}
     * @pre entityName != null
     * @pre delay >= 0.0
     * @pre data != null
     * @post $none
     */
    protected void Send(String entityName, double delay, int gridSimTag,
            Object data)
    {
        send(entityName, delay, gridSimTag, data);
    }

    /**
     * Sends an event/message to another entity by <tt>delaying</tt>
     * the simulation time
     * from the current time, with a tag representing the event type.
     * <p>
     * It is <tt>recommended</tt> to use <tt>send()</tt> method with
     * an output port if the network bandwidth plays an important role in
     * this simulation. However, the entity must have the network entities, i.e.
     * <tt>Input</tt> and <tt>Output</tt> port (specified during the creation
     * of the entity by giving a baud rate or bandwidth speed).
     * Below is an example on how to do:
     * <p>
     * <code>
     * ... // other code <br>
     * // object is the entity or message you want to send <br>
     * // size is the object size in bytes (rough estimation) <br>
     * // destination id is the entity ID you want to send the object to <br>
     * IO_data data = new IO_data(object, size, destinationID); <br>
     * <br> <br>
     * // If this entity extends from GridSim class, then you should use it <br>
     * // otherwise need to create a new <tt>Sim_port</tt> object <br>
     * Sim_port port = super.output; <br> <br>
     * // delay is the simulation time delay <br>
     * // tag is the event type (user-defined or choose one from
     *    <tt>GridSimTags</tt> class)<br>
     * send(port, delay, tag, data);<br>
     * ... // remaining other code <br>
     * </code>
     *
     * @param entityName the name of the destination entity
     * @param delay      how long the event/message should be delivered
     *                   from the current simulation time
     *                   If delay is a negative number, then it will be
     *                   changed to 0.0
     * @param gridSimTag an user-defined number representing the type of
     *                   an event/message
     * @param data       A reference to data to be sent with the event
     * @pre entityName != null
     * @pre delay >= 0.0
     * @pre data != null
     * @post $none
     */
    protected void send(String entityName, double delay, int gridSimTag,
            Object data)
    {
        if (entityName == null) {
            return;
        }

        // if delay is -ve, then it doesn't make sense. So resets to 0.0
        if (delay < 0.0) {
            delay = 0.0;
        }

        super.sim_schedule(GridSim.getEntityId(entityName), delay, gridSimTag,
                data);
    }

    /**
     * Sends an event/message to another entity by <tt>delaying</tt>
     * the simulation time
     * from the current time, with a tag representing the event type.
     * <p>
     * It is <tt>recommended</tt> to use <tt>send()</tt> method with
     * an output port if the network bandwidth plays an important role in
     * this simulation. However, the entity must have the network entities, i.e.
     * <tt>Input</tt> and <tt>Output</tt> port (specified during the creation
     * of the entity by giving a baud rate or bandwidth speed).
     * Below is an example on how to do:
     * <p>
     * <code>
     * ... // other code <br>
     * // object is the entity or message you want to send <br>
     * // size is the object size in bytes (rough estimation) <br>
     * // destination id is the entity ID you want to send the object to <br>
     * IO_data data = new IO_data(object, size, destinationID); <br>
     * <br> <br>
     * // If this entity extends from GridSim class, then you should use it <br>
     * // otherwise need to create a new <tt>Sim_port</tt> object <br>
     * Sim_port port = super.output; <br> <br>
     * // delay is the simulation time delay <br>
     * // tag is the event type (user-defined or choose one from
     *    <tt>GridSimTags</tt> class)<br>
     * send(port, delay, tag, data);<br>
     * ... // remaining other code <br>
     * </code>
     *
     * @param entityID   the id number of the destination entity
     * @param delay      how long the event/message should be delivered
     *                   from the current simulation time
     *                   If delay is a negative number, then it will be
     *                   changed to 0.0
     * @param gridSimTag an user-defined number representing the type of
     *                   an event/message
     * @deprecated As of GridSim 2.1, replaced by {@link #send(int, double,
     *             int)}
     * @pre entityID > 0
     * @pre delay >= 0.0
     * @post $none
     */
    protected void Send(int entityID, double delay, int gridSimTag) {
        send(entityID, delay, gridSimTag);
    }

    /**
     * Sends an event/message to another entity by <tt>delaying</tt>
     * the simulation time
     * from the current time, with a tag representing the event type.
     * <p>
     * It is <tt>recommended</tt> to use <tt>send()</tt> method with
     * an output port if the network bandwidth plays an important role in
     * this simulation. However, the entity must have the network entities, i.e.
     * <tt>Input</tt> and <tt>Output</tt> port (specified during the creation
     * of the entity by giving a baud rate or bandwidth speed).
     * Below is an example on how to do:
     * <p>
     * <code>
     * ... // other code <br>
     * // object is the entity or message you want to send <br>
     * // size is the object size in bytes (rough estimation) <br>
     * // destination id is the entity ID you want to send the object to <br>
     * IO_data data = new IO_data(object, size, destinationID); <br>
     * <br> <br>
     * // If this entity extends from GridSim class, then you should use it <br>
     * // otherwise need to create a new <tt>Sim_port</tt> object <br>
     * Sim_port port = super.output; <br> <br>
     * // delay is the simulation time delay <br>
     * // tag is the event type (user-defined or choose one from
     *    <tt>GridSimTags</tt> class)<br>
     * send(port, delay, tag, data);<br>
     * ... // remaining other code <br>
     * </code>
     *
     * @param entityID   the id number of the destination entity
     * @param delay      how long the event/message should be delivered
     *                   from the current simulation time
     *                   If delay is a negative number, then it will be
     *                   changed to 0.0
     * @param gridSimTag an user-defined number representing the type of
     *                   an event/message
     * @pre entityID > 0
     * @pre delay >= 0.0
     * @post $none
     */
    protected void send(int entityID, double delay, int gridSimTag)
    {
        if (entityID < 0) {
            return;
        }

        // if delay is -ve, then it doesn't make sense. So resets to 0.0
        if (delay < 0.0) {
            delay = 0.0;
        }

        super.sim_schedule(entityID, delay, gridSimTag);
    }

    /**
     * Sends an event/message to another entity by <tt>delaying</tt>
     * the simulation time
     * from the current time, with a tag representing the event type.
     * <p>
     * It is <tt>recommended</tt> to use <tt>send()</tt> method with
     * an output port if the network bandwidth plays an important role in
     * this simulation. However, the entity must have the network entities, i.e.
     * <tt>Input</tt> and <tt>Output</tt> port (specified during the creation
     * of the entity by giving a baud rate or bandwidth speed).
     * Below is an example on how to do:
     * <p>
     * <code>
     * ... // other code <br>
     * // object is the entity or message you want to send <br>
     * // size is the object size in bytes (rough estimation) <br>
     * // destination id is the entity ID you want to send the object to <br>
     * IO_data data = new IO_data(object, size, destinationID); <br>
     * <br> <br>
     * // If this entity extends from GridSim class, then you should use it <br>
     * // otherwise need to create a new <tt>Sim_port</tt> object <br>
     * Sim_port port = super.output; <br> <br>
     * // delay is the simulation time delay <br>
     * // tag is the event type (user-defined or choose one from
     *    <tt>GridSimTags</tt> class)<br>
     * send(port, delay, tag, data);<br>
     * ... // remaining other code <br>
     * </code>
     *
     * @param entityID   the id number of the destination entity
     * @param delay      how long the event/message should be delivered
     *                   from the current simulation time
     *                   If delay is a negative number, then it will be
     *                   changed to 0.0
     * @param gridSimTag an user-defined number representing the type of
     *                   an event/message
     * @param data       A reference to data to be sent with the event
     * @deprecated As of GridSim 2.1, replaced by {@link #send(int, double,
     *             int, Object)}
     * @pre entityID > 0
     * @pre delay >= 0.0
     * @pre data != null
     * @post $none
     */
    protected void Send(int entityID, double delay, int gridSimTag, Object data)
    {
        send(entityID, delay, gridSimTag, data);
    }

    /**
     * Sends an event/message to another entity by <tt>delaying</tt>
     * the simulation time
     * from the current time, with a tag representing the event type.
     * <p>
     * It is <tt>recommended</tt> to use <tt>send()</tt> method with
     * an output port if the network bandwidth plays an important role in
     * this simulation. However, the entity must have the network entities, i.e.
     * <tt>Input</tt> and <tt>Output</tt> port (specified during the creation
     * of the entity by giving a baud rate or bandwidth speed).
     * Below is an example on how to do:
     * <p>
     * <code>
     * ... // other code <br>
     * // object is the entity or message you want to send <br>
     * // size is the object size in bytes (rough estimation) <br>
     * // destination id is the entity ID you want to send the object to <br>
     * IO_data data = new IO_data(object, size, destinationID); <br>
     * <br> <br>
     * // If this entity extends from GridSim class, then you should use it <br>
     * // otherwise need to create a new <tt>Sim_port</tt> object <br>
     * Sim_port port = super.output; <br> <br>
     * // delay is the simulation time delay <br>
     * // tag is the event type (user-defined or choose one from
     *    <tt>GridSimTags</tt> class)<br>
     * send(port, delay, tag, data);<br>
     * ... // remaining other code <br>
     * </code>
     *
     * @param entityID   the id number of the destination entity
     * @param delay      how long the event/message should be delivered
     *                   from the current simulation time
     *                   If delay is a negative number, then it will be
     *                   changed to 0.0
     * @param gridSimTag an user-defined number representing the type of
     *                   an event/message
     * @param data       A reference to data to be sent with the event
     * @pre entityID > 0
     * @pre delay >= 0.0
     * @pre data != null
     * @post $none
     */
    protected void send(int entityID, double delay, int gridSimTag, Object data)
    {
        if (entityID < 0) {
            return;
        }

        // if delay is -ve, then it doesn't make sense. So resets to 0.0
        if (delay < 0.0) {
            delay = 0.0;
        }

        super.sim_schedule(entityID, delay, gridSimTag, data);
    }

    /**
     * Sends an event/message to another entity by <tt>delaying</tt>
     * the simulation time
     * from the current time, with a tag representing the event type.
     * <p>
     * It is <tt>recommended</tt> to use <tt>send()</tt> method with
     * an output port if the network bandwidth plays an important role in
     * this simulation. However, the entity must have the network entities, i.e.
     * <tt>Input</tt> and <tt>Output</tt> port (specified during the creation
     * of the entity by giving a baud rate or bandwidth speed).
     * Below is an example on how to do:
     * <p>
     * <code>
     * ... // other code <br>
     * // object is the entity or message you want to send <br>
     * // size is the object size in bytes (rough estimation) <br>
     * // destination id is the entity ID you want to send the object to <br>
     * IO_data data = new IO_data(object, size, destinationID); <br>
     * <br> <br>
     * // If this entity extends from GridSim class, then you should use it <br>
     * // otherwise need to create a new <tt>Sim_port</tt> object <br>
     * Sim_port port = super.output; <br> <br>
     * // delay is the simulation time delay <br>
     * // tag is the event type (user-defined or choose one from
     *    <tt>GridSimTags</tt> class)<br>
     * send(port, delay, tag, data);<br>
     * ... // remaining other code <br>
     * </code>
     *
     * @param destPort   A reference to the port to send the event out of
     * @param delay      how long the event/message should be delivered
     *                   from the current simulation time
     *                   If delay is a negative number, then it will be
     *                   changed to 0.0
     * @param gridSimTag an user-defined number representing the type of
     *                   an event/message
     * @deprecated As of GridSim 2.1, replaced by {@link #send(Sim_port, double,
     *             int)}
     * @pre destPort != null
     * @pre delay >= 0.0
     * @post $none
     */
    protected void Send(Sim_port destPort, double delay, int gridSimTag) {
        send(destPort, delay, gridSimTag);
    }

    /**
     * Sends an event/message to another entity by <tt>delaying</tt>
     * the simulation time
     * from the current time, with a tag representing the event type.
     * <p>
     * It is <tt>recommended</tt> to use <tt>send()</tt> method with
     * an output port if the network bandwidth plays an important role in
     * this simulation. However, the entity must have the network entities, i.e.
     * <tt>Input</tt> and <tt>Output</tt> port (specified during the creation
     * of the entity by giving a baud rate or bandwidth speed).
     * Below is an example on how to do:
     * <p>
     * <code>
     * ... // other code <br>
     * // object is the entity or message you want to send <br>
     * // size is the object size in bytes (rough estimation) <br>
     * // destination id is the entity ID you want to send the object to <br>
     * IO_data data = new IO_data(object, size, destinationID); <br>
     * <br> <br>
     * // If this entity extends from GridSim class, then you should use it <br>
     * // otherwise need to create a new <tt>Sim_port</tt> object <br>
     * Sim_port port = super.output; <br> <br>
     * // delay is the simulation time delay <br>
     * // tag is the event type (user-defined or choose one from
     *    <tt>GridSimTags</tt> class)<br>
     * send(port, delay, tag, data);<br>
     * ... // remaining other code <br>
     * </code>
     *
     * @param destPort   A reference to the port to send the event out of
     * @param delay      how long the event/message should be delivered
     *                   from the current simulation time
     *                   If delay is a negative number, then it will be
     *                   changed to 0.0
     * @param gridSimTag an user-defined number representing the type of
     *                   an event/message
     * @pre destPort != null
     * @pre delay >= 0.0
     * @post $none
     */
    protected void send(Sim_port destPort, double delay, int gridSimTag)
    {
        if (destPort == null) {
            return;
        }

        // if delay is -ve, then it doesn't make sense. So resets to 0.0
        if (delay < 0.0) {
            delay = 0.0;
        }

        super.sim_schedule(destPort, delay, gridSimTag);
    }

    /**
     * Sends an event/message to another entity by <tt>delaying</tt>
     * the simulation time
     * from the current time, with a tag representing the event type.
     * <p>
     * It is <tt>recommended</tt> to use <tt>send()</tt> method with
     * an output port if the network bandwidth plays an important role in
     * this simulation. However, the entity must have the network entities, i.e.
     * <tt>Input</tt> and <tt>Output</tt> port (specified during the creation
     * of the entity by giving a baud rate or bandwidth speed).
     * Below is an example on how to do:
     * <p>
     * <code>
     * ... // other code <br>
     * // object is the entity or message you want to send <br>
     * // size is the object size in bytes (rough estimation) <br>
     * // destination id is the entity ID you want to send the object to <br>
     * IO_data data = new IO_data(object, size, destinationID); <br>
     * <br> <br>
     * // If this entity extends from GridSim class, then you should use it <br>
     * // otherwise need to create a new <tt>Sim_port</tt> object <br>
     * Sim_port port = super.output; <br> <br>
     * // delay is the simulation time delay <br>
     * // tag is the event type (user-defined or choose one from
     *    <tt>GridSimTags</tt> class)<br>
     * send(port, delay, tag, data);<br>
     * ... // remaining other code <br>
     * </code>
     *
     * @param destPort  A reference to the port to send the event out of
     * @param delay      how long the event/message should be delivered
     *                   from the current simulation time
     *                   If delay is a negative number, then it will be
     *                   changed to 0.0
     * @param gridSimTag an user-defined number representing the type of
     *                   an event/message
     * @param data       A reference to data to be sent with the event
     * @deprecated As of GridSim 2.1, replaced by {@link #send(Sim_port, double,
     *             int, Object)}
     * @pre destPort != null
     * @pre delay >= 0.0
     * @pre data != null
     * @post $none
     */
    protected void Send(Sim_port destPort, double delay, int gridSimTag,
            Object data)
    {
        send(destPort, delay, gridSimTag, data);
    }

    /**
     * Sends an event/message to another entity by <tt>delaying</tt>
     * the simulation time
     * from the current time, with a tag representing the event type.
     * <p>
     * It is <tt>recommended</tt> to use <tt>send()</tt> method with
     * an output port if the network bandwidth plays an important role in
     * this simulation. However, the entity must have the network entities, i.e.
     * <tt>Input</tt> and <tt>Output</tt> port (specified during the creation
     * of the entity by giving a baud rate or bandwidth speed).
     * Below is an example on how to do:
     * <p>
     * <code>
     * ... // other code <br>
     * // object is the entity or message you want to send <br>
     * // size is the object size in bytes (rough estimation) <br>
     * // destination id is the entity ID you want to send the object to <br>
     * IO_data data = new IO_data(object, size, destinationID); <br>
     * <br> <br>
     * // If this entity extends from GridSim class, then you should use it <br>
     * // otherwise need to create a new <tt>Sim_port</tt> object <br>
     * Sim_port port = super.output; <br> <br>
     * // delay is the simulation time delay <br>
     * // tag is the event type (user-defined or choose one from
     *    <tt>GridSimTags</tt> class)<br>
     * send(port, delay, tag, data);<br>
     * ... // remaining other code <br>
     * </code>
     *
     * @param destPort  A reference to the port to send the event out of
     * @param delay      how long the event/message should be delivered
     *                   from the current simulation time
     *                   If delay is a negative number, then it will be
     *                   changed to 0.0
     * @param gridSimTag an user-defined number representing the type of
     *                   an event/message
     * @param data       A reference to data to be sent with the event
     * @pre destPort != null
     * @pre delay >= 0.0
     * @pre data != null
     * @post $none
     */
    protected void send(Sim_port destPort, double delay, int gridSimTag,
            Object data)
    {
        if (destPort == null) {
            return;
        }

        // if delay is -ve, then it doesn't make sense. So resets to 0.0
        if (delay < 0.0) {
            delay = 0.0;
        }

        super.sim_schedule(destPort, delay, gridSimTag, data);
    }

    /**
     * Gets the data passed in this event
     * @return A reference to the data
     * @deprecated As of GridSim 2.1, replaced by {@link #receiveEventObject()}
     * @pre $none
     * @post $result != null
     */
    protected Object ReceiveEventObject() {
        return receiveEventObject();
    }

    /**
     * Gets the data passed in this event
     * @return A reference to the data
     * @pre $none
     * @post $result != null
     */
    protected Object receiveEventObject()
    {
        super.sim_get_next(ev_);
        return ev_.get_data();
    }

    /**
     * Gets the data passed in this event
     * @param sourcePort a Sim_port object which is used to connect entities
     *                    for  event passing
     * @return A reference to the data or <tt>null</tt> if the source port is
     *         empty
     * @see eduni.simjava.Sim_port
     * @pre sourcePort != null
     * @post $none
     */
    protected Object ReceiveEventObject(Sim_port sourcePort) {
        return receiveEventObject(sourcePort);
    }

    /**
     * Gets the data passed in this event
     * @param sourcePort a Sim_port object which is used to connect entities
     *                    for  event passing
     * @return A reference to the data or <tt>null</tt> if the source port is
     *         empty
     * @see eduni.simjava.Sim_port
     * @pre sourcePort != null
     * @post $none
     */
    protected Object receiveEventObject(Sim_port sourcePort)
    {
        if (sourcePort == null) {
            return null;
        }

        super.sim_get_next( new Sim_from_port(sourcePort), ev_ );
        return ev_.get_data();
    }

    /**
     * Sends a Gridlet to the destination GridResource ID <tt>without</tt>
     * any delay and <tt>without</tt> an acknowledgement.
     * @param gl       a Gridlet object to be sent
     * @param resID    an unique resource ID
     * @return <tt>true</tt> if this Gridlet has been submitted to the
     *         destination GridResource, <tt>false</tt> otherwise.
     *         Submitting a Gridlet can be failed for the one or more
     *         following reasons:
     *         <ul>
     *              <li> if a GridResource ID doesn't exist
     *              <li> if a Gridlet ID doesn't exist
     *              <li> if a Gridlet's user ID doesn't exist
     *              <li> if a Gridlet object is <tt>null</tt> or empty;
     *              <li> if a Gridlet object has <tt>finished</tt> executing
     *                   beforehand
     *         </ul>
     *
     * @deprecated As of GridSim 2.1, replaced by {@link #gridletSubmit(Gridlet,
     *             int)}
     * @pre gl != null
     * @pre resID >= 0
     * @post $none
     */
    protected boolean GridletSubmit(Gridlet gl, int resID) {
        return gridletSubmit(gl, resID);
    }

    /**
     * Sends a Gridlet to the destination GridResource ID <tt>without</tt>
     * any delay and <tt>without</tt> an acknowledgement.
     * @param gl       a Gridlet object to be sent
     * @param resID    an unique resource ID
     * @return <tt>true</tt> if this Gridlet has been submitted to the
     *         destination GridResource, <tt>false</tt> otherwise.
     *         Submitting a Gridlet can be failed for the one or more
     *         following reasons:
     *         <ul>
     *              <li> if a GridResource ID doesn't exist
     *              <li> if a Gridlet ID doesn't exist
     *              <li> if a Gridlet's user ID doesn't exist
     *              <li> if a Gridlet object is <tt>null</tt> or empty;
     *              <li> if a Gridlet object has <tt>finished</tt> executing
     *                   beforehand
     *         </ul>
     * @pre gl != null
     * @pre resID >= 0
     * @post $none
     */
    protected boolean gridletSubmit(Gridlet gl, int resID) {
        return gridletSubmit(gl, resID, GridSimTags.SCHEDULE_NOW, false);
    }

    /**
     * Sends a Gridlet to the destination GridResource ID <tt>with</tt>
     * a specified delay.
     * @param gl            a Gridlet object to be sent
     * @param resourceID    an unique resource ID
     * @param delay         delay time or <tt>0.0</tt> if want to execute NOW
     * @param ack           an acknowledgment status. <tt>true</tt> if want to
     *                      know the result of this method, <tt>false</tt>
     *                      otherwise or don't care.
     * @return <tt>true</tt> if this Gridlet has been submitted to the
     *         destination GridResource, <tt>false</tt> otherwise.
     *         Submitting a Gridlet can be failed for the one or more
     *         following reasons:
     *         <ul>
     *              <li> if the acknowledgment status in the parameter of this
     *                   method is set to <tt>false</tt>
     *              <li> if a GridResource ID doesn't exist
     *              <li> if a Gridlet ID doesn't exist
     *              <li> if a Gridlet's user ID doesn't exist
     *              <li> if the delay time is negative
     *              <li> if a Gridlet object is <tt>null</tt> or empty;
     *              <li> if a Gridlet object has <tt>finished</tt> executing
     *                   beforehand
     *         </ul>
     * @pre gl != null
     * @pre resourceID >= 0
     * @pre delay >= 0.0
     * @post $none
     */
    protected boolean gridletSubmit(Gridlet gl, int resourceID,
                                    double delay, boolean ack)
    {
        if (gl == null || delay < 0.0) {
            return false;
        }

        boolean valid = false;
        try
        {
            String errorMsg = "GridSim.gridletSubmit(): ";
            valid = validateValue(errorMsg, gl.getGridletID(),
                                  gl.getUserID(), resourceID);

            // if any of the above value invalid, then return
            if (valid == false) {
                return false;
            }

            // sends the gridlet to a destination GridResource id with ACK
            if (ack == true)
            {
                send(this.output, delay, GridSimTags.GRIDLET_SUBMIT_ACK,
                     new IO_data(gl, gl.getGridletFileSize(), resourceID)
                );

                // waiting for a response back from the GridResource
                Sim_type_p tag = new Sim_type_p(GridSimTags.GRIDLET_SUBMIT_ACK
                                                + gl.getGridletID() );

                // only look for ack for same Gridlet ID
                super.sim_get_next(tag, ev_);
                System.out.println("GridSim.gridletSubmit(): " +
                        ev_.get_data());
                valid = ( (Boolean) ev_.get_data() ).booleanValue();
            }
            else   // sends without ACK
            {
                valid = false;
                send(this.output, delay, GridSimTags.GRIDLET_SUBMIT,
                     new IO_data(gl, gl.getGridletFileSize(), resourceID)
                );
            }
        }
        catch (Sim_exception sim)
        {
            valid = false;
            System.out.println(
                "GridSim.gridletSubmit(): Error from SimJava occurs.");
            System.out.println( sim.getMessage() );
        }
        catch (Exception e)
        {
            valid = false;
            System.out.println("GridSim.gridletSubmit(): Error occurs.");
            System.out.println( e.getMessage() );
        }

        return valid;
    }

    /**
     * Gets a Gridlet object passed in this event
     * @return A reference to Gridlet object
     * @deprecated As of GridSim 2.1, replaced by {@link #gridletReceive()}
     * @pre $none
     * @post $result != null
     */
    protected Gridlet GridletReceive() {
        return gridletReceive();
    }

    /**
     * Gets a Gridlet object passed in this event
     * @return A reference to Gridlet object
     * @pre $none
     * @post $result != null
     */
    protected Gridlet gridletReceive() {
        return (Gridlet) receiveEventObject(this.input);
    }

    /**
     * Sends a Gridlet based on their event tag to the destination resource ID.
     * @param errorMsg   a message containing which operation it belongs to
     * @param gridletId  a Gridlet ID
     * @param userId     a Gridlet's user or owner ID
     * @param resourceId    a GridResource ID
     * @param delay      sending delay time
     * @param tag        event tag (such as GridSimTags.GRIDLET_PAUSE, etc...)
     * @param ack        denotes whether want to have an acknowledgment or not
     * @return <tt>true</tt> if the Gridlet has been sent successfully,
     *         <tt>false</tt> otherwise
     * @pre errorMsg != null
     * @pre gridletId > 0
     * @pre userId > 0
     * @pre resourceId > 0
     * @pre delay >= 0.0
     * @pre tag > 0
     * @post $result = true || false
     */
    private boolean sendGridlet(String errorMsg, int gridletId, int userId,
                           int resourceId, double delay, int tag, boolean ack)
    {
        boolean valid = validateValue(errorMsg, gridletId, userId, resourceId);
        if (valid == false || delay < 0.0) {
            return false;
        }

        int size = 14;  // size of having 3 ints + 2 bytes overhead
        dataArray_[0] = gridletId;
        dataArray_[1] = userId;
        dataArray_[2] = NOT_FOUND;  // this index is only used by gridletMove()

        // if an ack is required, then change the tag
        int newTag = tag;
        if (ack == true)
        {
            switch (tag)
            {
                case GridSimTags.GRIDLET_PAUSE:
                    newTag = GridSimTags.GRIDLET_PAUSE_ACK;
                    break;

                case GridSimTags.GRIDLET_RESUME:
                    newTag = GridSimTags.GRIDLET_RESUME_ACK;
                    break;

                default:
                    break;
            }
        }

        // send this Gridlet
        send(this.output, delay, newTag,
             new IO_data(dataArray_, size, resourceId)
        );

        return true;
    }

    /**
     * Performs validation of the given parameters before sending a Gridlet
     * to a GridResource entity
     * @param msg        a message containing which operation it belongs to
     * @param gridletId  a Gridlet ID
     * @param userId     a Gridlet's user or owner ID
     * @param resourceId    a GridResource ID
     * @return <tt>true</tt> if the validation has passed successfully,
     *         <tt>false</tt> otherwise
     * @pre msg != null
     * @pre gridletId > 0
     * @pre userId > 0
     * @pre resourceId > 0
     * @post $result = true || false
     */
    private boolean validateValue(String msg, int gridletId, int userId,
                int resourceId)
    {
        boolean valid = true;

        // Gridlet ID must be 0 or positive
        if (gridletId < 0)
        {
            valid = false;
            System.out.println(msg + "Error - Gridlet ID must be >= 0, not " +
                    gridletId);
        }

        // User ID must be 0 or positive
        if (userId < 0)
        {
            valid = false;
            System.out.println(msg + "Error - User ID must be >= 0, not " +
                    userId);
        }

        // GridResource ID must be 0 or positive
        if (resourceId < 0)
        {
            valid = false;
            System.out.println(msg +
                    "Error - GridResource ID must be >= 0, not " + resourceId);
        }

        // if a grid resource ID doesn't exist in GIS list
        if (gis_.isResourceExist(resourceId) == false)
        {
            valid = false;
            System.out.println(msg + "Error - GridResource ID #" + resourceId +
                               " doesn't exist");
        }

        return valid;
    }

    /**
     * Cancels a Gridlet that is currently executing in a given GridResource
     * ID <tt>with</tt> a delay. <br>
     * <b>NOTE:</b> Canceling a Gridlet operation can take a long time over a
     *              slow network if the Gridlet size is big.
     * @param gl            a Gridlet object to be canceled
     * @param resourceId    an unique resource ID
     * @param delay         delay time or <tt>0.0</tt> if want to cancel NOW
     * @return the canceled Gridlet or <tt>null</tt if this operation fails.
     *         If a Gridlet has <tt>finished</tt> in time of cancellation, then
     *         this method will return the finished Gridlet.
     *         Canceling a Gridlet can be failed for the one or more
     *         following reasons:
     *         <ul>
     *              <li> if a GridResource ID doesn't exist
     *              <li> if a Gridlet ID doesn't exist
     *              <li> if a Gridlet's user ID doesn't exist
     *              <li> if the delay time is negative
     *              <li> if a Gridlet object is <tt>null</tt> or empty;
     *         </ul>
     * @pre gl != null
     * @pre resourceId >= 0
     * @pre delay >= 0.0
     * @post $none
     */
    protected Gridlet gridletCancel(Gridlet gl, int resourceId, double delay)
    {
        if (gl == null || delay < 0.0) {
            return null;
        }

        Gridlet obj = gridletCancel( gl.getGridletID(), gl.getUserID(),
                                    resourceId, delay );
        return obj;
    }

    /**
     * Cancels a Gridlet that is currently executing in a given GridResource
     * ID <tt>with</tt> a delay. <br>
     * <b>NOTE:</b> Canceling a Gridlet operation can be slow over a slow
     *              network if the Gridlet size is big.
     * @param gridletId     a Gridlet ID
     * @param userId        the user or owner ID of this Gridlet
     * @param resourceId    an unique resource ID to which this Gridlet was
     *                      previously sent to
     * @param delay         delay time or <tt>0.0</tt> if want to cancel NOW
     * @return the canceled Gridlet or <tt>null</tt if this operation fails.
     *         If a Gridlet has <tt>finished</tt> in time of cancellation, then
     *         this method will return the finished Gridlet.
     *         Canceling a Gridlet can be failed for the one or more
     *         following reasons:
     *         <ul>
     *              <li> if a GridResource ID doesn't exist
     *              <li> if a Gridlet ID doesn't exist
     *              <li> if a Gridlet's user ID doesn't exist
     *              <li> if the delay time is negative
     *         </ul>
     * @pre gridletId >= 0
     * @pre userId >= 0
     * @pre resourceId >= 0
     * @pre delay >= 0.0
     * @post $none
     */
    protected Gridlet gridletCancel(int gridletId, int userId, int resourceId,
                                    double delay)
    {
        Gridlet gl = null;
        try
        {
            String errorMsg = "GridSim.gridletCancel(): ";
            boolean valid = sendGridlet(errorMsg, gridletId, userId, resourceId,
                                    delay, GridSimTags.GRIDLET_CANCEL, false);

            if (valid == true)
            {
                // waiting for a response from the GridResource entity
                Sim_type_p tag = new Sim_type_p(GridSimTags.GRIDLET_CANCEL
                                                + gridletId);

                // only look for this type of ack for same Gridlet ID
                super.sim_get_next(tag, ev_);
                gl = (Gridlet) ev_.get_data();
            }
        }
        catch (Sim_exception e)
        {
            gl = null;
            System.out.println("GridSim.gridletCancel(): Error occurs.");
            System.out.println( e.getMessage() );
        }
        catch (Exception e)
        {
            gl = null;
            System.out.println("GridSim.gridletCancel(): Error occurs.");
            System.out.println( e.getMessage() );
        }

        return gl;
    }

    /**
     * Pauses a Gridlet that is currently executing in a given GridResource
     * ID <tt>with</tt> a delay.
     * @param gl            a Gridlet object to be sent
     * @param resId         an unique resource ID
     * @param delay         delay time or <tt>0.0</tt> if want to execute NOW
     * @return <tt>true</tt> if this Gridlet has been paused successfully in the
     *         destination GridResource, <tt>false</tt> otherwise.
     *         Pausing a Gridlet can be failed for the one or more
     *         following reasons:
     *         <ul>
     *              <li> if a GridResource ID doesn't exist
     *              <li> if a Gridlet ID doesn't exist
     *              <li> if a Gridlet's user ID doesn't exist
     *              <li> if the delay time is negative
     *              <li> if a Gridlet object is <tt>null</tt> or empty;
     *              <li> if a Gridlet object has <tt>finished</tt> executing
     *                   beforehand. The Gridlet needs to be retrieved by
     *                   using {@link #gridletReceive()}
     *         </ul>
     * @pre gl != null
     * @pre resId >= 0
     * @pre delay >= 0.0
     * @post $none
     */
    protected boolean gridletPause(Gridlet gl, int resId, double delay)
    {
        if (gl == null || delay < 0.0) {
            return false;
        }

        return gridletPause(gl.getGridletID(),gl.getUserID(),resId,delay,true);
    }

    /**
     * Pauses a Gridlet that is currently executing in a given GridResource
     * ID <tt>with</tt> a delay.
     * @param gridletId     a Gridlet ID
     * @param userId        the user or owner ID of this Gridlet
     * @param resourceId    an unique resource ID
     * @param delay         delay time or <tt>0.0</tt> if want to execute NOW
     * @param ack   an acknowledgement, i.e. <tt>true</tt> if wanted to know
     *        whether this operation is success or not, <tt>false</tt>
     *        otherwise (don't care)
     * @return <tt>true</tt> if this Gridlet has been paused successfully in the
     *         destination GridResource, <tt>false</tt> otherwise.
     *         Pausing a Gridlet can be failed for the one or more
     *         following reasons:
     *         <ul>
     *              <li> if a GridResource ID doesn't exist
     *              <li> if a Gridlet ID doesn't exist
     *              <li> if a Gridlet's user ID doesn't exist
     *              <li> if the delay time is negative
     *              <li> if a Gridlet object has <tt>finished</tt> executing
     *                   beforehand. The Gridlet needs to be retrieved by using
     *                   {@link #gridletReceive()}
     *         </ul>
     * @pre gridletId >= 0
     * @pre userId >= 0
     * @pre resourceId >= 0
     * @pre delay >= 0.0
     * @post $none
     */
    protected boolean gridletPause(int gridletId, int userId, int resourceId,
                                   double delay, boolean ack)
    {
        boolean valid = false;
        try
        {
            String errorMsg = "GridSim.gridletPause(): ";
            valid = sendGridlet(errorMsg, gridletId, userId, resourceId, delay,
                                GridSimTags.GRIDLET_PAUSE, ack);

            if (valid == true && ack == true)
            {
                // waiting for a response from the GridResource
                Sim_type_p tag = new Sim_type_p(GridSimTags.GRIDLET_PAUSE_ACK
                                                + gridletId);

                // only look for this type of ack for same Gridlet ID
                super.sim_get_next(tag, ev_);
                valid = ( (Boolean) ev_.get_data() ).booleanValue();
            }
        }
        catch (Sim_exception e)
        {
            valid = false;
            System.out.println("GridSim.gridletPause(): Error occurs.");
            System.out.println( e.getMessage() );
        }
        catch (Exception e)
        {
            valid = false;
            System.out.println("GridSim.gridletPause(): Error occurs.");
            System.out.println( e.getMessage() );
        }

        return valid;
    }

    /**
     * Gets the current status of this Gridlet in a given GridResource ID
     * @param gl    a Gridlet object
     * @param resourceId   a GridResource ID that executes this Gridlet object
     * @return the current Gridlet status or <tt>-1</tt> if not found.
     *         The various Gridlet status can be found in GridSimTags class.
     * @see gridsim.GridSimTags
     * @pre gl != null
     * @pre resourceId > 0
     * @post $none
     */
    protected int gridletStatus(Gridlet gl, int resourceId)
    {
        if (gl == null) {
            return NOT_FOUND;
        }

        return gridletStatus(gl.getGridletID(), gl.getUserID(), resourceId);
    }

    /**
     * Gets the current status of this Gridlet in a given GridResource ID
     * @param gridletId    a Gridlet ID
     * @param userId       the user or owner of this Gridlet object
     * @param resourceId   a GridResource ID that executes this Gridlet object
     * @return the current Gridlet status or <tt>-1</tt> if not found.
     *         The various Gridlet status can be found in GridSimTags class.
     * @see gridsim.GridSimTags
     * @pre gridletId > 0
     * @pre userId > 0
     * @pre resourceId > 0
     * @post $none
     */
    protected int gridletStatus(int gridletId, int userId, int resourceId)
    {
        int status = NOT_FOUND;
        try
        {
            String errorMsg = "GridSim.gridletStatus(): ";
            boolean valid = sendGridlet(errorMsg, gridletId, userId, resourceId,
                                        0.0, GridSimTags.GRIDLET_STATUS, false);

            if (valid == false) {
                return status;
            }

            // waiting for a response from the GridResource
            Sim_type_p tag = new Sim_type_p(GridSimTags.GRIDLET_STATUS
                                            + gridletId);

            // only look for this type of ack for same Gridlet ID
            super.sim_get_next(tag, ev_);
            status = ( (Integer) ev_.get_data() ).intValue();
        }
        catch (Sim_exception e)
        {
            status = NOT_FOUND;
            System.out.println("GridSim.gridletStatus(): Error occurs.");
            System.out.println( e.getMessage() );
        }
        catch (Exception e)
        {
            status = NOT_FOUND;
            System.out.println("GridSim.gridletStatus(): Error occurs.");
            System.out.println( e.getMessage() );
        }

        return status;
    }

    /**
     * Resumes a Gridlet that is currently pausing in a given GridResource
     * ID <tt>with</tt> a delay.<br>
     * <b>NOTE:</b> Resuming a Gridlet only works if it is currently on paused.
     * @param gl            a Gridlet object to be sent
     * @param resId         an unique resource ID
     * @param delay         delay time or <tt>0.0</tt> if want to execute NOW
     * @return <tt>true</tt> if this Gridlet has been resumed successfully in
     *         the destination GridResource, <tt>false</tt> otherwise.
     *         Resuming a Gridlet can be failed for the one or more
     *         following reasons:
     *         <ul>
     *              <li> if a GridResource ID doesn't exist
     *              <li> if a Gridlet ID doesn't exist
     *              <li> if a Gridlet's user ID doesn't exist
     *              <li> if the delay time is negative
     *              <li> if a Gridlet object is <tt>null</tt> or empty
     *              <li> if a Gridlet is not currently on paused
     *         </ul>
     * @pre gl != null
     * @pre resId >= 0
     * @pre delay >= 0.0
     * @post $none
     */
    protected boolean gridletResume(Gridlet gl, int resId, double delay)
    {
        if (gl == null || delay < 0.0) {
            return false;
        }

        return gridletResume(gl.getGridletID(),gl.getUserID(),resId,delay,true);
    }

    /**
     * Resumes a Gridlet that is currently pausing in a given GridResource
     * ID <tt>with</tt> a delay. <br>
     * <b>NOTE:</b> Resuming a Gridlet only works if it is currently on paused.
     * @param gridletId     a Gridlet ID
     * @param userId        the user or owner ID of this Gridlet
     * @param resourceId    an unique resource ID
     * @param delay         delay time or <tt>0.0</tt> if want to execute NOW
     * @param ack   an acknowledgement, i.e. <tt>true</tt> if wanted to know
     *        whether this operation is success or not, <tt>false</tt>
     *        otherwise (don't care)
     * @return <tt>true</tt> if this Gridlet has been resumed successfully in
     *         the destination GridResource, <tt>false</tt> otherwise.
     *         Resuming a Gridlet can be failed for the one or more
     *         following reasons:
     *         <ul>
     *              <li> if a GridResource ID doesn't exist
     *              <li> if a Gridlet ID doesn't exist
     *              <li> if a Gridlet's user ID doesn't exist
     *              <li> if the delay time is negative
     *              <li> if a Gridlet is not currently on paused
     *         </ul>
     * @pre gridletId >= 0
     * @pre userId >= 0
     * @pre resourceId >= 0
     * @pre delay >= 0.0
     * @post $none
     */
    protected boolean gridletResume(int gridletId, int userId, int resourceId,
                                    double delay, boolean ack)
    {
        boolean valid = false;
        try
        {
            String errorMsg = "GridSim.gridletResume(): ";
            valid = sendGridlet(errorMsg, gridletId, userId, resourceId, delay,
                                GridSimTags.GRIDLET_RESUME, ack);

            if (valid == true && ack == true)
            {
                // waiting for a response from the GridResource
                Sim_type_p tag = new Sim_type_p(GridSimTags.GRIDLET_RESUME_ACK
                                                + gridletId);

                // only look for this type of ack for same Gridlet ID
                super.sim_get_next(tag, ev_);
                valid = ( (Boolean) ev_.get_data() ).booleanValue();
            }
        }
        catch (Sim_exception e)
        {
            valid = false;
            System.out.println("GridSim.gridletResume(): Error occurs.");
            System.out.println( e.getMessage() );
        }
        catch (Exception e)
        {
            valid = false;
            System.out.println("GridSim.gridletResume(): Error occurs.");
            System.out.println( e.getMessage() );
        }

        return valid;
    }

    /**
     * Moves a Gridlet to the destination GridResource ID
     * @param gl   a Gridlet object
     * @param srcId   the GridResource ID that is currently executing this Gridlet
     * @param destId  the new GridResource ID
     * @param delay   simulation delay
     * @return <tt>true</tt> if this Gridlet has moved successfully,
     *         <tt>false</tt> otherwise. Moving a Gridlet can be failed, due
     *         to one or more following reasons:
     *      <ul>
     *         <li> if a Gridlet object is null
     *         <li> if one or both GridResource ID don't exist
     *         <li> if the delay is negative
     *         <li> if a Gridlet in the GridResource has completed beforehand.
     *              The Gridlet can be retrieved by using
     *              {@link #gridletReceive()}
     *      </ul>
     * @pre gl != null
     * @pre srcId > 0
     * @pre destId > 0
     * @pre delay >= 0.0
     * @post $result = true || false
     */
    protected boolean gridletMove(Gridlet gl,int srcId,int destId,double delay)
    {
        if (gl == null || delay < 0.0) {
            return false;
        }

        boolean success = gridletMove(gl.getGridletID(), gl.getUserID(),
                                      srcId, destId, delay, true);

        return success;
    }

    /**
     * Moves a Gridlet to the destination GridResource ID
     * @param gridletId   a Gridlet ID
     * @param userId      the owner or user ID of this Gridlet
     * @param srcId   the GridResource ID that is currently executing this
     *                Gridlet
     * @param destId  the new GridResource ID
     * @param delay   simulation delay
     * @param ack   an acknowledgement, i.e. <tt>true</tt> if wanted to know
     *        whether this operation is success or not, <tt>false</tt>
     *        otherwise (don't care)
     * @return <tt>true</tt> if this Gridlet has moved successfully,
     *         <tt>false</tt> otherwise. Moving a Gridlet can be failed, due
     *         to one or more following reasons:
     *      <ul>
     *         <li> if a Gridlet ID doesn't exist
     *         <li> if the owner of user ID of this Gridlet doesn't exist
     *         <li> if one or both GridResource ID don't exist
     *         <li> if the delay is negative
     *         <li> if a Gridlet in the GridResource has completed beforehand.
     *              The Gridlet can be retrieved by using
     *              {@link #gridletReceive()}
     *      </ul>
     * @pre gridletId > 0
     * @pre userId > 0
     * @pre srcId > 0
     * @pre destId > 0
     * @pre delay >= 0.0
     * @post $result = true || false
     */
    protected boolean gridletMove(int gridletId, int userId, int srcId,
                                  int destId, double delay, boolean ack)
    {
        String errorMsg = "GridSim.gridletMove(): ";

        // check whether the source Id is the same as destination Id
        if (srcId == destId)
        {
            System.out.println(errorMsg + "Error - Can't move a Gridlet to " +
                   "the same GridResource.");
            return false;
        }

        boolean valid = validateValue(errorMsg, gridletId, userId, srcId);
        if (valid == false || delay < 0.0) {
            return false;
        }

        // if a destination grid resource ID doesn't exist in GIS list
        if (gis_.isResourceExist(destId) == false)
        {
            System.out.println(errorMsg + "Error - GridResource ID #" + destId +
                   " doesn't exist. Hence, can't move Gridlet #" + gridletId);

            return false;
        }

        try
        {
            int size = 14;  // size of having 3 ints + 2 bytes overhead
            dataArray_[0] = gridletId;
            dataArray_[1] = userId;
            dataArray_[2] = destId;

            if (ack == true)
            {
                // sends the info to the destination GridResource
                send(this.output, delay, GridSimTags.GRIDLET_MOVE_ACK,
                     new IO_data(dataArray_, size, srcId)
                );

                // waiting for a response from the GridResource
                Sim_type_p tag = new Sim_type_p(GridSimTags.GRIDLET_SUBMIT_ACK
                                                + gridletId);

                // only look for this type of ack for same Gridlet ID
                super.sim_get_next(tag, ev_);
                valid = ( (Boolean) ev_.get_data() ).booleanValue();
            }
            else
            {
                valid = false;
                send(this.output, delay, GridSimTags.GRIDLET_MOVE,
                     new IO_data(dataArray_, size, srcId)
                );
            }
        }
        catch (Sim_exception e)
        {
            valid = false;
            System.out.println("GridSim.gridletMove(): Error occurs.");
            System.out.println( e.getMessage() );
        }
        catch (Exception ex)
        {
            valid = false;
            System.out.println("GridSim.gridletMove(): Error occurs.");
            System.out.println( ex.getMessage() );
        }

        return valid;
    }

    /////////////////  ENTITY  COMMUNICATION METHODS END ////////////


    ////////////////  ENTITY ID/NAME CONVERSION METHODS START ////////

    /**
     * Gets the name of this entity
     * @return the Entity name or <tt>null</tt> if this object does not have
     *         one
     * @deprecated As of GridSim 2.1, replaced by {@link #getEntityName()}
     * @pre $none
     * @post $none
     */
    public String GetEntityName() {
        return getEntityName();
    }

    /**
     * Gets the name of this entity
     * @return the Entity name or <tt>null</tt> if this object does not have
     *         one
     * @pre $none
     * @post $none
     */
    public String getEntityName()
    {
        try {
            return super.get_name();
        }
        catch (Sim_exception e) {
            return null;
        }
        catch (Exception e) {
            return null;
        }
    }

    /**
     * Gets name of the entity given its entity ID
     * @param entityID  the entity ID
     * @return the Entity name or <tt>null</tt> if this object does not have
     *         one
     * @deprecated As of GridSim 2.1, replaced by {@link #getEntityName(int)}
     * @pre entityID > 0
     * @post $none
     */
    public static String GetEntityName(int entityID) {
        return getEntityName(entityID);
    }

    /**
     * Gets name of the entity given its entity ID
     * @param entityID  the entity ID
     * @return the Entity name or <tt>null</tt> if this object does not have
     *         one
     * @pre entityID > 0
     * @post $none
     */
    public static String getEntityName(int entityID)
    {
        try {
            return Sim_system.get_entity(entityID).get_name();
        }
        catch (Sim_exception e) {
            return null;
        }
        catch (Exception e) {
            return null;
        }
    }

    /**
     * Gets the entity ID
     * @param entityName    an Entity name
     * @return the Entity ID or <tt>-1</tt> if it is not found
     * @deprecated As of GridSim 2.1, replaced by {@link #getEntityId(String)}
     * @pre entityName != null
     * @post $result >= -1
     */
    public static int GetEntityId(String entityName) {
        return getEntityId(entityName);
    }

    /**
     * Gets the entity ID given its name
     * @param entityName    an Entity name
     * @return the Entity ID or <tt>-1</tt> if it is not found
     * @pre entityName != null
     * @post $result >= -1
     */
    public static int getEntityId(String entityName)
    {
        try {
            return (int) Sim_system.get_entity_id(entityName);
        }
        catch (Sim_exception e) {
            return NOT_FOUND;
        }
        catch (Exception e) {
            return NOT_FOUND;
        }
    }

    /**
     * Gets the entity id of <tt>GridStatistics</tt>
     * @return the Entity ID or <tt>-1</tt> if it is not found
     * @deprecated As of GridSim 2.1, replaced by {@link
     *             #getGridStatisticsEntityId()}
     * @pre $none
     * @post $result >= -1
     */
    public static int GridStatisticsEntityId() {
        return getGridStatisticsEntityId();
    }

    /**
     * Gets the entity ID of <tt>GridStatistics</tt>
     * @return the Entity ID or <tt>-1</tt> if it is not found
     * @pre $none
     * @post $result >= -1
     */
    public static int getGridStatisticsEntityId()
    {
        try {
            return (int) Sim_system.get_entity_id("GridStatistics");
        }
        catch (Sim_exception e) {
            return NOT_FOUND;
        }
        catch (Exception e) {
            return NOT_FOUND;
        }
    }

    /**
     * Gets the entity ID of <tt>GridInformationService</tt>
     * @return the Entity ID or <tt>-1</tt> if it is not found
     * @deprecated As of GridSim 2.1, replaced by {@link
     *             #getGridInfoServiceEntityId()}
     * @pre $none
     * @post $result >= -1
     */
    public static int GridInformationServiceEntityId() {
        return getGridInfoServiceEntityId();
    }

    /**
     * Gets the entity ID of <tt>GridInformationService</tt>
     * @return the Entity ID or <tt>-1</tt> if it is not found
     * @pre $none
     * @post $result >= -1
     */
    public static int getGridInfoServiceEntityId()
    {
        try {
            return (int) Sim_system.get_entity_id("GridInformationService");
        }
        catch (Sim_exception e) {
            return NOT_FOUND;
        }
        catch (Exception e) {
            return NOT_FOUND;
        }
    }

    /**
     * Gets the entity id of <tt>GridSimShutdown</tt>
     * @return the Entity ID or <tt>-1</tt> if it is not found
     * @deprecated As of GridSim 2.1, replaced by {@link
     *             #getGridSimShutdownEntityId()}
     * @pre $none
     * @post $result >= -1
     */
    public static int GridSimShutdownEntityId() {
        return getGridSimShutdownEntityId();
    }

    /**
     * Gets the entity id of <tt>GridSimShutdown</tt>
     * @return the Entity ID or <tt>-1</tt> if it is not found
     * @pre $none
     * @post $result >= -1
     */
    public static int getGridSimShutdownEntityId()
    {
        try {
            return (int) Sim_system.get_entity_id("GridSimShutdown");
        }
        catch (Sim_exception e) {
            return NOT_FOUND;
        }
        catch (Exception e) {
            return NOT_FOUND;
        }
    }

    /**
     * Tells all user entities to shut down the simulation.
     * <b>gridsim.GridSimShutdown</b> entity waits for all users
     * termination before shuting down other entities.
     * @see gridsim.GridSimShutdown
     * @deprecated As of GridSim 2.1, replaced by {@link
     *             #shutdownUserEntity()}
     * @pre $none
     * @post $none
     */
    protected void ShutdownUserEntity() {
        shutdownUserEntity();
    }

    /**
     * Tells all user entities to shut down the simulation.
     * <b>gridsim.GridSimShutdown</b> entity waits for all users
     * termination before shuting down other entities.
     * @see gridsim.GridSimShutdown
     * @pre $none
     * @post $none
     */
    protected void shutdownUserEntity()
    {
        send(GridSim.getGridSimShutdownEntityId(), GridSimTags.SCHEDULE_NOW,
                GridSimTags.END_OF_SIMULATION);
    }

    /**
     * Tells the <tt>GridStatistics</tt> entity the end of the simulation
     * @deprecated As of GridSim 2.1, replaced by {@link
     *             #shutdownGridStatisticsEntity()}
     * @pre $none
     * @post $none
     */
    protected void ShutdownGridStatisticsEntity() {
        shutdownGridStatisticsEntity();
    }

    /**
     * Tells the <tt>GridStatistics</tt> entity the end of the simulation
     * @pre $none
     * @post $none
     */
    protected void shutdownGridStatisticsEntity()
    {
        send(GridSim.getGridStatisticsEntityId(), GridSimTags.SCHEDULE_NOW,
                GridSimTags.END_OF_SIMULATION);
    }

    /**
     * Sends a request to Grid Information Service (GIS) entity to get the
     * list of Grid resources
     * @return A LinkedList containing GridResource ID (as an Integer object)
     *         or <tt>null</tt> if a GIS entity hasn't been created before
     * @deprecated As of GridSim 2.1, replaced by {@link
     *             #getGridResourceList()}
     * @pre $none
     * @post $none
     */
    public static LinkedList GetGridResourceList() {
        return getGridResourceList();
    }

    /**
     * Sends a request to Grid Information Service (GIS) entity to get the
     * list of Grid resources
     * @return A LinkedList containing GridResource ID (as an Integer object)
     *         or <tt>null</tt> if a GIS entity hasn't been created before
     * @pre $none
     * @post $none
     */
    public static LinkedList getGridResourceList()
    {
        if (gis_ == null) {
            return null;
        }

        return gis_.getList();
    }

    /**
     * Gets a ResourceCharacteristics object for a given GridResource ID
     * @param resourceID   the resource ID
     * @return An object of ResourceCharacteristics or <tt>null</tt> if a
     *         GridResource doesn't exist
     * @see gridsim.ResourceCharacteristics
     * @deprecated As of GridSim 2.1, replaced by {@link
     *             #getResourceCharacteristics(int)}
     * @pre resourceID > 0
     * @post $none
     */
    public ResourceCharacteristics GetResourceCharacteristics(int resourceID) {
        return getResourceCharacteristics(resourceID);
    }

    /**
     * Gets a ResourceCharacteristics object for a given GridResource id
     * @param resourceID   the resource ID
     * @return An object of ResourceCharacteristics or <tt>null</tt> if a
     *         GridResource doesn't exist
     * @see gridsim.ResourceCharacteristics
     * @pre resourceID > 0
     * @post $none
     */
    public ResourceCharacteristics getResourceCharacteristics(int resourceID)
    {
        if (gis_.isResourceExist(resourceID) == false) {
            return null;
        }

        // Get Resource Characteristic Info: Send Request and Receive Event/Msg
        send(this.output, 0.0, GridSimTags.RESOURCE_CHARACTERISTICS,
             new IO_data( new Integer(super.get_id()), 4, resourceID)
        );

        return (ResourceCharacteristics) receiveEventObject(this.input);
    }

    /**
     * Gets the GridResource dynamic information
     * @param resourceID   the resource ID
     * @return An object of Accumulator containing the GridResource load or
     *         <tt>null</tt> if a GridResource doesn't exist
     * @deprecated As of GridSim 2.1, replaced by {@link
     *             #getResourceDynamicInfo(int)}
     * @pre resourceID > 0
     * @post $none
     */
    public Accumulator GetResourceDynamicInformation(int resourceID) {
        return getResourceDynamicInfo(resourceID);
    }

    /**
     * Gets the GridResource dynamic fnformation
     * @param resourceID   the resource ID
     * @return An object of Accumulator containing the GridResource load or
     *         <tt>null</tt> if a GridResource doesn't exist
     * @pre resourceID > 0
     * @post $none
     */
    public Accumulator getResourceDynamicInfo(int resourceID)
    {
        if (gis_.isResourceExist(resourceID) == false) {
            return null;
        }

        // Get Resource Dynamic information
        send(this.output, 0.0, GridSimTags.RESOURCE_DYNAMICS,
             new IO_data( new Integer(super.get_id()), 4, resourceID )
        );

        return (Accumulator) receiveEventObject(this.input);
    }

    ///////////////  ENTITY ID/NAME CONVERSION METHODS END ////////////



    ////////////  METHOD FOR RECORDING Statistics Information BEGIN ////////

    /**
     * Records statistics during the event
     * @param category  a category name
     * @param data      a value to be recorded
     * @deprecated As of GridSim 2.1, replaced by {@link
     *             #recordStatistics(String, double)}
     * @pre category != null
     * @pre data >= 0.0
     * @post $none
     */
    public void RecordStatistics(String category, double data) {
        recordStatistics(category, data);
    }

    /**
     * Records statistics during the event
     * @param category  a category name
     * @param data      a value to be recorded
     * @pre category != null
     * @pre data >= 0.0
     * @post $none
     */
    public void recordStatistics(String category, double data)
    {
        send(getGridStatisticsEntityId(), 0.0, GridSimTags.RECORD_STATISTICS,
             new Stat( GridSim.clock(), category, super.get_name(), data )
        );
    }

    /**
     * Records statistics during the event
     * @param category  a category name
     * @param data      a value to be recorded
     * @deprecated As of GridSim 2.1, replaced by {@link
     *             #recordStatistics(String, int)}
     * @pre category != null
     * @pre data >= 0.0
     * @post $none
     */
    public void RecordStatistics(String category, int data) {
        recordStatistics(category, data);
    }

    /**
     * Records statistics during the event
     * @param category  a category name
     * @param data      a value to be recorded
     * @pre category != null
     * @pre data >= 0.0
     * @post $none
     */
    public void recordStatistics(String category, int data)
    {
        send(getGridStatisticsEntityId(), 0.0, GridSimTags.RECORD_STATISTICS,
             new Stat( GridSim.clock(), category, super.get_name(), data )
        );
    }

    /**
     * Records statistics during the event
     * @param category  a category name
     * @param data      a value to be recorded
     * @deprecated As of GridSim 2.1, replaced by {@link
     *             #recordStatistics(String, String)}
     * @pre category != null
     * @pre data != null
     * @post $none
     */
    public void RecordStatistics(String category, String data) {
        recordStatistics(category, data);
    }

    /**
     * Records statistics during the event
     * @param category  a category name
     * @param data      a value to be recorded
     * @pre category != null
     * @pre data != null
     * @post $none
     */
    public void recordStatistics(String category, String data)
    {
        send(getGridStatisticsEntityId(), 0.0, GridSimTags.RECORD_STATISTICS,
             new Stat( GridSim.clock(), category, super.get_name(), data )
        );
    }

    /**
     * Records statistics during the event
     * @param category  a category name
     * @param data      a value to be recorded
     * @deprecated As of GridSim 2.1, replaced by {@link
     *             #recordStatistics(String, boolean)}
     * @pre category != null
     * @pre data == true || false
     * @post $none
     */
    public void RecordStatistics(String category, boolean data) {
        recordStatistics(category, data);
    }

    /**
     * Records statistics during the event
     * @param category  a category name
     * @param data      a value to be recorded
     * @pre category != null
     * @pre data == true || false
     * @post $none
     */
    public void recordStatistics(String category, boolean data)
    {
        send(getGridStatisticsEntityId(), 0.0, GridSimTags.RECORD_STATISTICS,
             new Stat( GridSim.clock(), category, super.get_name(), data )
        );
    }

} // end class

