/*
 * Title:        GridSim Toolkit
 * Description:  GridSim (Grid Simulation) Toolkit for Modeling and Simulation
 *               of Parallel and Distributed Systems such as Clusters and Grids
 * Licence:      GPL - http://www.gnu.org/copyleft/gpl.html
 *
 * $Id: GridResource.java,v 1.33 2003/12/11 08:47:21 anthony Exp $
 */

package gridsim;

import java.util.LinkedList;
import eduni.simjava.Sim_event;
import eduni.simjava.Sim_system;


/**
 * GridSim GridResource extends the <b>gridsim.GridSim</b> class and
 * gains communication and
 * concurrent entity capability. An instance of this class stimulates a resource
 * with properties defined in an object of
 * <b>gridsim.ResourceCharacteristics</b> class.
 * <p>
 * The process of creating a Grid resource is as follows:
 * <ol>
 * <li> create PE (Processing Element) objects with a suitable MIPS (Million
 *      Instructions Per Second) or SPEC (Standard Performance Evaluation
 *      Corporation) rating;
 * <li> assemble them together to create a machine;
 * <li> group one or more objects of the machine to form a Grid resource
 * </ol>
 * <p>
 * A resource having a single machine with one or more PEs (Processing Elements)
 * is managed as a time-shared system using a round-robin scheduling algorithm.
 * A resource with multiple machines is treated as a distributed memory cluster
 * and is managed as a space-shared system using FCFS (First Come Firt Serve)
 * scheduling policy or its variants.
 * <p>
 * Since GridSim 2.2, other scheduling algorithm can be added externally
 * (without compiling or replacing the existing GridSim JAR file) into a
 * Grid resource. For more information, look on tutorial page or AllocPolicy
 * class.
 *
 * @author       Manzur Murshed and Rajkumar Buyya
 * @version      2.2, December 2003
 * @see gridsim.GridSim
 * @see gridsim.ResourceCharacteristics
 * @see gridsim.AllocPolicy
 * @invariant $none
 */
public class GridResource extends GridSim
{
    private ResourceCharacteristics resource_;
    private ResourceCalendar resCalendar_;
    private AllocPolicy policy_;  // handles this GridResource scheduling
    private int policyType_;      // scheduling type of this GridResource
    private int resourceID_;      // this resource id
    private final String resourceName_;    // this resource name


    /**
     * Allocates a new GridResource object
     * @param name       the name to be associated with this entity (as
     *                   required by Sim_entity class from simjava package)
     * @param baud_rate  network communication or bandwidth speed
     * @param seed       the initial seed
     * @param resource   an object of ResourceCharacteristics
     * @param peakLoad   the load during peak times
     * @param offPeakLoad   the load during off peak times
     * @param relativeHolidayLoad   the load during holiday times
     * @param weekends   a linked-list contains the weekend days
     * @param holidays   a linked-list contains the public holidays
     * @throws Exception This happens when one of the following scenarios occur:
     *      <ul>
     *          <li> creating this entity before initializing GridSim package
     *          <li> this entity name is <tt>null</tt> or empty
     *          <li> this entity has <tt>zero</tt> number of PEs (Processing
     *              Elements). <br>
     *              No PEs mean the Gridlets can't be processed.
     *              A GridResource must contain one or more Machines.
     *              A Machine must contain one or more PEs.
     *      </ul>
     * @see gridsim.GridSim#init(int, Calendar, boolean, String[], String[],
     *          String)
     * @pre $none
     * @post $none
     */
    public GridResource(String name, double baud_rate, long seed,
            ResourceCharacteristics resource, double peakLoad,
            double offPeakLoad, double relativeHolidayLoad,
            LinkedList weekends, LinkedList holidays) throws Exception
    {
        super(name, baud_rate);
        resource_ = resource;
        resourceName_ = name;

        resCalendar_ = new ResourceCalendar(resource_.getResourceTimeZone(),
                peakLoad, offPeakLoad, relativeHolidayLoad, weekends,
                holidays, seed);

        policy_ = null;
        init();
    }

    /**
     * Allocates a new GridResource object
     * @param name       the name to be associated with this entity (as
     *                   required by Sim_entity class from simjava package)
     * @param baud_rate  network communication or bandwidth speed
     * @param resource   an object of ResourceCharacteristics
     * @param calendar   an object of ResourceCalendar
     * @throws Exception This happens when one of the following scenarios occur:
     *      <ul>
     *          <li> creating this entity before initializing GridSim package
     *          <li> this entity name is <tt>null</tt> or empty
     *          <li> this entity has <tt>zero</tt> number of PEs (Processing
     *              Elements). <br>
     *              No PEs mean the Gridlets can't be processed.
     *              A GridResource must contain one or more Machines.
     *              A Machine must contain one or more PEs.
     *      </ul>
     * @see gridsim.GridSim#init(int, Calendar, boolean, String[], String[],
     *          String)
     * @pre $none
     * @post $none
     */
    public GridResource(String name, double baud_rate,
            ResourceCharacteristics resource, ResourceCalendar calendar)
                throws Exception
    {
        super(name, baud_rate);
        resource_ = resource;
        resourceName_ = name;

        resCalendar_ = calendar;
        policy_ = null;
        init();
    }

    /**
     * Allocates a new GridResource object
     * @param name       the name to be associated with this entity (as
     *                   required by Sim_entity class from simjava package)
     * @param baud_rate  network communication or bandwidth speed
     * @param resource   an object of ResourceCharacteristics
     * @param calendar   an object of ResourceCalendar
     * @param policy     a scheduling policy for this Grid resource. If no
     *                   scheduling policy is defined, the default one is
     *                   <tt>SpaceShared</tt>
     * @throws Exception This happens when one of the following scenarios occur:
     *      <ul>
     *          <li> creating this entity before initializing GridSim package
     *          <li> this entity name is <tt>null</tt> or empty
     *          <li> this entity has <tt>zero</tt> number of PEs (Processing
     *              Elements). <br>
     *              No PEs mean the Gridlets can't be processed.
     *              A GridResource must contain one or more Machines.
     *              A Machine must contain one or more PEs.
     *      </ul>
     * @see gridsim.GridSim#init(int, Calendar, boolean, String[], String[],
     *          String)
     * @see gridsim.AllocPolicy
     * @pre $none
     * @post $none
     */
    public GridResource(String name, double baud_rate,
            ResourceCharacteristics resource, ResourceCalendar calendar,
            AllocPolicy policy) throws Exception
    {
        super(name, baud_rate);
        resource_ = resource;
        resourceName_ = name;

        resCalendar_ = calendar;
        policy_ = policy;  // the order between policy and init() is important
        init();
    }

    /**
     * Handles external events that are coming to this GridResource entity.
     * This method also registers this GridResource entity to
     * <tt>GridInformationService</tt> class.
     * <p>
     * The services available to other GridSim entities are:
     * <ul>
     *      <li> GridSimTags.RESOURCE_CHARACTERISTICS </li>
     *      <li> GridSimTags.RESOURCE_DYNAMICS </li>
     *      <li> GridSimTags.GRIDLET_SUBMIT </li>
     *      <li> GridSimTags.GRIDLET_CANCEL </li>
     *      <li> GridSimTags.GRIDLET_PAUSE </li>
     *      <li> GridSimTags.GRIDLET_RESUME </li>
     *      <li> GridSimTags.GRIDLET_MOVE </li>
     *      <li> GridSimTags.GRIDLET_STATUS </li>
     * </ul>
     * @pre $none
     * @post $none
     */
    public void body()
    {
        // send the registration to GIS
        super.send(super.output, GridSimTags.SCHEDULE_NOW,
                   GridSimTags.REGISTER_RESOURCE,
                   new IO_data( new Integer(resourceID_), 4,
                                GridSim.getGridInfoServiceEntityId() )
        );

        // Process events until END_OF_SIMULATION is received from the
        // GridSimShutdown Entity
        Sim_event ev = new Sim_event();
        while ( Sim_system.running() )
        {
            super.sim_get_next(ev);

            // if the simulation finishes then exit the loop
            if (ev.get_tag() == GridSimTags.END_OF_SIMULATION)
            {
                policy_.setEndSimulation();
                break;
            }

            // process the received event
            processEvent(ev);
        }

        // remove I/O entities created during construction of this entity
        super.terminateIOEntities();
    }

    /**
     * As of GridSim 2.2, this method is <b>OBSOLETE</b>.
     * Allocates one of the PEs to Gridlet for execution and schedules
     * an internal event to be delivered at completion time.
     * @param gl a Gridlet to be processed
     * @deprecated As of GridSim 2.2, this method is <b>OBSOLETE</b>.
     * @pre gl != null
     * @post $none
     */
    public void SpaceShare_AllocatePEtoGridlet(Gridlet gl) {
        this.spaceShared_AllocatePEtoGridlet(gl);
    }

    /**
     * As of GridSim 2.2, this method is <b>OBSOLETE</b>.
     * Allocates one of the PEs to Gridlet for execution and schedules
     * an internal event to be delivered at completion time.
     * @param gl a Gridlet to be processed
     * @deprecated As of GridSim 2.2, this method is <b>OBSOLETE</b>.
     * @pre gl != null
     * @post $none
     */
    public void spaceShared_AllocatePEtoGridlet(Gridlet gl)
    {
        System.out.println("GridResource.spaceShared_AllocatePEtoGridlet()" +
                " is OBSOLETE since GridSim 2.2. Don't use this method.");
    }

    //////////////////// PRIVATE METHODS ///////////////////////////////////

    /**
     * Initializes the resource allocation policy
     * @throws Exception    If number of PEs is zero
     * @pre $none
     * @post $none
     */
    private void init() throws Exception
    {
        // If this resource doesn't have any PEs then no useful at all...
        if (resource_.getNumPE() == 0)
        {
            throw new Exception("GridResource() : Error - this entity " +
                    "has no PEs. Therefore, can't process any Gridlets.");
        }

        // stores id of this class
        resourceID_ = super.get_id();
        resource_.setResourceID(resourceID_);

        // if internal allocation policy is used
        if (policy_ == null)
        {
            policyType_ = resource_.getResourceAllocationPolicy();
            switch (policyType_) {
                case ResourceCharacteristics.TIME_SHARED:
                    policy_ = new TimeShared(resourceName_, "TimeShared");
                    break;

                case ResourceCharacteristics.SPACE_SHARED:
                case ResourceCharacteristics.OTHER_POLICY_SAME_RATING:
                case ResourceCharacteristics.OTHER_POLICY_DIFFERENT_RATING:
                    policy_ = new SpaceShared(resourceName_, "SpaceShared");
                    break;

                default:
                    throw new Exception("GridResource: Invalid Resource " +
                                        "Allocation policy.");
            }
        }

        policy_.init(resource_, resCalendar_, super.output);
    }

    /**
     * Processes events or services that are available for this GridResource
     * @param ev    a Sim_event object
     * @pre ev != null
     * @post $none
     */
    private void processEvent(Sim_event ev)
    {
        int src_id = -1;
        switch ( ev.get_tag() )
        {
                // Resource characteristics inquiry
            case GridSimTags.RESOURCE_CHARACTERISTICS:
                src_id = ( (Integer) ev.get_data() ).intValue();
                super.send( super.output, 0.0, ev.get_tag(),
                        new IO_data(resource_, resource_.getByteSize(), src_id)
                );
                break;

                // Resource dynamic info inquiry
            case GridSimTags.RESOURCE_DYNAMICS:
                src_id = ( (Integer) ev.get_data() ).intValue();
                super.send( super.output, 0.0, ev.get_tag(),
                            new IO_data(policy_.getTotalLoad(),
                                        Accumulator.getByteSize(), src_id)
                );
                break;

                // New Gridlet arrives
            case GridSimTags.GRIDLET_SUBMIT:
                processGridletSubmit(ev, false);
                break;

                // New Gridlet arrives, but the sender asks for an ack
            case GridSimTags.GRIDLET_SUBMIT_ACK:
                processGridletSubmit(ev, true);
                break;

                // Cancels a previously submitted Gridlet
            case GridSimTags.GRIDLET_CANCEL:
                processGridlet(ev, GridSimTags.GRIDLET_CANCEL);
                break;

                // Pauses a previously submitted Gridlet
            case GridSimTags.GRIDLET_PAUSE:
                processGridlet(ev, GridSimTags.GRIDLET_PAUSE);
                break;

                // Pauses a previously submitted Gridlet, but the sender
                // asks for an acknowledgement
            case GridSimTags.GRIDLET_PAUSE_ACK:
                processGridlet(ev, GridSimTags.GRIDLET_PAUSE_ACK);
                break;

                // Resumes a previously submitted Gridlet
            case GridSimTags.GRIDLET_RESUME:
                processGridlet(ev, GridSimTags.GRIDLET_RESUME);
                break;

                // Resumes a previously submitted Gridlet, but the sender
                // asks for an acknowledgement
            case GridSimTags.GRIDLET_RESUME_ACK:
                processGridlet(ev, GridSimTags.GRIDLET_RESUME_ACK);
                break;

                // Moves a previously submitted Gridlet to a different resource
            case GridSimTags.GRIDLET_MOVE:
                processGridletMove(ev, GridSimTags.GRIDLET_MOVE);
                break;

                // Moves a previously submitted Gridlet to a different resource
            case GridSimTags.GRIDLET_MOVE_ACK:
                processGridletMove(ev, GridSimTags.GRIDLET_MOVE_ACK);
                break;

                // Checks the status of a Gridlet
            case GridSimTags.GRIDLET_STATUS:
                processGridletStatus(ev);
                break;

            default:
                System.out.println(resourceName_ + ".body(): Unable to " +
                        "handle request from GridSimTags " +
                        "with constant number " + ev.get_tag() );
                break;
        }
    }

    /**
     * Process the event for an User who wants to know the status of a Gridlet.
     * This GridResource will then send the status back to the User.
     * @param ev   a Sim_event object
     * @pre ev != null
     * @post $none
     */
    private void processGridletStatus(Sim_event ev)
    {
        int gridletId = 0;
        int userId = 0;
        int status = -1;

        try
        {
            // if a sender using gridletXXX() methods
            int data[] = (int[]) ev.get_data();
            gridletId = data[0];
            userId = data[1];

            status = policy_.gridletStatus(gridletId, userId);
        }

        // if a sender using normal send() methods
        catch (ClassCastException c)
        {
            try
            {
                Gridlet gl = (Gridlet) ev.get_data();
                gridletId = gl.getGridletID();
                userId = gl.getUserID();

                status = policy_.gridletStatus(gridletId, userId);
            }
            catch (Exception e)
            {
                System.out.println(resourceName_ +
                         ": Error in processing GridSimTags.GRIDLET_STATUS");
                System.out.println( e.getMessage() );
                return;
            }
        }
        catch (Exception e)
        {
            System.out.println(resourceName_ +
                         ": Error in processing GridSimTags.GRIDLET_STATUS");
            System.out.println( e.getMessage() );
            return;
        }

        // unique tag = operation tag + Gridlet ID
        int tag = GridSimTags.GRIDLET_STATUS + gridletId;
        super.send(super.output, GridSimTags.SCHEDULE_NOW, tag,
                   new IO_data(new Integer(status), 8, userId)
        );
    }

    /**
     * Processes a Gridlet based on the event type
     * @param ev   a Sim_event object
     * @param type event type
     * @pre ev != null
     * @pre type > 0
     * @post $none
     */
    private void processGridlet(Sim_event ev, int type)
    {
        int gridletId = 0;
        int userId = 0;

        try
        {
            // if a sender using gridletXXX() methods
            int data[] = (int[]) ev.get_data();
            gridletId = data[0];
            userId = data[1];
        }

        // if a sender using normal send() methods
        catch (ClassCastException c)
        {
            try
            {
                Gridlet gl = (Gridlet) ev.get_data();
                gridletId = gl.getGridletID();
                userId = gl.getUserID();
            }
            catch (Exception e)
            {
                System.out.println(resourceName_ +
                                   ": Error in processing Gridlet");
                System.out.println( e.getMessage() );
                return;
            }
        }
        catch (Exception e)
        {
            System.out.println(resourceName_ + ": Error in processing Gridlet");
            System.out.println( e.getMessage() );
            return;
        }

        // begins executing ....
        switch (type)
        {
            case GridSimTags.GRIDLET_CANCEL:
                policy_.gridletCancel(gridletId, userId);
                break;

            case GridSimTags.GRIDLET_PAUSE:
                policy_.gridletPause(gridletId, userId, false);
                break;

            case GridSimTags.GRIDLET_PAUSE_ACK:
                policy_.gridletPause(gridletId, userId, true);
                break;

            case GridSimTags.GRIDLET_RESUME:
                policy_.gridletResume(gridletId, userId, false);
                break;

            case GridSimTags.GRIDLET_RESUME_ACK:
                policy_.gridletResume(gridletId, userId, true);
                break;

            default:
                break;
        }

    }

    /**
     * Process the event for an User who wants to know the move of a Gridlet.
     * @param ev   a Sim_event object
     * @param type  event tag
     * @pre ev != null
     * @pre type > 0
     * @post $none
     */
    private void processGridletMove(Sim_event ev, int type)
    {
        boolean ack = false;
        if (type == GridSimTags.GRIDLET_MOVE_ACK) {
            ack = true;
        }

        try
        {
            // if a sender using gridletMove() methods
            int data[] = (int[]) ev.get_data();
            int gridletId = data[0];
            int userId = data[1];
            int destId = data[2];

            policy_.gridletMove(gridletId, userId, destId, ack);
        }
        catch (Exception e)
        {
            System.out.println(resourceName_ + ": Error in moving a Gridlet.");
            System.out.println( e.getMessage() );
        }
    }

    /**
     * Processes a Gridlet submission
     * @param ev  a Sim_event object
     * @param ack  an acknowledgement
     * @pre ev != null
     * @post $none
     */
    private void processGridletSubmit(Sim_event ev, boolean ack)
    {
        // gets the Gridlet object
        Gridlet gl = (Gridlet) ev.get_data();

        // checks whether this Gridlet has finished or not
        if (gl.isFinished() == true)
        {
            System.out.println(resourceName_ + ": Error - Gridlet #" +
                               gl.getGridletID() + " for User #" +
                               gl.getUserID() + " is already finished.");

            if (ack == true)
            {
                // unique tag = operation tag + Gridlet ID
                int tag = GridSimTags.GRIDLET_SUBMIT_ACK + gl.getGridletID();
                super.send(super.output, GridSimTags.SCHEDULE_NOW, tag,
                           new IO_data( new Boolean(false), 8, gl.getUserID() )
                );
            }

            super.send(super.output, GridSimTags.SCHEDULE_NOW,
                   GridSimTags.GRIDLET_RETURN,
                   new IO_data( gl, gl.getGridletOutputSize(), gl.getUserID() )
            );

            return;
        }

        // process this Gridlet to this GridResource
        gl.setResourceParameter( this.resourceID_, resource_.getCostPerSec() );
        policy_.gridletSubmit(gl, ack);
    }

} // end class

