/*
 * Title:        GridSim Toolkit
 * Description:  GridSim (Grid Simulation) Toolkit for Modeling and Simulation
 *               of Parallel and Distributed Systems such as Clusters and Grids
 * Licence:      GPL - http://www.gnu.org/copyleft/gpl.html
 *
 * $Id: GridInformationService.java,v 1.15 2003/12/13 15:48:22 anthony Exp $
 */

package gridsim;

import java.util.LinkedList;
import java.util.Iterator;
import eduni.simjava.Sim_event;
import eduni.simjava.Sim_system;

/*---------------------------------------------------------------------------*/
// Classes related to GridSim Resources: GIS, Local Resource Schedulers
/*---------------------------------------------------------------------------*/

/**
 * A Grid Information Service (GIS) is a GridSim entity that provides Grid
 * resource registration, indexing and discovery services. The Grid resources
 * register their readiness to process Gridlets by registering themselves with
 * this entity.
 * <p>
 * Other entities such as the resource broker can contact
 * this class for resource discovery service, which returns a list of
 * registered resource entities and their contact address. For example,
 * scheduling entities use this service for resource discovery.
 * <p>
 * In summary, it acts like a yellow page service.
 *
 * @author       Manzur Murshed and Rajkumar Buyya
 * @version      2.2, December 2003
 * @invariant $none
 */
public class GridInformationService extends GridSim
{
    private LinkedList resList_;    // for all type of resources

    /**
     * Allocates a new GridInformationService object with networked I/O ports
     * @param name       the name to be associated with this entity (as
     *                   required by Sim_entity class from simjava package)
     * @param baud_rate  communication speed
     * @throws Exception This happens when creating this entity before
     *                   initializing GridSim package or this entity name is
     *                   <tt>null</tt> or empty
     * @see gridsim.GridSim#init(int, Calendar, boolean, String[], String[],
     *          String)
     * @see eduni.simjava.Sim_entity
     * @pre name != null
     * @pre baud_rate > 0.0
     * @post $none
     */
    public GridInformationService(String name, double baud_rate)
                    throws Exception
    {
        super(name, baud_rate);
        resList_ = new LinkedList();
    }

    /**
     * A method that gets one event at a time, and serves it based
     * on its request.
     * <p>
     * The services available to other GridSim entities are:
     * <ul>
     *      <li> GridSimTags.REGISTER_RESOURCE </li>
     *      <li> GridSimTags.RESOURCE_LIST </li>
     * </ul>
     * @pre $none
     * @post $none
     */
    public void body()
    {
        int id = -1;  // requester id

        // Process events
        Sim_event ev = new Sim_event();
        while ( Sim_system.running() )
        {
            super.sim_get_next(ev);

            // if the simulation finishes then exit the loop
            if (ev.get_tag() == GridSimTags.END_OF_SIMULATION) {
                break;
            }

            switch ( ev.get_tag() )
            {
                // A resource is requesting to register.
                case GridSimTags.REGISTER_RESOURCE:
                    resList_.add( (Integer) ev.get_data() );
                    break;

                // A Broker is requesting for a list of all resources.
                case GridSimTags.RESOURCE_LIST:

                    // Get ID of Broker that send this event
                    id = ( (Integer) ev.get_data() ).intValue();

                    // Send the event or message
                    super.send(super.output, 0.0, ev.get_tag(),
                            new IO_data(resList_, 4*resList_.size(), id)
                    );
                    break;

                default:
                    System.out.println("GridInformationSevice.body() : " +
                            "Unable to handle request from GridSimTags " +
                            "with constant number " + ev.get_tag() );
                    break;
            }
        }

        // after finish the simulation, disconnect I/O ports
        super.terminateIOEntities();
    }

    /**
     * Gets the list of GridResource entity
     * @return a LinkedList object of GridResource entity
     * @pre $none
     * @post $none
     */
    public LinkedList getList() {
        return resList_;
    }

    /**
     * Checks whether the given GridResource ID exists or not
     * @param id    a GridResource id
     * @return <tt>true</tt> if the given ID exists, <tt>false</tt> otherwise
     * @pre id >= 0
     * @post $none
     */
    public boolean isResourceExist(int id)
    {
        boolean flag = false;
        if (id < 0) {
            return flag;
        }

        Integer obj = null;

        // a loop to find the match the resource id in a list
        for (int i = 0; i < resList_.size(); i++)
        {
            obj = (Integer) resList_.get(i);
            if (obj.intValue() == id)
            {
                flag = true;
                break;
            }
        }

        return flag;
    }

} // end class

