/*
 * Title:        Grid Broker
 * Description:  A Grid Scheduler for Application Scheduling on Grid based on
 *               Deadline and Budget Constrained Scheduling Algorithms
 * Licence:      GPL - http://www.gnu.org/copyleft/gpl.html
 * $Id: UserEntity.java,v 1.5 2003/06/30 05:05:59 anthony Exp $
 */

package gridbroker;

import gridbroker.Experiment;
import gridsim.*;
import eduni.simjava.Sim_system;
import java.util.Random;


/**
 * UserEntity class is an entity that simulates the user. 
 * It invokes the broker and passes the user requirements. When it receives the
 * results of application processing, it records parameters of interest with
 * <tt>gridsim.Statistics</tt> entity. When it has no more processing
 * requirements, it sends <tt>END_OF_SIMULATION</tt> event to
 * <tt>gridsim.GridSimShutdown</tt> entity.
 *
 * @author       Manzur Murshed and Rajkumar Buyya
 * @version      2.1, June 2003
 * @invariant $none
 */
public class UserEntity extends GridSim
{
    // Experiment for Submission with Gridlet to be processed
    private Experiment expSend_;    
    private Experiment expReceived_;    // Experiment Processed
    private String brokerName_;         // Name of the Broker Entity
    private Random randomDelay_;
    private double successiveExpDelay_;
    private boolean resultDisplay_;  // true -display gridlet processing report

    // Utilisaton Factor: How much fraction of given quantity is utilised
    // if UF is 1, 100% is utilised if UF 0.5 is 50% utilised or processed.
    private Accumulator gridletCompletionFactor_;
    private Accumulator timeUtilizationFactor_;
    private Accumulator budgetUtilizationFactor_;


    /**
     * Allocates a new UserEntity object
     * @param name  the entity name
     * @param expt  an object of Experiment
     * @param baudRate     the communication speed
     * @param seed  the initial seed
     * @param successiveExperimentDelay     the delay between successive
     *                                      experiment
     * @param resultDisplay    display Gridlet processing report or not
     * @throws Exception This happens when creating this entity before 
     *                   initializing GridSim package or this entity name is
     *                   <tt>null</tt> or empty
     * @see gridsim.GridSim#init(int, Calendar, boolean, String[], String[],
     *          String)
     * @see Experiment
     * @pre name != null
     * @pre expt != null
     * @pre baudRate >= 0.0
     * @pre seed > 0
     * @pre successiveExperimentDelay >= 0.0
     * @post $none
     */
    public UserEntity(String name, Experiment expt, double baudRate, 
                long seed, double successiveExperimentDelay, 
                boolean resultDisplay) throws Exception
    {
        super(name);
        randomDelay_ = new Random(seed);

        // Create a Broker with entity name as "Broker_UserEntityName"
        brokerName_ = "Broker_" + name;
        new Broker(brokerName_, baudRate);
        
        expSend_ = expt;
        successiveExpDelay_ = successiveExperimentDelay;
        resultDisplay_ = resultDisplay;
        gridletCompletionFactor_ = new Accumulator();
        timeUtilizationFactor_ = new Accumulator();
        budgetUtilizationFactor_ = new Accumulator();
        expReceived_ = null;
    }


    /**
     * A method that records statistics, sends entity and shuts down when
     * the simulation finish
     * @pre $none
     * @post $none
     */
    public void body()
    {
        double range = 0.1;
        super.recordStatistics("USER.B_factor", expSend_.getBudgetFactor());
        super.recordStatistics("USER.D_factor", expSend_.getDeadlineFactor());
        super.gridSimHold( GridSimRandom.real(successiveExpDelay_, range, 
                range, randomDelay_.nextDouble()) );

        // Send Experiment to the Broker.
        super.send(brokerName_, GridSimTags.SCHEDULE_NOW, 
                GridSimTags.EXPERIMENT, expSend_);

        // Wait for Experiment Completion event
        expReceived_ = (Experiment) super.receiveEventObject();
        
        // set current simulation time as expt. completion time
        expReceived_.setEndTime();

        gridletCompletionFactor_.add(expReceived_.getNumSuccessfulGridlet());
        timeUtilizationFactor_.add( expReceived_.getEndTime() -
                expReceived_.getStartTime() );
        
        budgetUtilizationFactor_.add( expReceived_.getBudgetSpent() );

        super.recordStatistics("USER.Experiment.No", 
                expReceived_.getExperimentID());
        super.recordStatistics("USER.Experiment.StartTime", 
                expReceived_.getStartTime());
        super.recordStatistics("USER.Experiment.EndTime", 
                expReceived_.getEndTime());

        super.recordStatistics("USER.Experiment.GridletCompletion", 
                expReceived_.getNumSuccessfulGridlet());
        super.recordStatistics("USER.Experiment.TimeUtilization", 
                expReceived_.getEndTime()-expReceived_.getStartTime());
        super.recordStatistics("USER.Experiment.BudgetUtilization", 
                expReceived_.getBudgetSpent());

        if (resultDisplay_)
        {
            // Gridlet execution data
            System.out.println(super.get_name() + ":: Experiment Schedule");
            System.out.println("-----------------------------------------");
            
            System.out.println("UserID | GridLetID | Status | Resource ID | " +
                    "GL_SubmissionTime | GL_StartTime | Gl_FinishTime | " +
                    "GL_actualCPUtime | GL_WallClockTime | Resource Name");
            
            Gridlet gl;
            String res_name;
            for (int i = 0; i < expSend_.getGridletList().size(); i++)
            {
                gl = (Gridlet) expSend_.getGridletList().get(i);
                System.out.print(super.get_name() + " | " + gl.getGridletID() +
                        " | " + gl.getGridletStatusString() + 
                        " | " + gl.getResourceID() + 
                        " | " + gl.getSubmissionTime() + 
                        " | " + gl.getExecStartTime() + 
                        " | " + gl.getFinishTime() +
                        " | " + gl.getActualCPUTime() + 
                        " | " + gl.getWallClockTime() + " | ");
                
                if (gl.getGridletStatus() == Gridlet.CREATED) {
                    System.out.println("No Resource Assigned");
                }
                else
                {
                    res_name = Sim_system.get_entity( gl.getResourceID() 
                                ).get_name();
                    
                    System.out.println(res_name);
                }
            }
            
            // Experiment Statistics
            System.out.println("******** Experiment Report ********");
            System.out.println( "No. of Gridlets: " +
                    expReceived_.getNumGridlet() );
            System.out.println( "Deadline Factor: " + 
                    expReceived_.getDeadlineFactor() );
            System.out.println( "Deadline: " + expReceived_.getDeadline() );
            System.out.println( "Deadline Time: " + 
                    expReceived_.getDeadlineTime() );
            System.out.println( "Budget Factor: " + 
                    expReceived_.getBudgetFactor() );
            System.out.println( "Budget: " + expReceived_.getBudget() );
            System.out.println( "Stratergy: " +
                    expReceived_.getOptimizationStrategyName() );
            System.out.println( "Start Time: " + expReceived_.getStartTime() );
            System.out.println( "End Time: " + expReceived_.getEndTime() );
            System.out.println( "Budget Spent: " + 
                    expReceived_.getBudgetSpent() );
            System.out.println( "Gridlets Executed: " +
                    expReceived_.getNumSuccessfulGridlet() );
            System.out.println( "Gridlet Completion Factor: " +
                    expReceived_.getGridletCompletionFactor() );
        }

        super.recordStatistics( "*.USER.GridletCompletion", 
                gridletCompletionFactor_.getMean() );
        super.recordStatistics( "*.USER.TimeUtilization", 
                timeUtilizationFactor_.getMean() );
        super.recordStatistics( "*.USER.BudgetUtilization", 
                budgetUtilizationFactor_.getMean() );

        System.out.println( "GridletCompletion: " +
                (int) (gridletCompletionFactor_.getMean()) + "\t" +
                "TimeUtilization: "+ (int)(timeUtilizationFactor_.getMean()) +
                "\t"+ "BudgetUtilization: " + 
                (int)(budgetUtilizationFactor_.getMean()) );

        // Signal the END to broker.
        super.send(brokerName_, GridSimTags.SCHEDULE_NOW, 
                GridSimTags.END_OF_SIMULATION);

        // Signal the this User's End to GridSim: This should be invoked 
        // at the end.
        super.shutdownUserEntity();
    }
    
} // end class

