/*
 * Title:        Grid Broker
 * Description:  A Grid Scheduler for Application Scheduling on Grid based on
 *               Deadline and Budget Constrained Scheduling Algorithms
 * Licence:      GPL - http://www.gnu.org/copyleft/gpl.html
 * $Id: RunConfiguration.java,v 1.4 2003/06/30 05:10:55 anthony Exp $
 */

package gridbroker;

import java.util.StringTokenizer;
import java.io.*;


/**
 * A class that handles the file configuration
 *
 * @author       Manzur Murshed and Rajkumar Buyya
 * @version      2.1, June 2003
 * @invariant $none
 */
public class RunConfiguration
{
    private String configFile_;
    private int numUser_;
    private int userSamplePoint_;
    private int numResource_;
    private int resourceSamplePoint_;


    /**
     * Allocates a new RunConfiguration object
     * @param filename  the configuration file name
     * @pre filename != null
     * @post $none
     */
    public RunConfiguration(String filename) 
    {
        configFile_ = filename;
        numUser_ = 0;
        userSamplePoint_ = 0;
        numResource_ = 0;
        resourceSamplePoint_ = 0;
    }

    /**
     * Reads the configuration file 
     * @return <tt>true</tt> if the configuration file can be read, otherwise
     *         <tt>false</tt>
     * @deprecated As of GridBroker 2.1, replaced by {@link #readConfigFile()}
     * @pre $none
     * @post $none
     */
    public boolean ReadConfig() {
        return this.readConfigFile();
    }

    /**
     * Reads the configuration file 
     * @return <tt>true</tt> if the configuration file can be read, otherwise
     *         <tt>false</tt>
     * @pre $none
     * @post $none
     */
    public boolean readConfigFile()
    {
        boolean flag = true;
        try 
        {
            FileInputStream inFile = new FileInputStream(configFile_);
            BufferedReader reader = new BufferedReader(
                                            new InputStreamReader(inFile) );

            // File Format:
            // numUser_  userSamplePoint_  numResource_ resourceSamplePoint_

            String conf;
            conf = reader.readLine();
            StringTokenizer st = new StringTokenizer(conf);

            Integer I;
            I = new Integer( (String) st.nextElement() );
            numUser_ = I.intValue();

            I = new Integer( (String) st.nextElement() );
            userSamplePoint_ = I.intValue();

            I = new Integer( (String) st.nextElement() );
            numResource_ = I.intValue();

            I = new Integer( (String) st.nextElement() );
            resourceSamplePoint_ = I.intValue();
        }

        catch (FileNotFoundException e)
        {
            System.out.println("RunConfiguration.readConfigFile() : Error - " +
                    "unable to open \"" + configFile_ + "\".");

            flag = false;
        }
        catch (IOException e)
        {
            System.out.println("RunConfiguration.readConfigFile() : Error - " +
                    "unexpected error during reading \"" + configFile_ + "\".");
            
            flag = false;
        }
        catch (Exception e)
        {
            System.out.println("RunConfiguration.readConfigFile() : Error - " +
                    "unexpected error during parsing \"" + configFile_ + "\".");

            flag = false;
        }


        return flag;
    }

    /** 
     * Reads a string from the keyboard until the ENTER key is pressed.
     * @return the string read or <tt>null</tt> if it is empty
     * @pre $none
     * @post $none
     */
    public static String readString()
    {
        String text = null;
        int num = 0;
        
        // read until ENTER key is pressed.
        while (true)
        {
            try {
                num = System.in.read();
            }
            catch (IOException e)
            {
                System.out.println("RunConfiguration.readString() : Error - " +
                        "unable to read from keyboard.");
            }
            
                // if Carriage Return is pressed, then end.
            if (num == '\n') { 
                break;
            }
            
            text = text + (char) num;  // concatenate character
        }
        return text;
    }

    /**
     * Sets the configuration file name
     * @param name  file name
     * @pre name != null
     * @post $none
     */
    public void setConfigFileName(String name) {
        configFile_ = name;
    }

    /**
     * Gets the configuration file name 
     * @return file name
     * @pre $none
     * @post $result != null
     */
    public String getConfigFileName() {
        return configFile_;
    }

    /**
     * Sets number of users 
     * @param num   number of users
     * @pre num >= 0
     * @post $none
     */
    public void setNumUser(int num) {
        numUser_ = num;
    }

    /**
     * Gets number of users  
     * @return number of users
     * @pre $none
     * @post $result >= 0
     */
    public int getNumUser() {
        return numUser_;
    }

    /**
     * Sets the user sample point 
     * @param point  sample point
     * @pre point >= 0
     * @post $none
     */
    public void setUserSamplePoint(int point) {
        userSamplePoint_ = point;
    }

    /**
     * Gets the user sample point 
     * @return sample point
     * @pre $none
     * @post $result >= 0
     */
    public int getUserSamplePoint() {
        return userSamplePoint_;
    }

    /**
     * Sets the number of resources 
     * @param num   number of resources
     * @pre num >= 0
     * @post $none
     */
    public void setNumResource(int num) {
        numResource_ = num;
    }
    
    /**
     * Gets number of resources 
     * @return number of resources
     * @pre $none
     * @post $result >= 0
     */
    public int getNumResource() {
        return numResource_;
    }

    /**
     * Sets the resource sample point 
     * @param point   sample point
     * @pre point >= 0
     * @post $none
     */
    public void setResourceSamplePoint(int point) {
        resourceSamplePoint_ = point;
    }

    /**
     * Gets the resource sample point 
     * @return sample point
     * @pre $none
     * @post $result >= 0
     */
    public int getResourceSamplePoint() {
        return resourceSamplePoint_;
    }
   
} // end class

