/*
 * Title:        Grid Broker
 * Description:  A Grid Scheduler for Application Scheduling on Grid based on
 *               Deadline and Budget Constrained Scheduling Algorithms
 * Licence:      GPL - http://www.gnu.org/copyleft/gpl.html
 * $Id: ResDef.java,v 1.2 2003/06/30 05:05:59 anthony Exp $
 */

package gridbroker;


/**
 * A class that handles resource parameters
 *
 * @author       Manzur Murshed and Rajkumar Buyya
 * @version      2.1, June 2003
 * @invariant $none
 */
public class ResDef 
{
    private String arch_;
    private String OS_;
    private int numPE_;
    private int rating_;   // PE SPEC Rating
    private int policy_;   // allocation policy: TIME_SHARED or SPACE_SHARED
    private double cost_;  // cost per sec


    /**
     * Allocates a new ResDef object
     * @param arch  the architecture name
     * @param os    the Operating System name
     * @param numPE             the number of PEs (Processing Elements)
     * @param PE_SPEC_Rating    the rating of those PEs
     * @param allocationPolicy  the allocatio policy of a resource
     * @param costPerSec        the cost of a PE per second
     * @pre arch != null
     * @pre os != null
     * @pre numPE >= 0
     * @pre PE_SPEC_Rating >= 0
     * @pre allocationPolicy >= 0
     * @pre costPerSec >= 0.0
     * @post $none
     */
    public ResDef(String arch, String os, int numPE, int PE_SPEC_Rating, 
                int allocationPolicy, double costPerSec)
    {
        this.arch_ = arch;
        this.OS_ = os;
        this.numPE_ = numPE;
        this.rating_ = PE_SPEC_Rating;
        this.policy_ = allocationPolicy;
        this.cost_ = costPerSec;    
    }

    /**
     * Gets the resource architecture name
     * @return architecture name
     * @pre $none
     * @post $result != null
     */
    public String getArchitecture() {
        return arch_;
    }

    /**
     * Gets the resource operating system
     * @return operating system name
     * @pre $none
     * @post $result != null
     */
    public String getOS() {
        return OS_;
    }
    
    /**
     * Gets the number of PEs (Processing Elements) a resource has
     * @return number of PEs
     * @pre $none
     * @post $result >= 0
     */
    public int getNumPE() {
        return numPE_;
    }
    
    /**
     * Gets the rating of resource's PEs
     * @return PEs' rating
     * @pre $none
     * @post $result >= 0
     */
    public int getRating() {
        return rating_;
    }

    /**
     * Gets the resource allocation policy
     * @return allocation policy
     * @pre $none
     * @post $result >= 0
     */
    public int getAllocationPolicy() {
        return policy_;
    }
    
    /**
     * Gets the resource cost per sec
     * @return cost per sec
     * @pre $none
     * @post $result >= 0.0
     */
    public double getCostPerSec() {
        return cost_;
    }

    /**
     * Sets the resource architecture name
     * @param name  resource architecture name
     * @pre name != null
     * @post $none 
     */
    public void setArchitecture(String name) {
        arch_ = name;
    }
    
    /**
     * Sets the resource operating system name
     * @param os    operating system
     * @pre os != null
     * @post $none 
     */
    public void setOS(String os) {
        OS_ = os;
    }
    
    /**
     * Sets the number of PEs (Processing Elements)
     * @param num   number of PEs
     * @pre num >= 0
     * @post $none 
     */
    public void setNumPE(int num) {
        numPE_ = num;
    }
    
    /**
     * Sets the rating for resource's PEs
     * @param rating    PEs' rating  
     * @pre rating >= 0
     * @post $none 
     */
    public void setRating(int rating) {
        rating_ = rating;
    }
    
    /**
     * Sets the resource allocation policy
     * @param policy   allocation policy
     * @see gridsim.ResourceCharacteristics
     * @pre policy >= 0
     * @post $none 
     */
    public void setAllocationPolicy(int policy) {
        policy_ = policy;
    }
    
    /**
     * Sets the resource cost per sec
     * @param cost  resource cost
     * @pre cost >= 0.0
     * @post $none 
     */
    public void setCostPerSec(double cost) {
        cost_ = cost;
    }
    
} // end class

