/*
 * Title:        Grid Broker
 * Description:  A Grid Scheduler for Application Scheduling on Grid based on
 *               Deadline and Budget Constrained Scheduling Algorithms
 * Licence:      GPL - http://www.gnu.org/copyleft/gpl.html
 * $Id: ReportWriter.java,v 1.6 2003/12/14 00:54:13 anthony Exp $
 */

package gridbroker;

import java.io.*;
import gridsim.*;
import eduni.simjava.Sim_event;

/**
 * ReportWriter class is a user-defined and optional entity.
 * It is meant for creating a
 * report at the end of each simulation by interacting with the
 * <tt>gridsim.Statistics</tt> entity. If the user does not want to create
 * a report, then he/she can pass <tt>null</tt> as the name of the ReportWriter
 * entity. Note that users can chose any name for this entity and for the class
 * name since all entities are identified by their name defined at runtime.
 *
 * @author       Manzur Murshed and Rajkumar Buyya
 * @version      2.1, June 2003
 * @invariant $none
 */
public class ReportWriter extends GridSim
{
    private int numUser_;
    private int numResource_;
    private String reportFilename_;
    private double[] colValue_;       // values to be printed for column label

    // statistics category: e.g., "*.USER.TimeUtilization"
    private String[] statCategory_;

    // start of a new simulation with a new scenario
    private boolean newGridReportFlag_;

    // true - write report to next line.
    // false - append to the previous row (after inserting TAB ("\t") character
    private boolean matrixReportNextRowFlag_;

    // the value to be printed at the start when matrixReportNextRowFlag
    // is true
    private double rowValue_;


    /**
     * Allocates a new ReportWriter object
     * @param entityName   the name to be associated with this entity
     * @param numUser   the number of users
     * @param numResource   the number of resources
     * @param reportFilename   the report file name
     * @param statCategory     the statistics category
     * @param newGridReportFlag    a flag to denote the start of a new
     *                             simulation with a new scenario
     * @param colValue      the values to be printed for column label
     * @param matrixReportNextRowFlag    if it is <tt>true</tt>, then writes
     *                  the report to the next line, otherwise it appends to the
     *                  previous row
     * @param rowValue      the value to be printed at the start when
     *                      <tt>matrix_report_new_row_flag</tt> is true
     * @throws Exception This happens when creating this entity before
     *                   initializing GridSim package or this entity name is
     *                   <tt>null</tt> or empty
     * @see gridsim.GridSim#init(int, Calendar, boolean, String[], String[],
     *          String)
     * @pre entityName != null
     * @pre numUser >= 0
     * @pre numResource >= 0
     * @pre reportFilename != null
     * @post $none
     */
    public ReportWriter( String entityName, int numUser,
                int numResource, String reportFilename,
                String [] statCategory, boolean newGridReportFlag,
                double [] colValue, boolean matrixReportNextRowFlag,
                double rowValue ) throws Exception
    {
        super(entityName);
        this.numUser_ = numUser;
        this.numResource_ = numResource;
        this.reportFilename_ = reportFilename;
        this.statCategory_ = statCategory;
        this.newGridReportFlag_ = newGridReportFlag;
        this.colValue_ = colValue;
        this.matrixReportNextRowFlag_ = matrixReportNextRowFlag;
        this.rowValue_ = rowValue;
    }

    /**
     * A method that waits until the end of a simulation to be able to generate
     * a report
     * @pre $none
     * @post $none
     */
    public void body()
    {
        Sim_event ev = new Sim_event();

        // Wait for END_OF_SIMULATION
        for ( sim_get_next(ev); ev.get_tag() != GridSimTags.END_OF_SIMULATION;
                sim_get_next(ev) )
        {
            System.out.println("ReportWriter.body() : Unwanted event.");
        }

        generateReport();

        // GridStatistics should be shutdown after generating report
        super.shutdownGridStatisticsEntity();
    }

    //////////////////// INTERNAL METHODS /////////////////////////////

    /**
     * Generates the report
     * @pre $none
     * @post $none
     */
    private void generateReport()
    {
        try
        {
            // open file for append mode.
            FileWriter fw = new FileWriter(reportFilename_, true);
            BufferedWriter bw = new BufferedWriter(fw);
            PrintWriter outFile = new PrintWriter(bw);

            outFile.print(numUser_ + "\t" + numResource_);

            int length = statCategory_.length;
            for (int i = 0; i < length; i++)
            {
                super.send(GridSim.getGridStatisticsEntityId(),
                        GridSimTags.SCHEDULE_NOW,
                        GridSimTags.RETURN_ACC_STATISTICS_BY_CATEGORY,
                        statCategory_[i]);

                Accumulator acc = (Accumulator) super.receiveEventObject();
                outFile.print( "\t" + acc.getMean() );

                // if reportFilename_ = "abc" and first category, then cateory
                // filename is "abc.0", for 2nd, "abc.1"...
                String categoryReportFilename = reportFilename_ + "." + i;
                writeToCategoryReportFile(categoryReportFilename, acc);
            }

            outFile.println();
            outFile.close();
        }
        catch (IOException e)
        {
            System.out.println("ReportWriter.generateReport() : Error - " +
                    "unable to open \"" + reportFilename_ + "\"");
        }
    }

    /**
     * Writes down the category into the report file
     * @param categoryName    category report file name
     * @param ac              an Accumulator object
     * @pre categoryName != null
     * @pre ac != null
     * @post $none
     */
    private void writeToCategoryReportFile(String categoryName, Accumulator ac)
    {
        try
        {
            // open file for append mode.
            FileWriter fw = new FileWriter(categoryName, true);
            BufferedWriter bw = new BufferedWriter(fw);
            PrintWriter outFile = new PrintWriter(bw);

            if (newGridReportFlag_)
            {
                outFile.println();  // start on newline

                int length = colValue_.length;
                for (int i = 0; i < length; i++) {
                    outFile.print( "\t" + colValue_[i] );
                }
            }

            if (matrixReportNextRowFlag_)
            {
                outFile.println();  // start on newline.
                outFile.print( rowValue_ + "\t" + ac.getMean() );
            }
            else {
                outFile.print( "\t" + ac.getMean() );
            }

            outFile.close();
        }
        catch (IOException e)
        {
            System.out.println("ReportWriter.writeToCategoryReportFile() : " +
                    "Error - unable to open \"" + reportFilename_ + "\"");
        }
    }

} // end class

