/*
 * Title:        Grid Broker
 * Description:  A Grid Scheduler for Application Scheduling on Grid based on
 *               Deadline and Budget Constrained Scheduling Algorithms
 * Licence:      GPL - http://www.gnu.org/copyleft/gpl.html
 * $Id: Experiment.java,v 1.8 2003/06/30 08:21:46 anthony Exp $
 */

package gridbroker;

import gridsim.*;

/**
 * Experiment class acts as a placeholder for representing simulation
 * experiment configuration that includes synthesized application (a set of
 * Gridlets stored in GridletList) and user requirements such as D- and B-factor
 * or deadline and budget constraints, and optimization strategy.
 * <p>
 * It provides methods for updating and querying the experiment parameters and
 * status. The user entity invokes the broker entity and passes its requirements
 * via the experiment object. On receiving an experiment from its user, the
 * broker schedules Gridlets according to the optimization policy set for the
 * experiment.
 *
 * @author       Manzur Murshed and Rajkumar Buyya
 * @version      2.1, June 2003
 * @invariant $none
 */
public class Experiment
{
    private int id_;                // experiment ID
    private GridletList glList_;    // the list of Gridlets
    
    // if true, compute experiment deadline & budget from d-factor and b-factor
    private boolean dbFactorFlag_;   
    private double dFactor_;        // Deadline Factor
    private double bFactor_;        // Budget Factor
    
    // Absolute Deadline: User Defined or Calculated based on d-factor
    private double deadline_;    
    private double budget_;         // calculated based on b-factor
    private double startTime_;      // set by user or current time
    private double endTime_;        // endTime_ <= startTime_ + deadline
    private String reportFile_;     // the name of a report file.
    
    // shows the order in which resources schedule is reported.
    private String [] resNameList_;   

    private int optStrategy_;       // optimization strategy

    /** Aims for Cost minimization */
    public static final int OPTIMIZE_COST = 0;  
    
    /** 
     * Aims for Cost minimization. However, powerful machines come first 
     * if they have the same cost. 
     */
    public static final int OPTIMIZE_COST_PLUS = 1;
    
    /** Aims for Cost minimization and Time Optimisation if possible */
    public static final int OPTIMIZE_COST_TIME = 2;

    /** Aims for Time minimization */
    public static final int OPTIMIZE_TIME = 3;  

    /** Aims for NO minimization */
    public static final int OPTIMIZE_NONE = 4;  

    
    /**
     * Allocates a new Experiment object 
     * @param id        experiment ID
     * @param glList    A list containing Gridlets that has details on the 
     *                  list of the jobs
     * @param optStrategy  the optimization strategy
     * @param factorFlag   <tt>true</tt> if the next two parameters are 
     *                     <tt>d-factor</tt> and <tt>b-factor</tt>, and 
     *                     the User expects the Broker to 
     *                     determine actual deadline and budget based on these
     *                     two relaxation parameters
     * @param D_value   if <tt>factorFlag</tt> is <tt>true</tt>, then it is 
     *                  <tt>d-factor</tt>, else it is deadline value
     * @param B_value   if <tt>factorFlag</tt> is <tt>true</tt>, then it is 
     *                  <tt>b-factor</tt>, else it is budget value
     * @param reportFile   the name of the file into which scheduling 
     *                     report is to placed
     * @param resNameList  a list of resource names
     * @pre id >= 0
     * @pre glList != null
     * @pre reportFile != null
     * @pre resNameList != null
     * @post $none
     */
    public Experiment(int id, GridletList glList, int optStrategy, 
                boolean factorFlag, double D_value, double B_value, 
                String reportFile, String[] resNameList)
    {
        this.id_ = id;
        this.glList_ = glList;
        this.optStrategy_ = optStrategy;
        this.dbFactorFlag_ = factorFlag;
        
        if (this.dbFactorFlag_)  // is true
        {
            this.dFactor_ = D_value;
            this.bFactor_ = B_value;
        }
        else  // the user provides absolute deadline and budget values
        {  
            this.deadline_ = D_value;
            this.budget_ = B_value;
        }

        this.reportFile_ = reportFile;
        this.resNameList_ = resNameList;
        this.startTime_ = 0.0;
        this.endTime_ = 0.0;
    }

    /**
     * Gets the experiment ID
     * @return the experiment ID
     * @deprecated As of GridBroker 2.1, replaced by 
     *             {@link #getExperimentID()}
     * @pre $none
     * @post $result >= 0
     */
    public int GetID() {
        return this.getExperimentID();
    }

    /**
     * Gets the experiment ID
     * @return the experiment ID
     * @pre $none
     * @post $result >= 0
     */
    public int getExperimentID() {
        return id_;
    }

    /**
     * Gets a list of Gridlets
     * @return an object of GridletList
     * @deprecated As of GridBroker 2.1, replaced by 
     *             {@link #getGridletList()}
     * @see gridsim.GridletList
     * @pre $none
     * @post $result != null
     */
    public GridletList GetGridletList() {
        return this.getGridletList();
    }

    /**
     * Gets a list of Gridlets
     * @return an object of GridletList
     * @see gridsim.GridletList
     * @pre $none
     * @post $result != null
     */
    public GridletList getGridletList() {
        return glList_;
    }

    /**
     * Gets a list of resources' names
     * @return an array of String the contains resources' names
     * @deprecated As of GridBroker 2.1, replaced by 
     *             {@link #getResourceNameList()}
     * @pre $none
     * @post $result != null
     */
    public String[] GetResourcesNameList() {
        return this.getResourceNameList();
    }

    /**
     * Gets a list of resources' names
     * @return an array of String the contains resources' names
     * @pre $none
     * @post $result != null
     */
    public String[] getResourceNameList() {
        return resNameList_;
    }

    /**
     * Gets the optimization strategy represented in an integer number
     * @return the optimization strategy
     * @deprecated As of GridBroker 2.1, replaced by 
     *             {@link #getOptimizationStrategy()}
     * @pre $none
     * @post $none
     */
    public int GetOptimizationStrategy() {
        return this.getOptimizationStrategy();
    }

    /**
     * Gets the optimization strategy represented in an integer number
     * @return the optimization strategy
     * @pre $none
     * @post $none
     */
    public int getOptimizationStrategy() {
        return optStrategy_;
    }
    
    /**
     * Gets the optimization strategy represented in a String object
     * @return the optimization strategy or <tt>null</tt> if no strategy 
     *         detected
     * @deprecated As of GridBroker 2.1, replaced by 
     *             {@link #getOptimizationStrategyName()}
     * @pre $none
     * @post $none
     */
    public String GetOptimizationStrategyName() {
        return this.getOptimizationStrategyName();
    }

    /**
     * Gets the optimization strategy represented in a String object
     * @return the optimization strategy or <tt>null</tt> if no strategy 
     *         detected
     * @pre $none
     * @post $none
     */
    public String getOptimizationStrategyName()
    {
        String nameString = null; 
        switch (optStrategy_)
        {
            case OPTIMIZE_COST:
                nameString = "OPTIMIZE_COST";
                break;
                
            case OPTIMIZE_COST_PLUS:
                nameString = "OPTIMIZE_COST_PLUS";
                break;
                
            case OPTIMIZE_COST_TIME:
                nameString = "OPTIMIZE_COST_TIME";
                break;
                
            case OPTIMIZE_TIME:
                nameString = "OPTIMIZE_TIME";
                break;
                
            case OPTIMIZE_NONE:
                nameString = "OPTIMIZE_NONE";
                break;

            default:
                break;
        }
        
        return nameString;
    }

    /**
     * Gets the deadline and budget factor flag
     * @return if <tt>true</tt>, then computes the experiment deadline and 
     *         budget from d-factor and b-factor
     * @deprecated As of GridBroker 2.1, replaced by 
     *             {@link #getFactorFlag()}
     * @pre $none
     * @post $none
     */
    public boolean Get_DB_factor_flag() {
        return this.getFactorFlag();
    }

    /**
     * Gets the deadline and budget factor flag
     * @return if <tt>true</tt>, then computes the experiment deadline and 
     *         budget from d-factor and b-factor
     * @pre $none
     * @post $none
     */
    public boolean getFactorFlag() {
        return dbFactorFlag_;
    }

    /**
     * Gets the budget factor
     * @return the budget factor
     * @deprecated As of GridBroker 2.1, replaced by 
     *             {@link #getBudgetFactor()}
     * @pre $none
     * @post $result >= 0.0
     */
    public double GetB_factor() {
        return this.getBudgetFactor();
    }

    /**
     * Gets the budget factor
     * @return the budget factor
     * @pre $none
     * @post $result >= 0.0
     */
    public double getBudgetFactor() {
        return bFactor_;
    }

    /**
     * Gets the deadline factor
     * @return the deadline factor
     * @deprecated As of GridBroker 2.1, replaced by 
     *             {@link #getDeadlineFactor()}
     * @pre $none
     * @post $result >= 0.0
     */
    public double GetD_factor() {
        return this.getDeadlineFactor();
    }

    /**
     * Gets the deadline factor
     * @return the deadline factor
     * @pre $none
     * @post $result >= 0.0
     */
    public double getDeadlineFactor() {
        return dFactor_;
    }

    /**
     * Sets the experiment deadline
     * @param deadline  the deadline value
     * @deprecated As of GridBroker 2.1, replaced by 
     *             {@link #setDeadline(double)}
     * @pre deadline >= 0.0
     * @post $none
     */
    public void SetDeadline(double deadline) {
        this.setDeadline(deadline);
    }

    /**
     * Sets the experiment deadline
     * @param deadline  the deadline value
     * @pre deadline >= 0.0
     * @post $none
     */
    public void setDeadline(double deadline) {
        deadline_ = deadline;
    }

    /**
     * Gets the experiment deadline
     * @return the deadline value
     * @deprecated As of GridBroker 2.1, replaced by 
     *             {@link #getDeadline()}
     * @pre $none
     * @post $result >= 0.0
     */
    public double GetDeadline() {
        return this.getDeadline();
    }

    /**
     * Gets the experiment deadline
     * @return the deadline value
     * @pre $none
     * @post $result >= 0.0
     */
    public double getDeadline() {
        return deadline_;
    }

    /**
     * Gets the deadline time
     * @return the deadline time
     * @deprecated As of GridBroker 2.1, replaced by 
     *             {@link #getDeadlineTime()}
     * @pre $none
     * @post $result >= 0.0
     */
    public double GetDeadlineTime() {
        return this.getDeadlineTime();
    }

    /**
     * Gets the deadline time
     * @return the deadline time
     * @pre $none
     * @post $result >= 0.0
     */
    public double getDeadlineTime() {
        return startTime_ + deadline_;
    }

    /**
     * Sets the experiment budget
     * @param budget    the budget value
     * @deprecated As of GridBroker 2.1, replaced by 
     *             {@link #setBudget(double)}
     * @pre budget >= 0.0
     * @post $none
     */
    public void SetBudget(double budget) {
        this.setBudget(budget);
    }

    /**
     * Sets the experiment budget
     * @param budget    the budget value
     * @pre budget >= 0.0
     * @post $none
     */
    public void setBudget(double budget) {
        budget_ = budget;
    }

    /**
     * Gets the experiment budget 
     * @return the budget value
     * @deprecated As of GridBroker 2.1, replaced by 
     *             {@link #getBudget()}
     * @pre $none
     * @post $result >= 0.0
     */
    public double GetBudget() {
        return this.getBudget();
    }

    /**
     * Gets the experiment budget 
     * @return the budget value
     * @pre $none
     * @post $result >= 0.0
     */
    public double getBudget() {
        return budget_;
    }

    /**
     * Sets the experiment deadline and budget
     * @param deadline      the deadline value
     * @param budget        the budget value
     * @deprecated As of GridBroker 2.1, replaced by 
     *             {@link #setDeadlineBudget(double, double)}
     * @pre deadline >= 0.0
     * @pre budget >= 0.0
     * @post $none
     */
    public void SetDeadlineBudget(double deadline, double budget) {
        this.setDeadlineBudget(deadline, budget);
    }

    /**
     * Sets the experiment deadline and budget
     * @param deadline      the deadline value
     * @param budget        the budget value
     * @pre deadline >= 0.0
     * @pre budget >= 0.0
     * @post $none
     */
    public void setDeadlineBudget(double deadline, double budget) 
    {
        deadline_ = deadline;
        budget_ = budget;
    }

    /**
     * Gets the name of a report file
     * @return the file name
     * @deprecated As of GridBroker 2.1, replaced by
     *             {@link #getReportFileName()}
     * @pre $none
     * @post $result != null
     */
    public String GetReportFileName() {
        return this.getReportFileName();
    }

    /**
     * Gets the name of a report file
     * @return the file name
     * @pre $none
     * @post $result != null
     */
    public String getReportFileName() {
        return reportFile_;
    }

    /**
     * Sets the current time as the experiment start time
     * @deprecated As of GridBroker 2.1, replaced by
     *             {@link #setStartTime()}
     * @pre $none
     * @post $none
     */
    public void SetStartTime() {
        this.setStartTime();
    }

    /**
     * Sets the current time as the experiment start time
     * @pre $none
     * @post $none
     */
    public void setStartTime() {
        startTime_ = GridSim.clock();
    }

    /**
     * Gets the experiment start time
     * @return the start time
     * @deprecated As of GridBroker 2.1, replaced by
     *             {@link #getStartTime()}
     * @pre $none
     * @post $result >= 0.0
     */
    public double GetStartTime() {
        return this.getStartTime();
    }

    /**
     * Gets the experiment start time
     * @return the start time
     * @pre $none
     * @post $result >= 0.0
     */
    public double getStartTime() {
        return startTime_;
    }

    /**
     * Sets the experiment end time
     * @deprecated As of GridBroker 2.1, replaced by
     *             {@link #setEndTime()}
     * @pre $none 
     * @post $none
     */
    public void SetEndTime() {
        this.setEndTime();
    }

    /**
     * Sets the experiment end time
     * @pre $none 
     * @post $none
     */
    public void setEndTime() {
        endTime_ = GridSim.clock();
    }

    /**
     * Gets the experiment end time 
     * @return the end time
     * @deprecated As of GridBroker 2.1, replaced by
     *             {@link #getEndTime()}
     * @pre $none
     * @post $result >= 0.0
     */
    public double GetEndTime() {
        return this.getEndTime();
    }

    /**
     * Gets the experiment end time 
     * @return the end time
     * @pre $none
     * @post $result >= 0.0
     */
    public double getEndTime() {
        return endTime_;
    }

    /**
     * Gets the number of Gridlets
     * @return the gridlet size or <tt>0</tt> if it is empty
     * @deprecated As of GridBroker 2.1, replaced by
     *             {@link #getNumGridlet()}
     * @pre $none
     * @post $result >= 0
     */
    public int GetNoOfGridlets() {
        return this.getNumGridlet();
    }

    /**
     * Gets the number of Gridlets
     * @return the gridlet size or <tt>0</tt> if it is empty
     * @pre $none
     * @post $result >= 0
     */
    public int getNumGridlet() {
        return glList_.size();
    }

    /**
     * Gets the total budget spent on a successful Gridlets' execution.
     * Calculated based on the actual CPU time * Cost per second of a Gridlet.
     * @return the budget spent
     * @deprecated As of GridBroker 2.1, replaced by
     *             {@link #getBudgetSpent()}
     * @pre $none
     * @post $result >= 0
     */
    public double GetBudgetSpent() {
        return this.getBudgetSpent();
    }

    /**
     * Gets the total budget spent on a successful Gridlets' execution.
     * Calculated based on the actual CPU time * Cost per second of a Gridlet.
     * @return the budget spent
     * @pre $none
     * @post $result >= 0
     */
    public double getBudgetSpent() 
    {
        double expenses = 0;
        Gridlet gl; 
        for (int i = 0; i < glList_.size(); i++)
        {
            gl = (Gridlet) glList_.get(i);
            if (gl.getGridletStatus() == Gridlet.SUCCESS) {
                expenses += gl.getActualCPUTime() * gl.getCostPerSec();
            }
        }
        
        return expenses;
    }

    /**
     * Gets the total number of successful Gridlets
     * @return the total number or <tt>0</tt> if none
     * @deprecated As of GridBroker 2.1, replaced by
     *             {@link #getNumSuccessfulGridlet()}
     * @pre $none
     * @post $result >= 0 
     */
    public int GetNoOfSuccessfulGridlets() {
        return this.getNumSuccessfulGridlet();
    }

    /**
     * Gets the total number of successful Gridlets
     * @return the total number or <tt>0</tt> if none
     * @pre $none
     * @post $result >= 0 
     */
    public int getNumSuccessfulGridlet() 
    {
        int count = 0;
        Gridlet gl;
        for (int i = 0; i < glList_.size(); i++)
        {
            gl = (Gridlet) glList_.get(i);
            if (gl.getGridletStatus() == Gridlet.SUCCESS) {
                count++;
            }
        }
        
        return count;
    }

    /**
     * Checks whether the Gridlets have been executed successfully or not
     * @return <tt>true</tt> if the Gridlets are finished, otherwise 
     *         <tt>false</tt>
     * @deprecated As of GridBroker 2.1, replaced by
     *             {@link #isSuccessful()}
     * @pre $none 
     * @post $none
     */
    public boolean IsSuccessful() {
        return this.isSuccessful();
    }

    /**
     * Checks whether the Gridlets have been executed successfully or not
     * @return <tt>true</tt> if the Gridlets are finished, otherwise 
     *         <tt>false</tt>
     * @pre $none 
     * @post $none
     */
    public boolean isSuccessful() 
    {
        if ( getNumSuccessfulGridlet() == getNumGridlet() ) {
            return true;
        }
        else {
            return false;
        }
    }

    /**
     * Gets the completion ratio based on the number of successful
     * Gridlets with the total Gridlets.
     * @return the completion ratio
     * @deprecated As of GridBroker 2.1, replaced by
     *             {@link #getGridletCompletionFactor()}
     * @pre $none
     * @post $result >= 0.0
     */
    public double GridletCompletionFactor() {
        return this.getGridletCompletionFactor();
    }

    /**
     * Gets the completion ratio based on the number of successful
     * Gridlets with the total Gridlets.
     * @return the completion ratio
     * @pre $none
     * @post $result >= 0.0
     */
    public double getGridletCompletionFactor() 
    {
        double factor = 0.0;
        int numGridlet = getNumGridlet();
        if (numGridlet != 0) {
            factor = (double) getNumSuccessfulGridlet() / numGridlet;
        }

        return factor;
    }

    /**
     * Gets the time utilization factor
     * @return the time utilization factor
     * @deprecated As of GridBroker 2.1, replaced by
     *             {@link #getTimeUtilFactor()}
     * @pre $none
     * @post $none
     */
    public double TimeUtilizationFactor() {
        return this.getTimeUtilFactor();
    }

    /**
     * Gets the time utilization factor
     * @return the time utilization factor
     * @pre $none
     * @post $none
     */
    public double getTimeUtilFactor() {
        return (endTime_ - startTime_) / deadline_;
    }

    /**
     * Gets the budget utilization factor
     * @return the budget utilization factor
     * @deprecated As of GridBroker 2.1, replaced by
     *             {@link #getBudgetUtilFactor()}
     * @pre $none
     * @post $none
     */
    public double BudgetUtilizationFactor() {
        return this.getBudgetUtilFactor();
    }

    /**
     * Gets the budget utilization factor
     * @return the budget utilization factor
     * @pre $none
     * @post $none
     */
    public double getBudgetUtilFactor() {
        return getBudgetSpent() / budget_;
    }

    /**
     * Accumulates the length of all the Gridlets
     * @return an Accumulator object
     * @deprecated As of GridBroker 2.1, replaced by
     *             {@link #getAllGridletLengthAccumulator()}
     * @see gridsim.Accumulator
     * @pre $none
     * @post $result != null
     */
    public Accumulator GetAllGridletsLengthAccumulator() {
        return this.getAllGridletLengthAccumulator();
    }

    /**
     * Accumulates the length of all the Gridlets
     * @return an Accumulator object
     * @see gridsim.Accumulator
     * @pre $none
     * @post $result != null
     */
    public Accumulator getAllGridletLengthAccumulator() 
    {
        Accumulator accLength = new Accumulator();
        for (int i = 0; i < glList_.size(); i++) {
            accLength.add( ((Gridlet) glList_.get(i)).getGridletLength() );
        }

        return accLength;
    }
    
} // end class

