/*
 * Title:        Grid Broker
 * Description:  A Grid Scheduler for Application Scheduling on Grid based on
 *               Deadline and Budget Constrained Scheduling Algorithms
 * Licence:      GPL - http://www.gnu.org/copyleft/gpl.html
 * $Id: BrokerResource.java,v 1.6 2003/06/30 05:10:55 anthony Exp $
 */

package gridbroker;

import gridsim.*;
import eduni.simjava.Sim_event;


/**
 * BrokerResource class acts as a placeholder for the broker to maintain a
 * detailed record on the resources it uses for processing user applications. It
 * maintains resource characteristics, a list of Gridlets assigned to the
 * resource, the actual amount of MIPS (Millions Instruction Per Second)
 * available to the user, and a report on the Gridlets processed. These
 * measurements help in extrapolating and predicating the resource performance
 * from the user point of view and aid in scheduling job dynamically at
 * runtime.
 *
 * @author       Manzur Murshed and Rajkumar Buyya
 * @version      2.1, June 2003
 * @invariant $none
 */
public class BrokerResource
{
    /** A ResourceCharacteristics object */
    public ResourceCharacteristics resource;

    /** Denotes the latest load */
    public double LatestLoad;

    /** List of Gridlets that are currently assigned to a resource */
    public GridletList glList;

    /** Number of Gridlets that are finished so far */
    public int NoOfGridletsFinishedSoFar;

    /** Number of Gridlets that are dispatched so far */
    public int NoOfGridletsDispatchedSoFar;

    /** Processing expenses so far */
    public double ProcessingExpensesSoFar;

    /** Gridlets on a resource */
    public static final int PARAM_GRIDLETS_ON_RESOURCE = 1;

    /** Gridlets that are committed */
    public static final int PARAM_GRIDLETS_COMMITTED_FOR_RESOURCE = 2;

    /** Gridlets that are finished */
    public static final int PARAM_GRIDLETS_FINISHED = 3;

    /** Processing expenses */
    public static final int PARAM_PROCESSING_EXPENSES = 4;

    // Total MIPS actually available to the user on a resource
    private double availableMIPS_;

    // THIS IS USED IN SCHEDULING DECISION PURPOSE.
    private double availableMIPS_PrevSchedule_;


    /**
     * Allocates a new BrokerResource object
     * @param resObj   an object of ResourceCharacteristics
     * @param latestLoad    the latest resource load
     * @see gridsim.ResourceCharacteristics
     * @pre resObj != null
     * @pre latestLoad >= 0.0
     * @post $none
     */
    public BrokerResource(ResourceCharacteristics resObj, double latestLoad)
    {
        this.resource = resObj;
        this.LatestLoad = latestLoad;
        glList = new GridletList();

        try {
            availableMIPS_ = resource.getMIPSRating();
        }
        catch (Exception e) {
            availableMIPS_ = 0.0;
        }

        // FIRST time both last and current will be same.
        availableMIPS_PrevSchedule_ = availableMIPS_;
        NoOfGridletsFinishedSoFar = 0;
        NoOfGridletsDispatchedSoFar = 0;
        ProcessingExpensesSoFar = 0;
    }

    /**
     * Gets the parameter value
     * @param parameter     the type of a parameter
     * @return the parameter value
     * @deprecated As of GridBroker 2.1, replaced by
     *             {@link #getParameterValue(int)}
     * @pre parameter >= 1 && parameter <= 4
     * @post $result >= 0.0
     */
    public double GetParameterValue(int parameter) {
        return this.getParameterValue(parameter);
    }

    /**
     * Gets the parameter value
     * @param parameter     the type of a parameter
     * @return the parameter value
     * @pre parameter >= 1 && parameter <= 4
     * @post $result >= 0.0
     */
    public double getParameterValue(int parameter)
    {
        double value = -1;  // undefined
        switch (parameter)
        {
            case PARAM_GRIDLETS_ON_RESOURCE:
                value = getNumGridletInQueue() + getNumGridletInExec();
                break;

            case PARAM_GRIDLETS_COMMITTED_FOR_RESOURCE:
                value = getNumGridletCommitted();
                break;

            case PARAM_GRIDLETS_FINISHED:
                value = NoOfGridletsFinishedSoFar;
                break;

            case PARAM_PROCESSING_EXPENSES:
                value = ProcessingExpensesSoFar;
                break;

            default:
                break;
        }
        return value;
    }

    /**
     * Gets the available MIPS
     * @return the available MIPS
     * @deprecated As of GridBroker 2.1, replaced by {@link #getAvailableMIPS()}
     * @pre $none
     * @post $result >= 0
     */
    public double GetAvailableMIPS() {
        return this.getAvailableMIPS();
    }

    /**
     * Gets the available MIPS
     * @return the available MIPS
     * @pre $none
     * @post $result >= 0
     */
    public double getAvailableMIPS() {
        return availableMIPS_;
    }

    /**
     * Gets the available MIPS based on previous schedule
     * @return the available MIPS
     * @deprecated As of GridBroker 2.1, replaced by
     *             {@link #getAvailableMIPS_PreviousSchedule()}
     * @pre $none
     * @post $result >= 0
     */
    public double GetAvailableMIPS_PreviousSchedule() {
        return this.getAvailableMIPS_PreviousSchedule();
    }

    /**
     * Gets the available MIPS based on previous schedule
     * @return the available MIPS
     * @pre $none
     * @post $result >= 0
     */
    public double getAvailableMIPS_PreviousSchedule() {
        return availableMIPS_PrevSchedule_;
    }

    /**
     * Sets the available MIPS based on previous schedule
     * @deprecated As of GridBroker 2.1, replaced by
     *             {@link #setAvailableMIPS_PreviousSchedule()}
     * @pre $none
     * @post $none
     */
    public void SetAvailableMIPS_PreviousSchedule() {
        this.setAvailableMIPS_PreviousSchedule();
    }

    /**
     * Sets the available MIPS based on previous schedule
     * @pre $none
     * @post $none
     */
    public void setAvailableMIPS_PreviousSchedule() {
        availableMIPS_PrevSchedule_ = availableMIPS_;
    }

    /**
     * Gets the resource share since the last schedule.
     * If the value is greater than 1, then higher share is available.
     * @return the resource share
     * @deprecated As of GridBroker 2.1, replaced by
     *             {@link #getResourceShareVariation()}
     * @pre $none
     * @post $result >= 0
     */
    public double ResourceShareVariation() {
        return this.getResourceShareVariation();
    }

    /**
     * Gets the resource share since the last schedule.
     * If the value is greater than 1, then higher share is available.
     * @return the resource share or <tt>0</tt> if not available
     * @pre $none
     * @post $none
     */
    public double getResourceShareVariation()
    {
        // variation from previous schedule
        double variation = 0.0;
        if (availableMIPS_PrevSchedule_ != 0.0) {
            variation = availableMIPS_ / availableMIPS_PrevSchedule_;
        }

        return variation;
    }

    /**
     * Updates the total available MIPS
     * @param glFinished    a Gridlet object that has finished
     * @param brokerMaxGridletPerPELimit    the max. number of Gridlet per PE
     *              (Processing Element)
     * @see gridsim.Gridlet
     * @deprecated As of GridBroker 2.1, replaced by
     *             {@link #updateAvailableMIPS(Gridlet, int)}
     * @pre glFinished != null
     * @pre brokerMaxGridletPerPELimit >= 0
     * @post $none
     */
    public void UpdateAvailableMIPS(Gridlet glFinished,
                    int brokerMaxGridletPerPELimit)
    {
        this.updateAvailableMIPS(glFinished, brokerMaxGridletPerPELimit);
    }

    /**
     * Updates the total available MIPS
     * @param glFinished    a Gridlet object that has finished
     * @param brokerMaxGridletPerPELimit    the max. number of Gridlet per PE
     *              (Processing Element)
     * @see gridsim.Gridlet
     * @pre glFinished != null
     * @pre brokerMaxGridletPerPELimit >= 0
     * @post $none
     */
    public void updateAvailableMIPS(Gridlet glFinished,
                    int brokerMaxGridletPerPELimit)
    {
        double clockTime = glFinished.getWallClockTime();
        double MIPSPerGridlet = 0.0;
        if (clockTime != 0.0) {
            MIPSPerGridlet = glFinished.getGridletLength() / clockTime;
        }

        int numPE = resource.getNumPE();
        double MaxGridletsPerPE = 0.0;
        if (numPE > 0) {
            MaxGridletsPerPE = Math.max(1.0, (getNumGridletInExec()+1) / numPE);
        }

        // make sure that MaxGridletsPerPE is NOT more than broker
        // actually assigned
        MaxGridletsPerPE = Math.min(MaxGridletsPerPE,
                                brokerMaxGridletPerPELimit);

        availableMIPS_ = MIPSPerGridlet * MaxGridletsPerPE * numPE;
    }

    /**
     * Updates the available MIPS during simulation schedule
     * @deprecated As of GridBroker 2.1, replaced by
     *             {@link #updateAvailableMIPSDuringSchedule()}
     * @pre $none
     * @post $none
     */
    public void UpdateAvailableMIPSDuringSchedule() {
        this.updateAvailableMIPSDuringSchedule();
    }

    /**
     * Updates the available MIPS during simulation schedule
     * @pre $none
     * @post $none
     */
    public void updateAvailableMIPSDuringSchedule()
    {
        // save AvailableMIPS in the last schedule
        availableMIPS_PrevSchedule_ = availableMIPS_;
    }

    /**
     * Identifies the actual MIPS available per PE (Processing Element)
     * @return the actual MPIS available for the current user
     * @deprecated As of GridBroker 2.1, replaced by
     *             {@link #getAvailableMIPSPerPE()}
     * @pre $none
     * @post $result >= 0
     */
    public double GetAvailableMIPSPerPE() {
        return this.getAvailableMIPSPerPE();
    }

    /**
     * Identifies the actual MIPS available per PE (Processing Element)
     * @return the actual MPIS available for the current user
     * @pre $none
     * @post $result >= 0
     */
    public double getAvailableMIPSPerPE()
    {
        double available = 0.0;
        int numPE = resource.getNumPE();
        if (numPE > 0) {
            available = availableMIPS_ / numPE;
        }

        return available;
    }

    /**
     * Gets the time at which the first slot is available, considering the
     * current committed Gridlets
     * @return the available time
     * @deprecated As of GridBroker 2.1, replaced by
     *             {@link #getFirstAvailableSlotTime()}
     * @pre $none
     * @post $result >= 0
     */
    public double FirstAvailableSlotTime() {
        return this.getFirstAvailableSlotTime();
    }

    /**
     * Gets the time at which the first slot is available, considering the
     * current committed Gridlets
     * @return the available time
     * @pre $none
     * @post $result >= 0
     */
    public double getFirstAvailableSlotTime()
    {
        int numPE = resource.getNumPE();
        int pe_slots_committed = 0;
        if (numPE > 0) {
            pe_slots_committed = getNumGridletCommitted() / numPE;
        }

        Accumulator gl_acc = getUnFinishedGridletLengthAccumulator();
        double time_to_process_per_gridlet = 0.0;

        if (gl_acc.getCount() != 0)
        {
            double available = getAvailableMIPSPerPE();
            if (available > 0.0) {
                time_to_process_per_gridlet = gl_acc.getMean() / available;
            }
        }

        double first_available_slot = pe_slots_committed *
                                      time_to_process_per_gridlet;

        return first_available_slot;
    }

    /**
     * Gets the expected Grid completion time given its current commitmements
     * @param gl    a Gridlet object
     * @return the expected completion time
     * @deprecated As of GridBroker 2.1, replaced by
     *             {@link #getExpectedCompletionTime(Gridlet)}
     * @pre gl != null
     * @post $result >= 0
     */
    public double ExpectedCompletionTime(Gridlet gl) {
        return this.getExpectedCompletionTime(gl);
    }

    /**
     * Gets the expected Grid completion time given its current commitmements
     * @param gl    a Gridlet object
     * @return the expected completion time
     * @pre gl != null
     * @post $result >= 0
     */
    public double getExpectedCompletionTime(Gridlet gl)
    {
        double available = getAvailableMIPSPerPE();
        double time = 0.0;
        if (available > 0.0) {
            time = getFirstAvailableSlotTime() + gl.getGridletLength() /
                   available;
        }
        return time;
    }

    /**
     * Gets an Accumulator object for the length of all gridlets to be
     * completely processed.
     * It is also include all gridlets for a Broker Resource that either in
     * READY, QUEUED, or in INEXEC state.
     * @return an Accumulator object
     * @deprecated As of GridBroker 2.1, replaced by
     *             {@link #getUnFinishedGridletLengthAccumulator()}
     * @see gridsim.Accumulator
     * @pre $none
     * @post $result != null
     */
    public Accumulator GetUnFinishedGridletsLengthAccumulator() {
        return this.getUnFinishedGridletLengthAccumulator();
    }

    /**
     * Gets an Accumulator object for the length of all gridlets to be
     * completely processed.
     * It is also include all gridlets for a Broker Resource that either in
     * READY, QUEUED, or in INEXEC state.
     * @return an Accumulator object
     * @see gridsim.Accumulator
     * @pre $none
     * @post $result != null
     */
    public Accumulator getUnFinishedGridletLengthAccumulator()
    {
        Accumulator accLength = new Accumulator();
        for (int i = 0; i < glList.size(); i++)
        {
            int status = ( (Gridlet) glList.get(i) ).getGridletStatus();
            if ( status == Gridlet.READY || status == Gridlet.QUEUED ||
                 status == Gridlet.INEXEC )
            {
                accLength.add( ((Gridlet) glList.get(i)).getGridletLength() );
            }
        }
        return accLength;
    }

    /**
     * Gets the number of System Time units (such as second) from the
     * current time to the endTime (such as Deadline)
     * @param endTime   the ending time
     * @return the remainding time period
     * @deprecated As of GridBroker 2.1, replaced by
     *             {@link #getRemainingTimePeriod(double)}
     * @pre endTime >= 0.0
     * @post $result >= 0
     */
    public double RemainingTimePeriodFromNow(double endTime) {
        return this.getRemainingTimePeriod(endTime);
    }

    /**
     * Gets the number of System Time units (such as second) from the
     * current time to the endTime (such as Deadline)
     * @param endTime   the ending time
     * @return the remainding time period
     * @pre endTime >= 0.0
     * @post $result >= 0
     */
    public double getRemainingTimePeriod(double endTime) {
        return endTime - GridSim.clock();
    }

    /**
     * Checks the available MI (Millions Instruction) on a single PE (Processing
     * Element) within the available deadline period
     * @param gl    a Gridlet object
     * @param deadlineTime  the deadline time period
     * @return <tt>true</tt> if it is sufficient for a Gridlet to run
     *         successfully, otherwise <tt>false</tt>
     * @deprecated As of GridBroker 2.1, replaced by
     *             {@link #isSufficientMIAvailableOnSinglePE(Gridlet, double)}
     * @see gridsim.Gridlet
     * @pre $none
     * @post $none
     */
    public boolean IsSufficientMIAvailableOnSinglePE(Gridlet gl,
                        double deadlineTime)
    {
        return this.isSufficientMIAvailableOnSinglePE(gl, deadlineTime);
    }

    /**
     * Checks the available MI (Millions Instruction) on a single PE (Processing
     * Element) within the available deadline period
     * @param gl    a Gridlet object
     * @param deadlineTime  the deadline time period
     * @return <tt>true</tt> if it is sufficient for a Gridlet to run
     *         successfully, otherwise <tt>false</tt>
     * @see gridsim.Gridlet
     * @pre $none
     * @post $none
     */
    public boolean isSufficientMIAvailableOnSinglePE(Gridlet gl,
                        double deadlineTime)
    {
        double GridletNeedMI = gl.getGridletLength();
        double TimeSpan = getRemainingTimePeriod(deadlineTime);
        double AvailableMIonOnePE = getAvailableMIPSPerPE() * TimeSpan;

        if (AvailableMIonOnePE >= GridletNeedMI) {
            return true;
        }
        else {
            return false;
        }
    }

    /**
     * Gets the available MIs depending on resource load from the end time
     * @param endTime   the end time
     * @return the available MIs
     * @deprecated As of GridBroker 2.1, replaced by
     *             {@link #getAvailableMI(double)}
     * @pre endTime >= 0.0
     * @post $result >= 0.0
     */
    public double GetAvailableMI(double endTime) {
        return this.getAvailableMI(endTime);
    }

    /**
     * Gets the available MIs depending on resource load from the end time
     * @param endTime   the end time
     * @return the available MIs
     * @pre endTime >= 0.0
     * @post $result >= 0.0
     */
    public double getAvailableMI(double endTime)
    {
        double TimeSpan = getRemainingTimePeriod(endTime);
        double AvailableMI = availableMIPS_ * TimeSpan;

        // deduct MIs that are already committed (Queued/InExec) due to
        // assignment of Gridlet to a resource
        Gridlet gl = null;
        for (int i = 0; i < glList.size(); i++)
        {
            gl = (Gridlet) glList.get(i);
            if (gl.getGridletStatus() != Gridlet.READY) {
                AvailableMI -= gl.getGridletLength();
            }
        }
        return AvailableMI;
    }

    /**
     * Gets the number of Gridlets committed
     * @return the number of Gridlets committed
     * @deprecated As of GridBroker 2.1, replaced by
     *             {@link #getNumGridletCommitted()}
     * @pre $none
     * @post $result >= 0
     */
    public int NoOfGridletsCommitted() {
        return this.getNumGridletCommitted();
    }

    /**
     * Gets the number of Gridlets committed
     * @return the number of Gridlets committed
     * @pre $none
     * @post $result >= 0
     */
    public int getNumGridletCommitted()
    {
        int num = getNumGridletInReady() + getNumGridletInQueue() +
                  getNumGridletInExec();
        return num;
    }

    /**
     * Gets the number of Gridlets in ready
     * @return the number of Gridlets in ready
     * @deprecated As of GridBroker 2.1, replaced by
     *             {@link #getNumGridletInReady()}
     * @pre $none
     * @post $result >= 0
     */
    public int NoOfGridletsInReady() {
        return this.getNumGridletInReady();
    }

    /**
     * Gets the number of Gridlets in ready
     * @return the number of Gridlets in ready
     * @pre $none
     * @post $result >= 0
     */
    public int getNumGridletInReady() {
        return this.getNumGridlet(Gridlet.READY);
    }

    /**
     * Gets the number of Gridlets in queue
     * @return the number of Gridlets in queue
     * @deprecated As of GridBroker 2.1, replaced by
     *             {@link #getNumGridletInQueue()}
     * @pre $none
     * @post $result >= 0
     */
    public int NoOfGridletsInQueue() {
        return this.getNumGridletInQueue();
    }

    /**
     * Gets the number of Gridlets in queue
     * @return the number of Gridlets in queue
     * @pre $none
     * @post $result >= 0
     */
    public int getNumGridletInQueue() {
        return this.getNumGridlet(Gridlet.QUEUED);
    }

    /**
     * Gets the number of Gridlets in execution
     * @return the number of Gridlets in execution
     * @deprecated As of GridBroker 2.1, replaced by
     *             {@link #getNumGridletInExec()}
     * @pre $none
     * @post $result >= 0
     */
    public int NoOfGridletsInExec() {
        return this.getNumGridletInExec();
    }

    /**
     * Gets the number of Gridlets in execution
     * @return the number of Gridlets in execution
     * @pre $none
     * @post $result >= 0
     */
    public int getNumGridletInExec() {
        return this.getNumGridlet(Gridlet.INEXEC);
    }

    /**
     * Identifies an expected processing cost on the resource
     * @param gl    a Gridlet object
     * @return the expected processing cost or <tt>0</tt> if a Grid resource
     *         or Gridlet is empty
     * @see gridsim.Gridlet
     * @deprecated As of GridBroker 2.1, replaced by
     *             {@link #getExpectedProcessingCost(Gridlet)}
     * @pre gl != null
     * @post $result >= 0.0
     */
    public double ExpectedProcessingCost(Gridlet gl) {
        return this.getExpectedProcessingCost(gl);
    }

    /**
     * Identifies an expected processing cost on the resource
     * @param gl    a Gridlet object
     * @return the expected processing cost or <tt>0</tt> if a Grid resource
     *         or Gridlet is empty
     * @see gridsim.Gridlet
     * @pre gl != null
     * @post $result >= 0.0
     */
    public double getExpectedProcessingCost(Gridlet gl)
    {
        double result = gl.getGridletLength() * resource.getCostPerMI();
        if (result < 0.0) {
            result = 0.0;
        }
        return result;
    }

    ///////////////////////// PRIVATE METHODS /////////////////////////

    /**
     * Gets the number of Gridlets based on a specified Gridlet status
     * @param gridletStatus     the Gridlet status
     * @return the number of Gridlets
     * @pre gridletStatus >= 0
     * @post $result >= 0
     */
    private int getNumGridlet(int gridletStatus)
    {
        int count = 0;
        Gridlet gl = null;
        for (int i = 0; i < glList.size(); i++)
        {
            gl = (Gridlet) glList.get(i);
            if (gl.getGridletStatus() == gridletStatus) {
                count++;
            }
        }
        return count;
    }

} // end class

