/*
 * Title:        Grid Broker
 * Description:  A Grid Scheduler for Application Scheduling on Grid based on
 *               Deadline and Budget Constrained Scheduling Algorithms
 * Licence:      GPL - http://www.gnu.org/copyleft/gpl.html
 * $Id: Broker.java,v 1.7 2003/06/30 05:10:55 anthony Exp $
 */

package gridbroker;

import gridbroker.*;
import java.util.*;
import java.io.*;
import gridsim.*;
import eduni.simjava.Sim_event;


/**
 * Broker class simulates the Grid resource broker. On receiving an experiment
 * from the user entity, it carries out resource discovery, and determines
 * deadline and budget values based on D- and B-factor, and then proceeds with
 * scheduling.
 * <p>
 * Broker class schedules Gridlets on resources depending on user constraints,
 * optimization strategy, and cost of resources and their availability. When it
 * receives the results of application processing, it records parameters of
 * interest with the <tt>gridsim.Statistics</tt> entity. When it has no more
 * processing requirements, it sends the <tt>END_OF_SIMULATION</tt> event to the
 * <tt>gridsim.GridSimShutdown</tt> entity.
 *
 * @author       Manzur Murshed and Rajkumar Buyya
 * @version      2.1, June 2003
 * @invariant $none
 */
public class Broker extends GridSim
{
    private Experiment experiment_;
    private LinkedList resIDList_;    // Resource Enity ID List
    private LinkedList brokerResourceList_;
    private GridletList glUnfinishedList_;
    private GridletList glFinishedList_;
    private int gridletDispatched_;
    private int gridletReturned_;
    private double expenses_;     // the amount of budget spent so far
    private int maxGridletPerPE_; // Num Gridlets in Execution at any instance


    /**
     * Allocates a new Broker class
     * @param name          the entity name
     * @param baudRate     the communication speed
     * @throws Exception This happens when creating this entity before
     *                   initializing GridSim package or this entity name is
     *                   <tt>null</tt> or empty
     * @see gridsim.GridSim#init(int, Calendar, boolean, String[], String[],
     *          String)
     * @pre name != null
     * @pre baudRate >= 0.0
     * @post $none
     */
    public Broker(String name, double baudRate) throws Exception

    {
        super(name, baudRate);

        gridletDispatched_ = 0;
        gridletReturned_ = 0;
        expenses_ = 0.0;
        maxGridletPerPE_ = 1;

        experiment_ = null;
        resIDList_ = null;
        brokerResourceList_ = null;
        glUnfinishedList_ = null;
        glFinishedList_ = null;
    }

    ////////////// Start INTERNAL CLASSES /////////////////////////////////


    /**
     * This class can be used to sort the simulation cost
     * @invariant $none
     */
    private class OrderCost implements Comparator
    {
        /**
         * Allocates a new OrderCost object
         * @pre $none
         * @post $none
         */
        public OrderCost() {
            super();
        }

        /**
         * Compare two objects
         * @param a     the first Object to be compared
         * @param b     the second Object to be compared
         * @return the value 0 if both Objects are equal;
         *         a value less than 0 if the first Object is lexicographically
         *         less than the second Object;
         *         and a value greater than 0 if the first Object is
         *         lexicographically greater than the second Object.
         * @throws ClassCastException   <tt>a</tt> and <tt>b</tt> are expected
         *              to be of type <tt>BrokerResource</tt>
         * @see BrokerResource
         * @pre a != null
         * @pre b != null
         * @post $none
         */
        public int compare(Object a, Object b) throws ClassCastException
        {
            BrokerResource BRa = (BrokerResource) a;
            BrokerResource BRb = (BrokerResource) b;

            // Processing Cost of Resources
            Double d_c1 = new Double( BRa.resource.getCostPerMI() );
            Double d_c2 = new Double( BRb.resource.getCostPerMI() );

            return d_c1.compareTo(d_c2);
        }

    } // end internal class


    /**
     * This class can be used to sort the simulation cost and time
     * @invariant $none
     */
    private class OrderCostTime implements Comparator
    {
        /**
         * Allocates a new OrderCostTime object
         * @pre $none
         * @post $none
         */
        public OrderCostTime() {
            super();
        }

        /**
         * Compare two objects
         * @param a     the first Object to be compared
         * @param b     the second Object to be compared
         * @return the value 0 if both Objects are equal;
         *         a value less than 0 if the first Object is lexicographically
         *         less than the second Object;
         *         and a value greater than 0 if the first Object is
         *         lexicographically greater than the second Object.
         * @throws ClassCastException   <tt>a</tt> and <tt>b</tt> are expected
         *              to be of type <tt>BrokerResource</tt>
         * @see BrokerResource
         * @pre a != null
         * @pre b != null
         * @post $none
         */
        public int compare(Object a, Object b) throws ClassCastException
        {
            BrokerResource BRa = (BrokerResource) a;
            BrokerResource BRb = (BrokerResource) b;

            // Processing Cost of Resources
            Double d_c1 = new Double( BRa.resource.getCostPerMI() );
            Double d_c2 = new Double( BRb.resource.getCostPerMI() );

            // both resources cost the same price, them make powerful
            // resources as prefered first and put them to the begining
            // of the list. That is why
            // p2 and p1 are compared, instead of p1 and p2.
            if ( d_c1.equals(d_c2) )
            {
                // Processing Capacity of Resources
                Double d_p1 = new Double( BRa.resource.getMIPSRating() *
                                        (1 - BRa.LatestLoad) );
                Double d_p2 = new Double( BRb.resource.getMIPSRating() *
                                        (1 - BRb.LatestLoad) );

                return d_p2.compareTo(d_p1);
            }
            else {
                return d_c1.compareTo(d_c2);
            }
        }

    } // end internal class


    /**
     * A class to store record of earlier processing time of a given Gridlet on
     * a given resource
     * @invariant $none
     */
    private class BRGridletProcessingTime
    {
        /** A BrokerResource object */
        public BrokerResource br;

        /** Denotes Gridlet processing time */
        public double gridlet_processing_time;

        /**
         * Allocates a new BRGridletProcessingTime object
         * @param br    a BrokerResource object
         * @param gl    a Gridlet object
         * @see gridsim.Gridlet
         * @see gridbroker.BrokerResource
         * @pre br != null
         * @pre gl != null
         * @post $none
         */
        public BRGridletProcessingTime(BrokerResource br, Gridlet gl)
        {
            this.br = br;
            this.gridlet_processing_time = br.getExpectedCompletionTime(gl);
        }

    } // end internal class


    /**
     * Comparator Interface for Sorting records of BRGridletProcessingTime
     * in asceneding order
     */
    private class OrderBRGridletProcessingTime implements Comparator
    {
        /**
         * Allocates a new OrderBRGridletProcessingTime object
         * @pre $none
         * @post $none
         */
        public OrderBRGridletProcessingTime() {
            super();
        }

        /**
         * Compare two objects
         * @param a     the first Object to be compared
         * @param b     the second Object to be compared
         * @return the value 0 if both Objects are equal;
         *         a value less than 0 if the first Object is lexicographically
         *         less than the second Object;
         *         and a value greater than 0 if the first Object is
         *         lexicographically greater than the second Object.
         * @throws ClassCastException   <tt>a</tt> and <tt>b</tt> are expected
         *              to be of type <tt>BRGridletProcessingTime</tt>
         * @see gridbroker.Broker.BRGridletProcessingTime
         * @pre a != null
         * @pre b != null
         * @post $none
         */
        public int compare(Object a, Object b)
        {
            BRGridletProcessingTime brGlT1 = (BRGridletProcessingTime) a;
            BRGridletProcessingTime brGlT2 = (BRGridletProcessingTime) b;

            // Processing Time on Resources
            Double d_t1 = new Double(brGlT1.gridlet_processing_time);
            Double d_t2 = new Double(brGlT2.gridlet_processing_time);

            return d_t1.compareTo(d_t2);
        }

    } // end internal class

    ///////////////////// End of INTERNAL CLASSES //////////////////////////


    /**
     * Invokes a schedule report for every scheduling event.
     * <p>
     * In addition,
     * creates report files with <tt>".sched1"</tt>, <tt>".sched2"</tt>,
     * <tt>".sched3"</tt> and <tt>".sched31"</tt> extension.
     * @param expt      an Experiment object
     * @param BRList    a linked-list of broker resource
     * @param reportHeaderFlag    a report header flag
     * @deprecated As of GridBroker 2.1, replaced by
     *             {@link #scheduleReport(Experiment, LinkedList, boolean)}
     * @pre expt != null
     * @pre BRList != null
     * @post $none
     */
    public void ScheduleReport(Experiment expt, LinkedList BRList,
                    boolean reportHeaderFlag)
    {
        this.scheduleReport(expt, BRList, reportHeaderFlag);
    }

    /**
     * Invokes a schedule report for every scheduling event.
     * <p>
     * In addition,
     * creates report files with <tt>".sched1"</tt>, <tt>".sched2"</tt>,
     * <tt>".sched3"</tt> and <tt>".sched31"</tt> extension.
     * @param expt      an Experiment object
     * @param BRList    a linked-list of broker resource
     * @param reportHeaderFlag    a report header flag
     * @pre expt != null
     * @pre BRList != null
     * @post $none
     */
    public void scheduleReport(Experiment expt, LinkedList BRList,
                    boolean reportHeaderFlag)
    {
        // PRINT report for the User with Experiment ID = 0
        if (expt.getExperimentID() != 0) {
            return;
        }

        // print scheduling trace for experiment with ID 0 (i.e.,
        // for the first user)
        String report1_filename = expt.getReportFileName() + ".sched1";
        String report2_filename = expt.getReportFileName() + ".sched2";
        String report3_filename = expt.getReportFileName() + ".sched3";
        String report31_filename = expt.getReportFileName() + ".sched31";

        writeScheduleReport("Gridlets Finished Report", report1_filename,
            BrokerResource.PARAM_GRIDLETS_FINISHED, expt, BRList,
            reportHeaderFlag);

        writeScheduleReport("Gridlets Processing Expenses Report",
            report2_filename, BrokerResource.PARAM_PROCESSING_EXPENSES,
            expt, BRList, reportHeaderFlag);

        writeScheduleReport("Gridlets on Resource Report", report3_filename,
            BrokerResource.PARAM_GRIDLETS_ON_RESOURCE, expt, BRList,
            reportHeaderFlag);

        writeScheduleReport("Gridlets Committed for Resource Report",
            report31_filename,
            BrokerResource.PARAM_GRIDLETS_COMMITTED_FOR_RESOURCE, expt,
            BRList, reportHeaderFlag);
    }

    /**
     * Writes a schedule report
     * @param reportTitle   the title of a report
     * @param reportFileFullname    the name of a report
     * @param parameter     a resource paramter
     * @param expt          an object of Experiment
     * @param BRList        a linked-list of broker resource
     * @param reportHeaderFlag      a report header flag
     * @see gridbroker.Experiment
     * @deprecated As of GridBroker 2.1, replaced by
     *             {@link #writeScheduleReport(String, String, int, Experiment,
     *             LinkedList, boolean)}
     * @pre reportTitle != null
     * @pre reportFileFullname != null
     * @pre parameter >= 0
     * @pre expt != null
     * @pre BRList != null
     * @post $none
     */
    public void WriteScheduleReport(String reportTitle,
                    String reportFileFullname, int parameter, Experiment expt,
                    LinkedList BRList, boolean reportHeaderFlag)
    {
        this.writeScheduleReport(reportTitle, reportFileFullname,
                    parameter, expt, BRList, reportHeaderFlag);
    }

    /**
     * Writes a schedule report
     * @param reportTitle   the title of a report
     * @param reportFileFullname    the name of a report
     * @param parameter     a resource paramter
     * @param expt          an object of Experiment
     * @param BRList        a linked-list of broker resource
     * @param reportHeaderFlag      a report header flag
     * @see gridbroker.Experiment
     * @pre reportTitle != null
     * @pre reportFileFullname != null
     * @pre parameter >= 0
     * @pre expt != null
     * @pre BRList != null
     * @post $none
     */
    public void writeScheduleReport(String reportTitle,
                    String reportFileFullname, int parameter, Experiment expt,
                    LinkedList BRList, boolean reportHeaderFlag)
    {
        try
        {
            // open file for append mode.
            FileWriter fw = new FileWriter(reportFileFullname, true);
            BufferedWriter bw = new BufferedWriter(fw);
            PrintWriter reportFile = new PrintWriter(bw);

            String [] resourceNameList = expt.getResourceNameList();
            int length = resourceNameList.length;

            if (reportHeaderFlag)
            {
                // TIME-STAMP "\t" Resource1-Name "\t" Resource2-Name "\t"
                // Resource3-Name ...
                reportFile.println(reportTitle);  // start on newline.
                reportFile.println("Deadline: " + expt.getDeadline() +
                        " Budget: " + expt.getBudget());

                reportFile.print("Time-Stamp");
                for (int i = 0; i < length; i++) {
                    reportFile.print("\t" + resourceNameList[i]);
                }

                reportFile.print("\t" + "Total");
                reportFile.println();   // end of header line.
            }
            else
            {
                // TIME-STAMP "\t" Resource1-Name "\t" Resource2-Name "\t"
                // Resource3-Name ...
                reportFile.print( GridSim.clock() );
                double total_finished = 0;
                double value = 0.0;
                for (int i = 0; i < length; i++)
                {
                    BrokerResource br = findBrokerResourceWithName(BRList,
                                            resourceNameList[i]);
                    value = br.getParameterValue(parameter);
                    reportFile.print("\t" + value);
                    total_finished += value;
                }

                reportFile.print("\t" + total_finished);
                reportFile.println();  // end of a report row.
            }

            reportFile.close();
        }
        catch (IOException e)
        {
            System.out.println("Broker.writeScheduleReport(): Error - unable " +
                    "to open \"" + reportFileFullname + "\"");
        }
    }

    /**
     * Writes a summary of the schedule report mentioning the total results
     * of the experiment.
     * <p>
     * In addition,
     * creates report files with <tt>".sched4"</tt>, <tt>".sched5"</tt>, and
     * <tt>".sched6"</tt> extension.
     * @param expt      an Experiment object
     * @param BRList    a linked-list of BrokerResource objects
     * @deprecated As of GridBroker 2.1, replaced by
     *             {@link #aggregatedScheduleReport(Experiment, LinkedList)}
     * @pre expt != null
     * @pre BRList != null
     * @post $none
     */
    public void AggregatedScheduleReport(Experiment expt, LinkedList BRList)
    {
        this.aggregatedScheduleReport(expt, BRList);
    }

    /**
     * Writes a summary of the schedule report mentioning the total results
     * of the experiment.
     * <p>
     * In addition,
     * creates report files with <tt>".sched4"</tt>, <tt>".sched5"</tt>, and
     * <tt>".sched6"</tt> extension.
     * @param expt      an Experiment object
     * @param BRList    a linked-list of BrokerResource objects
     * @pre expt != null
     * @pre BRList != null
     * @post $none
     */
    public void aggregatedScheduleReport(Experiment expt, LinkedList BRList)
    {
        if (expt.getExperimentID() != 0) {
            return;
        }

        // print scheduling trace for experiment with ID 0 (i.e., for the
        // first user)
        String report1_filename = expt.getReportFileName() + ".sched4";
        String report2_filename = expt.getReportFileName() + ".sched5";
        String report3_filename = expt.getReportFileName() + ".sched6";

        writeAggregateScheduleReport("Gridlets Finished Report",
            report1_filename, BrokerResource.PARAM_GRIDLETS_FINISHED, expt,
            BRList, false);

        writeAggregateScheduleReport("Gridlets Processing Expenses Report",
            report2_filename, BrokerResource.PARAM_PROCESSING_EXPENSES, expt,
            BRList, false);

        writeAggregateScheduleReport("Gridlets on Resource Report",
            report3_filename, BrokerResource.PARAM_GRIDLETS_ON_RESOURCE, expt,
            BRList, false);
    }

    /**
     * Writes a schedule report
     * @param reportTitle   the report title
     * @param reportFileFullname    the file name
     * @param parameter     the parameter of a broker resource
     * @param expt          an Experiment object
     * @param BRList        a linked-list of BrokerResource objects
     * @param reportHeaderFlag      a flag to denote writing header titles or
     * @deprecated As of GridBroker 2.1, replaced by
     *             {@link #writeAggregateScheduleReport(String, String, int,
     *             Experiment, LinkedList, boolean)}
     * @pre reportTitle != null
     * @pre reportFileFullname != null
     * @pre parameter >= 0
     * @pre expt != null
     * @pre BRList != null
     * @post $none
     */
    public void WriteAggregateScheduleReport(String reportTitle,
                    String reportFileFullname, int parameter, Experiment expt,
                    LinkedList BRList, boolean reportHeaderFlag)
    {
        this.writeAggregateScheduleReport(reportTitle, reportFileFullname,
                parameter, expt, BRList, reportHeaderFlag);
    }

    /**
     * Writes a schedule report
     * @param reportTitle   the report title
     * @param reportFileFullname    the file name
     * @param parameter     the parameter of a broker resource
     * @param expt          an Experiment object
     * @param BRList        a linked-list of BrokerResource objects
     * @param reportHeaderFlag     a flag to denote writing header titles or
     * @pre reportTitle != null
     * @pre reportFileFullname != null
     * @pre parameter >= 0
     * @pre expt != null
     * @pre BRList != null
     * @post $none
     */
    public void writeAggregateScheduleReport(String reportTitle,
                    String reportFileFullname, int parameter, Experiment expt,
                    LinkedList BRList, boolean reportHeaderFlag)
    {
        try
        {
            // open file for append mode.
            FileWriter fw = new FileWriter(reportFileFullname, true);
            BufferedWriter bw = new BufferedWriter(fw);
            PrintWriter reportFile = new PrintWriter(bw);

            String [] resourceNameList = expt.getResourceNameList();
            int length = resourceNameList.length;

            if (reportHeaderFlag)
            {
                // TIME-STAMP "\t" Resource1-Name "\t" Resource2-Name "\t"
                // Resource3-Name ...
                reportFile.println(reportTitle);  // start on newline.
                reportFile.println("Deadline: " + expt.getDeadline() +
                        " Budget: " + expt.getBudget());

                for (int i = 0; i < length; i++) {
                    reportFile.print("\t" + resourceNameList[i]);
                }

                reportFile.print("\t"+"Total");
                reportFile.println();  // end of header line.
            }
            else
            {
                // FOR Gridlet Completion on Resources, .. ,
                // TimeStamp Deadline  Budget   R0    R2   R3  ....
                // TIME-STAMP "\t" Resource1-Name "\t" Resource2-Name "\t"
                // Resource3-Name ...
                reportFile.print(GridSim.clock() + "\t" + expt.getDeadline() +
                        "\t" + expt.getBudget());

                double total_finished = 0;
                double value = 0.0;
                for (int i = 0; i < length; i++)
                {
                    BrokerResource br = findBrokerResourceWithName(BRList,
                            resourceNameList[i]);

                    value = br.getParameterValue(parameter);
                    reportFile.print("\t" + value);
                    total_finished += value;
                }
                reportFile.print("\t" + total_finished);
                reportFile.println();  // end of a report row.
            }

            reportFile.close();
        }
        catch (IOException e)
        {
            System.out.println("Broker.writeAggregateScheduleReport(): Error -"+
                    " unable to open \"" + reportFileFullname + "\"");
        }
    }

    /**
     * Processes one event at one time. Then calculating the gridlet deadline
     * and budget. In addition, starting the experiment time, records
     * statistics, and constructs report files.
     * @pre $none
     * @post $none
     */
    public void body()
    {
        Sim_event ev = new Sim_event();

        // Accept User Commands and Process
        for ( sim_get_next(ev); ev.get_tag() != GridSimTags.END_OF_SIMULATION;
                sim_get_next(ev) )
        {
            // Extra Experiment Details from the User Event
            if (ev.get_tag() != GridSimTags.EXPERIMENT)
            {
                System.out.println(super.getEntityName() +
                        ": A late delivery of Gridlet!");

                System.out.println("-->1. Please ensure that GridBroker waits" +
                        " for arrival of all Gridlets assigned to resource.");

                System.out.println("-->2. Make sure that Resources are NOT " +
                        "returning more Gridlets than assigned to it!");

                System.out.println( ev.get_data() );
                continue;
            }

            experiment_ = (Experiment) ev.get_data();
            int UserEntityID = ev.get_src();

            // Record Experiment Start Time.
            experiment_.setStartTime();

            // Set Gridlets OwnerID as this BrokerID so that Resources
            // knows where to return them.
            int i = 0;
            int id = super.get_id();

            for (i = 0; i < experiment_.getGridletList().size(); i++) {
                ( (Gridlet) experiment_.getGridletList().get(i) ).setUserID(id);
            }

            // RESOURCE DISCOVERY: Get Resource List from GIS + Property
            // from Resource Info Service

            // Requesting the GridInformationService to send a list of resources
            resIDList_ = super.getGridResourceList();
            super.recordStatistics("BROKER.Resource.Count", resIDList_.size());

            // Cretae Resource List with Static and Dynamic Info.
            brokerResourceList_ = new LinkedList();
            Accumulator accMIPSRatingAll = new Accumulator();
            Accumulator accMIPSRating = new Accumulator();
            Accumulator accMIPSRatingAllWithLoad = new Accumulator();
            Accumulator accCostPerMI = new Accumulator();

            // Create Broker Resource List: include Static and Dynamic
            // Characteristics
            for (i = 0; i < resIDList_.size(); i++)
            {
                // Get Resource ID
                int res_id = ( (Integer) resIDList_.get(i) ).intValue();

                // Get Resource Characteristic Info
                ResourceCharacteristics res = super.getResourceCharacteristics(
                                                    res_id);

                accMIPSRating.add( res.getMIPSRatingOfOnePE() );
                accMIPSRatingAll.add( res.getMIPSRating() );
                accCostPerMI.add( res.getCostPerMI() );

                // Get Resource Dynamic information
                Accumulator accLoad = super.getResourceDynamicInfo(res_id);
                accMIPSRatingAllWithLoad.add( res.getMIPSRating() *
                        (1 - accLoad.getMean()) );

                brokerResourceList_.add(
                        new BrokerResource(res, accLoad.getMean()) );
            }

            // calculate deadline and budget based on D_factor, B_factor,
            // workload and resource charactertiscs.
            if ( experiment_.getFactorFlag() )
            {
                Accumulator accLength =
                                experiment_.getAllGridletLengthAccumulator();

                double total_MI = accLength.getSum();

                // MAX and MIN_TIME for gridlet processing should be at least
                // time required to process a single gridlet
                // on a faster processor.

                double ratingMax = accMIPSRating.getMax();
                double lengthMax = accLength.getMax();

                double MAX_time = Math.max(
                                    total_MI/accMIPSRatingAllWithLoad.getMin(),
                                    lengthMax / ratingMax );

                double MIN_time = Math.max( total_MI/accMIPSRatingAll.getSum(),
                                    lengthMax / ratingMax );

                double deadline_duration =  MIN_time +
                    (experiment_.getDeadlineFactor() * (MAX_time - MIN_time));

                // ADDING 10% of COMMUNICATION OVERHEAD to the Computational
                // Deadline
                deadline_duration += deadline_duration * 0.1;

                // processing cost on the most expensive machine
                double MAX_cost = total_MI * accCostPerMI.getMax();

                // processing cost on the least expensive machine
                double MIN_cost = total_MI * accCostPerMI.getMin();
                double budget = MIN_cost +
                    (experiment_.getBudgetFactor() * (MAX_cost-MIN_cost));

                experiment_.setDeadlineBudget(deadline_duration, budget);
            }

            System.out.println(super.getEntityName() + ":: Deadline: " +
                experiment_.getDeadline()+" Budget: "+experiment_.getBudget());

            super.recordStatistics("USER.Experiment.Deadline",
                    experiment_.getDeadline());
            super.recordStatistics("USER.Experiment.Budget",
                    experiment_.getBudget());

            // INITIALISE Broker's global member variables
            glUnfinishedList_ = (GridletList)
                                    experiment_.getGridletList().clone();

            glFinishedList_ = new GridletList();

            gridletDispatched_ = 0;
            gridletReturned_ = 0;
            expenses_ = 0.0;

            // INITIAL Report Writter:
            // create a clone of brokerResourceList_ to maintain the listing
            // Order for Report Generation
            LinkedList BRListOriginal = (LinkedList)brokerResourceList_.clone();

            // initialise Report header.
            scheduleReport(experiment_, BRListOriginal, true);
            scheduleReport(experiment_, BRListOriginal, false);

            // SCHEDUDLING: Do until experiment finishes
            while (glFinishedList_.size() < experiment_.getGridletList().size())
            {
                // stop if there is no sufficient deadline or budget is
                // available.
                if ( (GridSim.clock() >= experiment_.getDeadlineTime()) ||
                     (expenses_ >= experiment_.getBudget()) )
                {
                    break;
                }

                // TODO: these 2 statements are unused
                // TODO: if commented this one, the gridlets are NOT assigned
                // to resources
                int scheduled = scheduleAdviser();

                // TODO: if commented this one, the gridlets are assigned to
                // resources but not executed.
                int just_dispatched = dispatcher();

                int just_received = gridletReceptor();
                scheduleReport(experiment_, BRListOriginal, false);

                // Broker entity should hold for sometime to advance simulation
                // time. Otherwise, no other entities can proceed nor any
                // events/messages are delivered to other entities.
                if (just_received == 0)
                {
                    double deadline_left = experiment_.getDeadlineTime() -
                                GridSim.clock();

                    // heurisitics for deciding hold condition
                    // hold period can be as high as the time required to
                    // process some Gridlets.
                    // HOLD PERIOD NEED TO BE AS HIGH AS SMALLEST PROCESSING
                    // TIME OF A GRIDLET
                    // Something like:
                    //      Math.min(deadline_left*0.01, GridLetProcessingTIME);
                    //---->WARNINING------------
                    // THE CAN LEAD  TO ERROR<PAST Events Detected>: if hold
                    // period is Larger than
                    // processing time of a Gridlet, because when Gridlet is
                    // dispatched and this is Still in Hold

                    double holdperiod = Math.max(deadline_left * 0.01, 1.0);

                    // holdperiod = 1.0;
                    // broker holds for some period for other activities to
                    // proceed.
                    // holdperiod = 1; OR less takes MORE simulation time,
                    // but makes Simulation more precise.
                    // REAL systems use higher HOLD-period because lower values
                    // introduces too much overhead on the broker.

                    //System.out.println("deadline_left = " + deadline_left + 
                    //        ", holdperiod = " + holdperiod);

                    super.gridSimHold(holdperiod);

                    // NOTE: GridSimHold may be changed to INTERRUPTABLE Hold,
                    // then event should be procesed. Otherwise, this can lead
                    // to Past events detected.
                }
            }

            // Capture all events related to Gridlets that have been scheduled
            // for execution, but not yet received.
            while (gridletDispatched_ > gridletReturned_)
            {
                gridletReceptor();
                scheduleReport(experiment_, BRListOriginal, false);
            }

            aggregatedScheduleReport(experiment_, BRListOriginal);
            super.send(UserEntityID, GridSimTags.SCHEDULE_NOW,
                    GridSimTags.EXPERIMENT, experiment_);
        }

        super.terminateIOEntities();
    }

    //////////////////////// PRIVATE METHODS /////////////////////////////

    /**
     * Finds the broker resource with a specified resource name
     * @param BRList    a linked-list containing BrokerResource objects
     * @param resourceName  a resource name entity
     * @return a <tt>BrokerResource</tt> object or <tt>null</tt> if not found
     * @pre BRList != null
     * @pre resourceName != null
     * @post $none
     */
    private BrokerResource findBrokerResourceWithName(LinkedList BRList,
                String resourceName)
    {
        for (int i = 0; i < BRList.size(); i++)
        {
            BrokerResource br = (BrokerResource) BRList.get(i);
            if (resourceName.compareTo( br.resource.getResourceName() ) == 0) {
                return br;
            }
        }
        return null;
    }

    /**
     * Receives a Gridlet one at a time
     * @return number of Gridlets received
     * @pre $none
     * @post $none
     */
    private int gridletReceptor()
    {
        int received = 0;

        if (gridletDispatched_ > gridletReturned_)
        {
            do
            {
                // Get Gridlet finished information via input entity
                Gridlet glFinished = super.gridletReceive();

                super.recordStatistics("BROKER.Gridlet.EndTime",
                        glFinished.getGridletID());

                // Updated Broker Resource:
                BrokerResource brServed = whoServed(glFinished);
                brServed.NoOfGridletsFinishedSoFar++;
                brServed.ProcessingExpensesSoFar +=
                        glFinished.getProcessingCost();

                // Update Available Resource Share
                brServed.updateAvailableMIPS(glFinished, maxGridletPerPE_);

                // move just finished Gridlet from Broker Resource to the
                // glFinishedList_
                brServed.glList.remove(glFinished);
                glFinishedList_.add(glFinished);

                // updated global parameters.
                gridletReturned_++;
                expenses_ += glFinished.getProcessingCost();
                received++;
            }
            while (super.sim_waiting(new Sim_from_port(super.input) ) > 0);
        }

        return received;
    }

    /**
     * Identified by a BrokerResource which processed the Gridlet
     * @param gl    a Gridlet object
     * @return <tt>BrokerResource</tt> that executed Gridlet
     * @pre gl != null
     * @post $result != null
     * @see gridsim.Gridlet
     * @see gridbroker.BrokerResource
     */
    private BrokerResource whoServed(Gridlet gl)
    {
        BrokerResource br = null;
        for (int i = 0; i < brokerResourceList_.size(); i++)
        {
            br = (BrokerResource) brokerResourceList_.get(i);
            if (br.glList.contains(gl)) {
                break;
            }
        }
        return br;
    }

    /**
     * Budget committed, but NOT yet spent.
     * Processing Cost of Gridlets in READY, INQUEUE, or INEXEC
     * @return Expected cost of processing Gridlets that are already
     *         committeed to resources
     * @pre $none
     * @post $result >= 0.0
     */
    private double budgetOnHold()
    {
        double amount = 0;
        int j = 0;
        int status = 0;

        for (int i = 0; i < brokerResourceList_.size(); i++)
        {
            BrokerResource br = (BrokerResource) brokerResourceList_.get(i);
            for (j = 0; j < br.glList.size(); j++)
            {
                Gridlet gl = (Gridlet) br.glList.get(j);
                status =  gl.getGridletStatus();
                if (status == Gridlet.READY || status == Gridlet.QUEUED ||
                        status == Gridlet.INEXEC)
                {
                    amount += br.getExpectedProcessingCost(gl);
                }
            }
        }
        return amount;
    }

    /**
     * Decide whether budget is available for processing Gridlet on a broker
     * resource
     * @param gl    a Gridlet object
     * @param obj   a BrokerResource object
     * @return <tt>true</tt> if budget available, <tt>false</tt>otherwise
     * @pre gl != null
     * @pre obj != null
     * @post $none
     */
    private boolean isBudgetAvailable(Gridlet gl, BrokerResource obj)
    {
        if ( (expenses_ + budgetOnHold() + obj.getExpectedProcessingCost(gl)) <=
                experiment_.getBudget() )
        {
            return true;
        }
        else {
            return false;
        }
    }

    /**
     * The budget available for allocation...after excluding "the budget spent
     * + committed
     * @return the remaining budget
     * @pre $none
     * @post $result >= 0.0
     */
    private double experimentRemainingBudget()
    {
        // budget spent + budget committed need to be taken into consideration
        return experiment_.getBudget() - ( expenses_ + budgetOnHold() );
    }

    /**
     * It takes the length of other Gridlets into consideration
     * @param gl    a Gridlet object
     * @return the proportional available budget for processing the Gridlet
     * @pre gl != null
     * @post $result >= 0.0
     */
    private double remainingBudgetForGridlet(Gridlet gl)
    {
        double remaining_budget = experimentRemainingBudget();

        // find out sum of all UnAssignedGridlet Length();
        double sum_gridlets_length = 0;
        for (int i = 0; i < glUnfinishedList_.size(); i++)
        {
            sum_gridlets_length +=
                    ( (Gridlet) glUnfinishedList_.get(i) ).getGridletLength();
        }

        double available_budget_per_MI = remaining_budget/sum_gridlets_length;
        return available_budget_per_MI * gl.getGridletLength();
    }

    /**
     * Depending on Optimisation Policy, it selects resources
     * @return the number of Gridlets scheduled
     * @pre $none
     * @post $result >= 0
     */
    private int scheduleAdviser()
    {
        if (experiment_.getOptimizationStrategy() == Experiment.OPTIMIZE_TIME)
        {
            // Removing all non-dispatched gridlets for making FRESH decision
            // on mapping Gridlets to resources
            // May be we can do Prediction to see how many to REMOVE from the
            // previous assignment
            int rating = 0;
            int gridletReady = 0;

            for (int i = 0; i < brokerResourceList_.size(); i++)
            {
                BrokerResource br = (BrokerResource) brokerResourceList_.get(i);
                rating = br.resource.getMIPSRating();
                double per1 = br.getAvailableMIPS_PreviousSchedule() / rating;
                double per2 = br.getAvailableMIPS() / rating;

                gridletReady = br.getNumGridletInReady();
                int ExpectedToFinish = (int) ( (per2/per1) * gridletReady );
                int ExcessGridlets = gridletReady - ExpectedToFinish;

                // Remove ExcessGridlets from the assigned list.
                // this is for scanning across whole list.
                GridletList glTempList = (GridletList) br.glList.clone();
                for (int j = 0; ExcessGridlets >0 && j< glTempList.size(); j++)
                {
                    Gridlet gl = (Gridlet) glTempList.get(j);
                    if (gl.getGridletStatus() == Gridlet.READY)
                    {
                        // change Gridlet Status back to Created, remove from
                        // the BR and add to glUnfinishedList_!
                        try {
                            gl.setGridletStatus(Gridlet.CREATED);
                        }
                        catch (Exception e) {
                            e.printStackTrace();
                        }

                        br.glList.remove(gl);
                        glUnfinishedList_.add(gl);
                        --ExcessGridlets;
                    } // end if
                } // end for

                br.setAvailableMIPS_PreviousSchedule();
            } // end for
        } // end if

        else
        {

            // Removing all non-dispatched gridlets for making FRESH decision
            // on mapping Gridlets to resources
            // May be we can do Prediction to see how many to REMOVE from the
            // previous assignment
            for (int i = 0; i < brokerResourceList_.size(); i++)
            {
                BrokerResource br = (BrokerResource) brokerResourceList_.get(i);

                // Create temporary GL list from the br.glList since using
                // br.glList.size() in LOOP + removing items
                // from br.glList leads to inconsistency.
                // this is for scanning across whole list.
                GridletList glTempList = (GridletList) br.glList.clone();
                for (int j = 0; j < glTempList.size(); j++)
                {
                    Gridlet gl = (Gridlet) glTempList.get(j);
                    if (gl.getGridletStatus() == Gridlet.READY)
                    {
                        // change Gridlet Status back to Created, remove from
                        // the BR and add to glUnfinishedList_!
                        try {
                            gl.setGridletStatus(Gridlet.CREATED);
                        }
                        catch (Exception e) {
                            e.printStackTrace();
                        }
                        br.glList.remove(gl);
                        glUnfinishedList_.add(gl);
                    } // end if
                } // end for
            } // end for
        } // end else

        int scheduled = experiment_.getNumGridlet() -
                            (glFinishedList_.size() + glUnfinishedList_.size());

        if (glUnfinishedList_.size() == 0) {
            return scheduled;
        }

        // sort by the ascending order of length to maximize packing AND it can
        // also be reverse to minimise the risk of exceeding too much beyond
        // deadline if resources do not perform as expected!
        glUnfinishedList_.sort();

        switch ( experiment_.getOptimizationStrategy() )
        {
            case Experiment.OPTIMIZE_COST:
                scheduled += scheduleWithDBC_CostOptimisation(new OrderCost());
                break;

            case Experiment.OPTIMIZE_COST_PLUS:
                scheduled += scheduleWithDBC_CostOptimisation(
                                    new OrderCostTime() );
                break;

            case Experiment.OPTIMIZE_COST_TIME:
                scheduled += scheduleWithDBC_CostTimeOptimisation();
                break;

            case Experiment.OPTIMIZE_TIME:
                scheduled += scheduleWithDBC_TimeOptimisation();
                break;

            case Experiment.OPTIMIZE_NONE:
                // do nothing
                break;

            default:
                break;
        }

        return scheduled;
    }

    /**
     * It performs Deadline and Budget Constratined (DBC) Cost Minimisation
     * scheduling
     * @param sortComparator    a Comparator object
     * @return the number of Gridlets scheduled
     * @pre sortComparator != null
     * @post $result >= 0
     */
    private int scheduleWithDBC_CostOptimisation(Comparator sortComparator)
    {
        int scheduled = 0;  // No of Gridlets scheduled during this iteration
        Collections.sort(brokerResourceList_, sortComparator);

        for (int i = 0; i < brokerResourceList_.size() && 
                    glUnfinishedList_.size() > 0; i++)
        {
            BrokerResource br = (BrokerResource) brokerResourceList_.get(i);
            double AvailableMI = br.getAvailableMI(
                                        experiment_.getDeadlineTime() );

            // this is for scanning across whole list.
            GridletList glTempList = (GridletList) glUnfinishedList_.clone();

            // schedule as long as there Gridlets and MIs are available!
            for (int j = 0; j < glTempList.size() && AvailableMI > 0; j++)
            {
                Gridlet gl = (Gridlet) glTempList.get(j);

                // assign only when budget is available for processing on a
                // resource and it can complete by deadline.
                if ( isBudgetAvailable(gl, br) )
                {
                    if ( br.isSufficientMIAvailableOnSinglePE(gl,
                                experiment_.getDeadlineTime()) )
                    {
                        try {
                            gl.setGridletStatus(Gridlet.READY);
                        }
                        catch (Exception e) {
                            e.printStackTrace();
                        }
                        gl.setResourceParameter(br.resource.getResourceID(),
                                br.resource.getCostPerSec());

                        glUnfinishedList_.remove(gl);
                        br.glList.add(gl);
                        AvailableMI -= gl.getGridletLength();
                        scheduled++;
                    } // end if
                } // end if
            } // end for
        } // end for

        return scheduled;
    }

    /**
     * It performs Deadline and Budget Constratined (DBC) Cost+Time
     * Minimisation scheduling
     * @return the number of Gridlets scheduled
     * @pre $none
     * @post $result >= 0
     */
    private int scheduleWithDBC_CostTimeOptimisation()
    {
        int scheduled = 0;  // No of Gridlets scheduled during this iteration
        Collections.sort(brokerResourceList_, new OrderCostTime());

        // Initialise BrokerResourcesWithSameCostPerMI parameters
        BrokerResource br;
        LinkedList BrokerResourcesWithSameCostPerMI = new LinkedList();

        for (int currentBRindex = 0; glUnfinishedList_.size() > 0 && 
                 currentBRindex < brokerResourceList_.size();
                 currentBRindex++)
        {
            // Create a List of BrokerResourcesWithSameCostPerMI
            BrokerResourcesWithSameCostPerMI.clear();   // clear old listing.

            // add current resource to the list
            br = (BrokerResource) brokerResourceList_.get(currentBRindex);
            BrokerResourcesWithSameCostPerMI.add((BrokerResource) br);

            // group all those resources with the same cost PerMI as the
            // current resource into the list BrokerResourcesWithSameCostPerMI
            while (currentBRindex < brokerResourceList_.size()-1 &&
                   br.resource.getCostPerMI() == ( (BrokerResource)
                            brokerResourceList_.get(currentBRindex+1)
                       ).resource.getCostPerMI() )
            {
                currentBRindex++;
                br = (BrokerResource) brokerResourceList_.get(currentBRindex);
                BrokerResourcesWithSameCostPerMI.add((BrokerResource) br);
            }

            // Allocate Tasks to BrokerResourcesWithSameCostPerMI with Time
            // Minimisation Style
            // schedule as long as there are Gridlets and MIs are available!
            LinkedList BRGridletTimeList = new LinkedList();

            // this is for scanning all Gridlets
            GridletList glTempList = (GridletList) glUnfinishedList_.clone();
            for (int j = 0; j < glTempList.size(); j++)
            {
                Gridlet gl = (Gridlet) glTempList.get(j);
                BRGridletTimeList.clear();

                // TimeMinimisation Algorithm:
                // 1. For each resource, calculate the next completion time
                // for an assigned job, taking
                // into account previously assigned jobs.
                for (int k = 0;k < BrokerResourcesWithSameCostPerMI.size();k++)
                {
                    BRGridletTimeList.add( new BRGridletProcessingTime(
                         (BrokerResource) BrokerResourcesWithSameCostPerMI.get(
                              k), gl) );
                }

                // 2. Sort resources by next completion time.
                Collections.sort(BRGridletTimeList,
                        new OrderBRGridletProcessingTime());

                // 3. Assign Gridlet to a resource whose cost per Gridlet is
                // less than or equal to the
                // remaining budget per Gridlet. AND gridlet can be finished
                // within the Deadline.
                boolean gridlet_assigned_flag = false;
                for (int k = 0; k < BRGridletTimeList.size() &&
                        !gridlet_assigned_flag; k++)
                {
                    BrokerResource myBR = ( (BRGridletProcessingTime)
                            BRGridletTimeList.get(k) ).br;

                    // RemainingBudgetForGridlet(gl) may be replaced by just
                    // BudgetAvailable ?
                    double Gl_completion_time = GridSim.clock() +
                        ( (BRGridletProcessingTime)BRGridletTimeList.get(k)
                         ).gridlet_processing_time;

                    if ( Gl_completion_time <= experiment_.getDeadlineTime() )
                    {
                        if ( isBudgetAvailable(gl, myBR) )
                        {
                            try {
                                gl.setGridletStatus(Gridlet.READY);
                            }
                            catch (Exception e) {
                                e.printStackTrace();
                            }
                            gl.setResourceParameter(
                                    myBR.resource.getResourceID(),
                                    myBR.resource.getCostPerSec() );

                            glUnfinishedList_.remove(gl);
                            myBR.glList.add(gl);
                            scheduled++;
                            gridlet_assigned_flag = true;
                        } // end if
                    } // end if
                } // end for
            } // end for
        } // end for

        return scheduled;
    }

    /**
     * It performs Deadline and Budget Constratined (DBC) Cost+Time
     * Minimisation scheduling
     * @return the number of Gridlets scheduled
     * @pre $none
     * @post $result >= 0
     */
    private int scheduleWithDBC_TimeOptimisation()
    {
        int scheduled = 0;  // No of Gridlets scheduled during this iteration
        Collections.sort(brokerResourceList_, new OrderCostTime());

        // Initialise BrokerResourcesWithSameCostPerMI parameters
        BrokerResource br;
        LinkedList BrokerResourcesToUse = new LinkedList();

        // Allocate Tasks to BrokerResourcesWithSameCostPerMI with Time
        // Minimisation Style
        // schedule as long as there are Gridlets and MIs are available!
        LinkedList BRGridletTimeList = new LinkedList();
        GridletList glTempList = (GridletList) glUnfinishedList_.clone();

        // Time Minimisation Algorithm:
        for (int j = 0; j < glTempList.size(); j++)
        {
            Gridlet gl = (Gridlet) glTempList.get(j);
            BRGridletTimeList.clear();

            // 0. Create AFFORDABLE Broker Resources List
            BrokerResourcesToUse.clear();   // clear old listing

            for (int k = 0; k < brokerResourceList_.size(); k++)
            {
                br = (BrokerResource) brokerResourceList_.get(k);
                if (br.getExpectedProcessingCost(gl) <=
                        remainingBudgetForGridlet(gl))
                {
                    BrokerResourcesToUse.add( (BrokerResource) br );
                }
            }

            // If there are Afforable resources for processing Gridlets,
            // do scheduling.
            if (BrokerResourcesToUse.size() > 0)
            {
                // 1. For each resource, calculate the next completion time for
                // an assigned job, taking
                // into account previously assigned jobs.

                for (int k = 0; k <  BrokerResourcesToUse.size(); k++)
                {
                    BRGridletTimeList.add( new BRGridletProcessingTime(
                        (BrokerResource) BrokerResourcesToUse.get(k), gl) );
                }

                // 2. Sort resources by next completion time.
                Collections.sort(BRGridletTimeList,
                    new OrderBRGridletProcessingTime());

                // 3. Assign Gridlet to a resource whose cost per Gridlet is
                // less than or equal to the
                // remaining budget per Gridlet. AND gridlet can be finished
                // within the Deadline.
                boolean gridlet_assigned_flag = false;
                for (int k = 0; k < BRGridletTimeList.size() &&
                        !gridlet_assigned_flag; k++)
                {
                    BrokerResource myBR = ( (BRGridletProcessingTime)
                            BRGridletTimeList.get(k) ).br;

                    // remainingBudgetForGridlet(gl) may be replaced by just
                    // BudgetAvailable ?
                    double Gl_completion_time = GridSim.clock() +
                        ( (BRGridletProcessingTime)BRGridletTimeList.get(k)
                        ).gridlet_processing_time;

                    if ( Gl_completion_time <= experiment_.getDeadlineTime() )
                    {
                        try {
                            gl.setGridletStatus(Gridlet.READY);
                        }
                        catch (Exception e) {
                            e.printStackTrace();
                        }
                        gl.setResourceParameter(myBR.resource.getResourceID(),
                                myBR.resource.getCostPerSec());

                        glUnfinishedList_.remove(gl);
                        myBR.glList.add(gl);
                        scheduled++;
                        gridlet_assigned_flag = true;
                    } // end if
                } // end for
            } // end if
        } // end for
        return scheduled;
    }

    /**
     * Responsibles for assigning Gridlets from
     * Gridlets list in Broker Resource to Grid Resource in such a way
     * that GridResource is not overload.
     * <p>
     * This decision is made based
     * current load on Grid resource and size along in case of Time shared
     * resources.
     * If resource is Space shared, then it is based on number of nodes that
     * are free.
     * @pre $none
     * @post $result >= 0
     */
    private int dispatcher()
    {
        // TODO: this method is unused
        int dispatched = 0;
        for (int i = 0; i < brokerResourceList_.size(); i++)
        {
            BrokerResource br = (BrokerResource) brokerResourceList_.get(i);
            int NoOfPEs = br.resource.getNumPE();
            int NoOfGridletsToBeDispatched = br.getNumGridletInReady();
            int NoOfGridletsOnResource = br.getNumGridletInExec() +
                        br.getNumGridletInQueue();

            int OptimalDisptachSize = 0;
            int NoOfGridletsCanBeDispatched = 0;

            // Determine How many Gridlets to be dispatched depending on Type
            // of Resource
            switch ( br.resource.getResourceAllocationPolicy() )
            {
                case ResourceCharacteristics.TIME_SHARED:
                    // How many to be sent to a resource can be based on the
                    // following:
                    // 1. Load on PEs or resource
                    // 2. How many of your Gridlets are running
                    // A simple heuristic can be be about 2 Gridlets on each PE
                    // GET IDEA from Nimrod-G or some Heuristic in Laterature
                    NoOfGridletsCanBeDispatched = NoOfPEs * maxGridletPerPE_ -
                            NoOfGridletsOnResource;

                    OptimalDisptachSize = Math.min(NoOfGridletsToBeDispatched,
                            NoOfGridletsCanBeDispatched);
                    break;

                case ResourceCharacteristics.SPACE_SHARED:
                    NoOfGridletsCanBeDispatched = NoOfPEs * maxGridletPerPE_ -
                            br.getNumGridletInQueue();

                    OptimalDisptachSize = Math.min(NoOfGridletsToBeDispatched,
                            NoOfGridletsCanBeDispatched);
                    break;

                case ResourceCharacteristics.ADVANCE_RESERVATION:
                    // TODO: not yet done....
                    break;

                default:
                    break;
            }

            for (int j = 0;j < br.glList.size() && OptimalDisptachSize > 0;j++)
            {
                Gridlet gl = (Gridlet) br.glList.get(j);
                if (gl.getGridletStatus() == Gridlet.READY)
                {
                    // queues gridlet for execution on a resource
                    super.gridletSubmit(gl, br.resource.getResourceID());
                    br.NoOfGridletsDispatchedSoFar++;
                    gridletDispatched_++;
                    OptimalDisptachSize--;
                    dispatched++;
                } // end if
            } // end for
        } // end for

        return dispatched;
    }

} // end class

